# UCTS - Universal Context Transfer System

> Turn any AI conversation into a deployable project in one command.

[![Python 3.9+](https://img.shields.io/badge/python-3.9+-blue.svg)](https://www.python.org/downloads/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Tests](https://img.shields.io/badge/tests-186%20passing-brightgreen.svg)](tests/)
[![Coverage](https://img.shields.io/badge/coverage-71%25-green.svg)](tests/)

## Overview

UCTS is a powerful tool that transforms AI conversations into fully-functional project workspaces. Whether you've been discussing a Python application with Claude, debugging TypeScript with ChatGPT, or planning a Rust project in VS Code, UCTS can extract the code, infer the structure, and generate a complete project.

### Key Features

- **Multi-Source Ingestion**: Claude Desktop, VS Code extensions, ChatGPT exports, plain transcripts
- **Intelligent Analysis**: Automatic language detection, dependency extraction, file structure inference
- **Multiple Output Targets**: VS Code workspaces, GitLab repositories, GitHub repositories
- **Smart Scaffolding**: Generates README, CI/CD configs, dependency files, and more
- **Production CI/CD**: Matrix testing, linting, coverage reports, automated publishing
- **Cross-Platform**: Full Windows, macOS, and Linux support
- **VS Code Extension**: Integrated capture and project generation from the IDE

## Installation

```bash
pip install ucts
```

Or install from source:

```bash
git clone https://gitlab.com/sentinel-protocol/ucts.git
cd ucts
pip install -e ".[dev]"
```

## Quick Start

### 1. Ingest a Conversation

```bash
# From a Claude Desktop export
ucts ingest ~/Downloads/claude_conversation.json

# From a VS Code Claude extension log
ucts ingest ~/.vscode/claude-chat-history.json --format vscode

# From a plain text transcript
ucts ingest conversation.md --format transcript
```

### 2. Forge a Project

```bash
# Create a VS Code workspace
ucts forge ~/Downloads/conversation.json ./my-project --target vscode

# Create and push to GitLab
export GITLAB_URL=https://gitlab.com
export GITLAB_TOKEN=your-token
ucts forge conversation.json my-project --target gitlab

# Create and push to GitHub
export GITHUB_TOKEN=your-token
ucts forge conversation.json my-project --target github
```

### 3. Analyze Without Generating

```bash
# See what would be generated
ucts analyze conversation.json
```

## CLI Commands

| Command | Description |
|---------|-------------|
| `ucts ingest <source>` | Ingest and analyze a conversation |
| `ucts forge <source> <output>` | Convert conversation to project |
| `ucts analyze <source>` | Analyze without generating |
| `ucts status` | Show UCTS status |
| `ucts transfer` | Create context transfer package |
| `ucts version` | Show version info |

## Supported Languages

UCTS automatically detects and extracts dependencies for:

| Language | Extensions | Dependency Files |
|----------|------------|------------------|
| Python | `.py` | `requirements.txt`, `pyproject.toml` |
| JavaScript | `.js`, `.mjs`, `.cjs` | `package.json` |
| TypeScript | `.ts`, `.tsx` | `package.json` |
| Rust | `.rs` | `Cargo.toml` |
| Go | `.go` | `go.mod` |
| Java | `.java` | `pom.xml`, `build.gradle` |
| C# | `.cs` | `.csproj` |
| Ruby | `.rb` | `Gemfile` |
| PHP | `.php` | `composer.json` |
| Kotlin | `.kt`, `.kts` | `build.gradle.kts` |
| Swift | `.swift` | `Package.swift` |
| C/C++ | `.c`, `.cpp`, `.h` | `CMakeLists.txt` |

## Supported Formats

### Input Sources

| Source | Format | Description |
|--------|--------|-------------|
| Claude Desktop | JSON | Official Claude Desktop exports |
| VS Code | JSON | Claude/Copilot extension logs |
| ChatGPT | JSON | ChatGPT conversation exports |
| Claude.ai | JSON | Claude.ai web exports |
| Transcript | MD/TXT | Plain text conversations |

### Output Targets

| Target | Description |
|--------|-------------|
| VS Code | Complete workspace with settings, tasks, and extensions |
| GitLab | Repository with .gitlab-ci.yml and full CI/CD |
| GitHub | Repository with GitHub Actions workflows |

## Python API

```python
from ucts import UniversalContextManager
from ucts.ingestion import get_ingester
from ucts.analysis import AnalysisEngine
from ucts.generators import VSCodeGenerator

# Ingest a session
ingester = get_ingester('auto', 'conversation.json')
session = ingester.ingest('conversation.json')

# Analyze
engine = AnalysisEngine()
structure = engine.analyze(session)

# Generate
generator = VSCodeGenerator()
generator.generate(structure, './output')
```

## Context Transfer

UCTS also provides context management for long AI conversations:

```python
from ucts import UniversalContextManager

# Create manager
manager = UniversalContextManager("my-project")

# Add messages
manager.add_message("user", "Help me build a REST API")
manager.add_message("assistant", "I'll help you create a Flask API...")

# Create transfer package
package = manager.create_transfer_package()
print(package['markdown'])
```

## VS Code Extension

UCTS includes a VS Code extension for integrated project generation.

### Installation

```bash
cd vscode-ucts
npm install
npm run compile
# Package as VSIX
npx vsce package
```

### Features

- **Session Capture**: Capture current workspace conversations
- **Project Forging**: Generate projects from captured sessions
- **Status Display**: Real-time UCTS status in the sidebar
- **Tree Views**: Browse sessions and projects

### Extension Commands

| Command | Description |
|---------|-------------|
| `UCTS: Capture Current Session` | Capture conversation from workspace |
| `UCTS: Forge New Project` | Generate project from session |
| `UCTS: Show Status` | Display UCTS status |
| `UCTS: Refresh Sessions` | Refresh session list |

## Configuration

UCTS stores configuration in platform-appropriate locations:

- **Windows**: `%APPDATA%\UCTS\config.yaml`
- **macOS**: `~/Library/Application Support/UCTS/config.yaml`
- **Linux**: `~/.ucts/config.yaml` or `$XDG_DATA_HOME/ucts/config.yaml`

Default configuration:

```yaml
version: '2.0.0'
auto_monitor: true
token_warning: 2500
token_critical: 3000
max_chunks: 5
retention_days: 30
```

## Environment Variables

| Variable | Description |
|----------|-------------|
| `GITLAB_URL` | GitLab server URL |
| `GITLAB_TOKEN` | GitLab API access token |
| `GITHUB_TOKEN` | GitHub personal access token |

## Development

```bash
# Install with dev dependencies
pip install -e ".[dev]"

# Run tests
pytest tests/ -v

# Format code
black ucts/
isort ucts/

# Lint
ruff check ucts/
```

## Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                      UCTS 2.0 Architecture                  │
├─────────────────────────────────────────────────────────────┤
│                                                             │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐      │
│  │ Claude       │  │ VS Code      │  │ Web Chat     │      │
│  │ Desktop      │  │ Extension    │  │ Export       │      │
│  └──────┬───────┘  └──────┬───────┘  └──────┬───────┘      │
│         │                 │                 │               │
│         ▼                 ▼                 ▼               │
│  ┌─────────────────────────────────────────────────────┐   │
│  │              Session Ingestion Layer                 │   │
│  └─────────────────────────┬───────────────────────────┘   │
│                            │                                │
│                            ▼                                │
│  ┌─────────────────────────────────────────────────────┐   │
│  │              Analysis Engine                         │   │
│  └─────────────────────────┬───────────────────────────┘   │
│                            │                                │
│         ┌──────────────────┼──────────────────┐            │
│         ▼                  ▼                  ▼            │
│  ┌────────────┐    ┌────────────┐    ┌────────────┐       │
│  │ VS Code    │    │ GitLab     │    │ GitHub     │       │
│  │ Generator  │    │ Generator  │    │ Generator  │       │
│  └────────────┘    └────────────┘    └────────────┘       │
│                                                             │
└─────────────────────────────────────────────────────────────┘
```

## Integration with Sentinel Protocol

UCTS was originally developed as part of the [Sentinel Protocol](https://sentinelprotocol.ai) project for AI-orchestrated insurance systems. It can integrate with Sentinel's Coherence orchestrator for cross-project workflow management.

## License

MIT License - see [LICENSE](LICENSE) for details.

## Contributing

Contributions are welcome! Please read our contributing guidelines and submit pull requests to the main repository.

---

**Built with love by the Sentinel Protocol team.**
