"""
Tests for AI Agent Mode

Tests the agent/autonomous.py module including:
- AgentState and StepType enums
- AgentConfig, AgentStep, ApprovalGate, AgentSession dataclasses
- AgentMode main class
- Singleton agent instance
"""

import asyncio
import json
import pytest
from datetime import datetime
from pathlib import Path
from unittest.mock import AsyncMock, MagicMock, patch

from ucts.agent.autonomous import (
    AgentState,
    StepType,
    AgentConfig,
    AgentStep,
    ApprovalGate,
    AgentSession,
    AgentMode,
    get_agent,
)


# ============================================================================
# AgentState Enum Tests
# ============================================================================

class TestAgentState:
    """Tests for AgentState enum"""

    def test_idle_value(self):
        """Test idle state value"""
        assert AgentState.IDLE.value == "idle"

    def test_planning_value(self):
        """Test planning state value"""
        assert AgentState.PLANNING.value == "planning"

    def test_generating_value(self):
        """Test generating state value"""
        assert AgentState.GENERATING.value == "generating"

    def test_testing_value(self):
        """Test testing state value"""
        assert AgentState.TESTING.value == "testing"

    def test_reviewing_value(self):
        """Test reviewing state value"""
        assert AgentState.REVIEWING.value == "reviewing"

    def test_awaiting_approval_value(self):
        """Test awaiting approval state value"""
        assert AgentState.AWAITING_APPROVAL.value == "awaiting_approval"

    def test_completed_value(self):
        """Test completed state value"""
        assert AgentState.COMPLETED.value == "completed"

    def test_failed_value(self):
        """Test failed state value"""
        assert AgentState.FAILED.value == "failed"

    def test_paused_value(self):
        """Test paused state value"""
        assert AgentState.PAUSED.value == "paused"


# ============================================================================
# StepType Enum Tests
# ============================================================================

class TestStepType:
    """Tests for StepType enum"""

    def test_plan_value(self):
        """Test plan step type"""
        assert StepType.PLAN.value == "plan"

    def test_generate_value(self):
        """Test generate step type"""
        assert StepType.GENERATE.value == "generate"

    def test_test_value(self):
        """Test test step type"""
        assert StepType.TEST.value == "test"

    def test_fix_value(self):
        """Test fix step type"""
        assert StepType.FIX.value == "fix"

    def test_review_value(self):
        """Test review step type"""
        assert StepType.REVIEW.value == "review"

    def test_approve_value(self):
        """Test approve step type"""
        assert StepType.APPROVE.value == "approve"


# ============================================================================
# AgentConfig Dataclass Tests
# ============================================================================

class TestAgentConfig:
    """Tests for AgentConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = AgentConfig()
        assert config.model == "claude-3-opus-20240229"
        assert config.api_key is None
        assert config.max_iterations == 10
        assert config.require_approval is True
        assert "plan" in config.approval_gates
        assert "deploy" in config.approval_gates
        assert config.test_driven is True
        assert config.auto_fix is True
        assert config.output_path == "./agent_output"
        assert config.checkpoint_interval == 1
        assert config.temperature == 0.7
        assert config.max_tokens == 4096

    def test_custom_values(self):
        """Test custom configuration values"""
        config = AgentConfig(
            model="gpt-4",
            api_key="test-key",
            max_iterations=5,
            require_approval=False,
            test_driven=False,
        )
        assert config.model == "gpt-4"
        assert config.api_key == "test-key"
        assert config.max_iterations == 5
        assert config.require_approval is False
        assert config.test_driven is False

    def test_from_env(self):
        """Test creating config from environment"""
        with patch.dict('os.environ', {
            'ANTHROPIC_API_KEY': 'test-anthropic-key',
            'UCTS_AGENT_MODEL': 'claude-3-sonnet'
        }):
            config = AgentConfig.from_env()
            assert config.api_key == 'test-anthropic-key'
            assert config.model == 'claude-3-sonnet'

    def test_from_env_openai_fallback(self):
        """Test OpenAI key fallback when Anthropic key missing"""
        with patch.dict('os.environ', {
            'OPENAI_API_KEY': 'test-openai-key'
        }, clear=True):
            config = AgentConfig.from_env()
            assert config.api_key == 'test-openai-key'


# ============================================================================
# AgentStep Dataclass Tests
# ============================================================================

class TestAgentStep:
    """Tests for AgentStep dataclass"""

    def test_creation(self):
        """Test basic step creation"""
        step = AgentStep(
            step_id=1,
            step_type=StepType.GENERATE,
            description="Generate code",
            input_context="Build a web app"
        )
        assert step.step_id == 1
        assert step.step_type == StepType.GENERATE
        assert step.description == "Generate code"
        assert step.output is None
        assert step.files_created == []
        assert step.tests_passed is None

    def test_with_all_fields(self):
        """Test step with all fields populated"""
        step = AgentStep(
            step_id=1,
            step_type=StepType.TEST,
            description="Run tests",
            input_context="test files",
            output="All tests passed",
            files_created=["test_main.py"],
            files_modified=["main.py"],
            tests_passed=True,
            duration_ms=1500,
            approved=True,
            error=None
        )
        assert step.tests_passed is True
        assert step.duration_ms == 1500
        assert step.approved is True

    def test_to_dict(self):
        """Test step serialization"""
        step = AgentStep(
            step_id=1,
            step_type=StepType.PLAN,
            description="Create plan",
            input_context="Build app",
            output="## Plan\n1. Step 1\n2. Step 2"
        )
        data = step.to_dict()

        assert data["step_id"] == 1
        assert data["step_type"] == "plan"
        assert data["description"] == "Create plan"
        assert "timestamp" in data

    def test_to_dict_truncates_output(self):
        """Test output is truncated in to_dict"""
        long_output = "x" * 1000
        step = AgentStep(
            step_id=1,
            step_type=StepType.GENERATE,
            description="Generate",
            input_context="context",
            output=long_output
        )
        data = step.to_dict()
        assert len(data["output"]) == 500


# ============================================================================
# ApprovalGate Dataclass Tests
# ============================================================================

class TestApprovalGate:
    """Tests for ApprovalGate dataclass"""

    def test_creation(self):
        """Test basic approval gate creation"""
        step = AgentStep(
            step_id=1,
            step_type=StepType.PLAN,
            description="Plan",
            input_context="context"
        )
        gate = ApprovalGate(
            gate_id="plan",
            description="Review the plan",
            step=step
        )
        assert gate.gate_id == "plan"
        assert gate.description == "Review the plan"
        assert gate.step == step
        assert gate.response is None
        assert gate.feedback is None

    def test_default_options(self):
        """Test default approval options"""
        step = AgentStep(
            step_id=1,
            step_type=StepType.PLAN,
            description="Plan",
            input_context="context"
        )
        gate = ApprovalGate(
            gate_id="deploy",
            description="Deploy approval",
            step=step
        )
        assert "approve" in gate.options
        assert "modify" in gate.options
        assert "reject" in gate.options


# ============================================================================
# AgentSession Dataclass Tests
# ============================================================================

class TestAgentSession:
    """Tests for AgentSession dataclass"""

    def test_creation(self):
        """Test basic session creation"""
        session = AgentSession(
            session_id="test-session",
            goal="Build a web app"
        )
        assert session.session_id == "test-session"
        assert session.goal == "Build a web app"
        assert session.state == AgentState.IDLE
        assert session.steps == []
        assert session.plan is None
        assert session.files == {}
        assert session.tests == {}

    def test_to_dict(self):
        """Test session serialization"""
        session = AgentSession(
            session_id="test-123",
            goal="Build app",
            state=AgentState.GENERATING
        )
        step = AgentStep(
            step_id=1,
            step_type=StepType.PLAN,
            description="Plan",
            input_context="context"
        )
        session.steps.append(step)
        session.files["main.py"] = "print('hello')"

        data = session.to_dict()
        assert data["session_id"] == "test-123"
        assert data["goal"] == "Build app"
        assert data["state"] == "generating"
        assert len(data["steps"]) == 1
        assert "main.py" in data["files"]


# ============================================================================
# AgentMode Tests
# ============================================================================

class TestAgentMode:
    """Tests for AgentMode main class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        agent = AgentMode()
        assert agent.config is not None
        assert agent.session is None
        assert agent._approval_callback is None

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = AgentConfig(max_iterations=5, test_driven=False)
        agent = AgentMode(config)
        assert agent.config.max_iterations == 5
        assert agent.config.test_driven is False

    def test_set_approval_callback(self):
        """Test setting approval callback"""
        agent = AgentMode()
        callback = MagicMock()
        agent.set_approval_callback(callback)
        assert agent._approval_callback == callback

    def test_get_session_none(self):
        """Test get_session returns None when no session"""
        agent = AgentMode()
        assert agent.get_session() is None

    def test_get_progress_no_session(self):
        """Test get_progress with no session"""
        agent = AgentMode()
        progress = agent.get_progress()
        assert progress == {"status": "no_session"}

    @pytest.mark.asyncio
    async def test_start_creates_session(self):
        """Test starting agent creates session"""
        config = AgentConfig(
            approval_gates=[],  # No approvals needed
            test_driven=False,  # Skip tests
        )
        agent = AgentMode(config)
        session = await agent.start("Build a simple app")

        assert session is not None
        assert session.session_id.startswith("agent-")
        assert session.goal == "Build a simple app"
        assert session.state in [AgentState.COMPLETED, AgentState.FAILED]

    @pytest.mark.asyncio
    async def test_start_runs_plan_phase(self):
        """Test that start runs planning phase"""
        config = AgentConfig(
            approval_gates=[],
            test_driven=False,
        )
        agent = AgentMode(config)
        session = await agent.start("Create a CLI tool")

        # Should have a plan step
        plan_steps = [s for s in session.steps if s.step_type == StepType.PLAN]
        assert len(plan_steps) >= 1
        assert session.plan is not None

    @pytest.mark.asyncio
    async def test_start_with_approval_gate(self):
        """Test start pauses at approval gate"""
        config = AgentConfig(
            approval_gates=["plan"],
            test_driven=False,
        )
        agent = AgentMode(config)

        # Set callback that rejects
        agent.set_approval_callback(lambda gate: "reject")

        session = await agent.start("Build app")
        assert session.state == AgentState.PAUSED

    @pytest.mark.asyncio
    async def test_start_approval_callback_approve(self):
        """Test approval callback with approve response"""
        config = AgentConfig(
            approval_gates=["plan"],
            test_driven=False,
        )
        agent = AgentMode(config)

        # Set callback that approves
        agent.set_approval_callback(lambda gate: "approve")

        session = await agent.start("Build app")
        # Should complete (not pause)
        assert session.state == AgentState.COMPLETED

    def test_mock_llm_response_plan(self):
        """Test mock LLM response for plan prompts"""
        agent = AgentMode()
        response = agent._mock_llm_response("Create a plan for the project")

        assert "## Overview" in response
        assert "## Architecture" in response
        assert "## Implementation Steps" in response

    def test_mock_llm_response_generate(self):
        """Test mock LLM response for generate prompts"""
        agent = AgentMode()
        response = agent._mock_llm_response("Generate code for the app")

        assert "```main.py" in response
        assert "def main():" in response

    def test_mock_llm_response_review(self):
        """Test mock LLM response for review prompts"""
        agent = AgentMode()
        response = agent._mock_llm_response("Review this output")

        # Review prompt returns "Review complete..."
        assert "Review" in response or "code" in response.lower()

    def test_parse_code_blocks(self):
        """Test parsing code blocks from response"""
        agent = AgentMode()
        response = """Here's the code:

```main.py
def main():
    print("Hello")
```

```utils.py
def helper():
    return True
```
"""
        files = agent._parse_code_blocks(response)
        assert "main.py" in files
        assert "utils.py" in files
        assert 'def main():' in files["main.py"]

    def test_parse_code_blocks_with_language(self):
        """Test parsing code blocks with language prefix"""
        agent = AgentMode()
        response = """```python
def func():
    pass
```
"""
        files = agent._parse_code_blocks(response)
        # Should create python.py or similar
        assert len(files) >= 1

    def test_build_context(self):
        """Test building context for LLM"""
        config = AgentConfig()
        agent = AgentMode(config)
        agent.session = AgentSession(
            session_id="test",
            goal="Build app",
            plan="## Plan\n1. Step 1"
        )
        agent.session.steps.append(AgentStep(
            step_id=1,
            step_type=StepType.GENERATE,
            description="Generate code",
            input_context="context"
        ))

        context = agent._build_context()
        assert "Goal: Build app" in context
        assert "Plan:" in context
        assert "Recent steps:" in context

    def test_format_files_for_prompt(self):
        """Test formatting files for prompt"""
        agent = AgentMode()
        agent.session = AgentSession(
            session_id="test",
            goal="Build app"
        )
        agent.session.files = {
            "main.py": "print('hello')",
            "utils.py": "def helper(): pass"
        }

        formatted = agent._format_files_for_prompt()
        assert "```main.py" in formatted
        assert "print('hello')" in formatted
        assert "```utils.py" in formatted

    @pytest.mark.asyncio
    async def test_write_files(self, tmp_path):
        """Test writing files to disk"""
        config = AgentConfig(output_path=str(tmp_path))
        agent = AgentMode(config)

        files = {
            "main.py": "print('test')",
            "utils/helper.py": "def help(): pass"
        }
        await agent._write_files(files)

        assert (tmp_path / "main.py").exists()
        assert (tmp_path / "utils" / "helper.py").exists()

    @pytest.mark.asyncio
    async def test_save_checkpoint(self, tmp_path):
        """Test saving checkpoint"""
        config = AgentConfig(output_path=str(tmp_path))
        agent = AgentMode(config)
        agent.session = AgentSession(
            session_id="test-checkpoint",
            goal="Test goal"
        )

        await agent._save_checkpoint()

        checkpoint_file = tmp_path / ".checkpoints" / "test-checkpoint.json"
        assert checkpoint_file.exists()

        with open(checkpoint_file) as f:
            data = json.load(f)
        assert data["session_id"] == "test-checkpoint"

    @pytest.mark.asyncio
    async def test_load_session(self, tmp_path):
        """Test loading session from checkpoint"""
        config = AgentConfig(output_path=str(tmp_path))
        agent = AgentMode(config)

        # Create checkpoint
        checkpoint_dir = tmp_path / ".checkpoints"
        checkpoint_dir.mkdir(parents=True)
        checkpoint_file = checkpoint_dir / "test-load.json"
        checkpoint_file.write_text(json.dumps({
            "session_id": "test-load",
            "goal": "Test goal",
            "state": "paused",
            "plan": "Test plan",
            "current_iteration": 2
        }))

        session = await agent._load_session("test-load")
        assert session is not None
        assert session.session_id == "test-load"
        assert session.goal == "Test goal"
        assert session.state == AgentState.PAUSED
        assert session.current_iteration == 2

    @pytest.mark.asyncio
    async def test_load_session_not_found(self, tmp_path):
        """Test loading nonexistent session returns None"""
        config = AgentConfig(output_path=str(tmp_path))
        agent = AgentMode(config)

        session = await agent._load_session("nonexistent")
        assert session is None

    @pytest.mark.asyncio
    async def test_resume_session(self, tmp_path):
        """Test resuming a paused session"""
        config = AgentConfig(
            output_path=str(tmp_path),
            test_driven=False,
            approval_gates=[]
        )
        agent = AgentMode(config)

        # Create checkpoint
        checkpoint_dir = tmp_path / ".checkpoints"
        checkpoint_dir.mkdir(parents=True)
        checkpoint_file = checkpoint_dir / "resume-test.json"
        checkpoint_file.write_text(json.dumps({
            "session_id": "resume-test",
            "goal": "Resume goal",
            "state": "paused",
            "plan": "Test plan",
            "current_iteration": 1
        }))

        session = await agent.resume("resume-test")
        assert session is not None
        assert session.session_id == "resume-test"

    @pytest.mark.asyncio
    async def test_resume_nonexistent_raises(self, tmp_path):
        """Test resuming nonexistent session raises error"""
        config = AgentConfig(output_path=str(tmp_path))
        agent = AgentMode(config)

        with pytest.raises(ValueError, match="Session not found"):
            await agent.resume("nonexistent")

    def test_get_progress_with_session(self):
        """Test get_progress with active session"""
        config = AgentConfig(max_iterations=10)
        agent = AgentMode(config)
        agent.session = AgentSession(
            session_id="progress-test",
            goal="Test",
            state=AgentState.GENERATING
        )
        agent.session.current_iteration = 3
        agent.session.files["main.py"] = "code"
        agent.session.tests["test_main.py"] = "test"
        agent.session.steps.append(AgentStep(
            step_id=1,
            step_type=StepType.PLAN,
            description="Plan",
            input_context="ctx"
        ))

        progress = agent.get_progress()
        assert progress["session_id"] == "progress-test"
        assert progress["state"] == "generating"
        assert progress["iteration"] == 3
        assert progress["max_iterations"] == 10
        assert progress["steps_completed"] == 1
        assert progress["files_created"] == 1
        assert progress["tests_created"] == 1


# ============================================================================
# LLM Call Tests
# ============================================================================

class TestLLMCalls:
    """Tests for LLM API calls"""

    @pytest.mark.asyncio
    async def test_call_llm_without_api_key(self):
        """Test LLM call without API key uses mock"""
        config = AgentConfig(api_key=None)
        agent = AgentMode(config)

        response = await agent._call_llm("Generate a plan")
        assert response is not None
        assert len(response) > 0

    @pytest.mark.asyncio
    async def test_call_llm_selects_anthropic(self):
        """Test LLM call selects Anthropic for Claude model"""
        config = AgentConfig(
            api_key="test-key",
            model="claude-3-opus"
        )
        agent = AgentMode(config)

        # Mock the Anthropic call
        with patch.object(agent, '_call_anthropic', new_callable=AsyncMock) as mock:
            mock.return_value = "Anthropic response"
            response = await agent._call_llm("Test prompt")
            mock.assert_called_once()

    @pytest.mark.asyncio
    async def test_call_llm_selects_openai(self):
        """Test LLM call selects OpenAI for non-Claude model"""
        config = AgentConfig(
            api_key="test-key",
            model="gpt-4"
        )
        agent = AgentMode(config)

        # Mock the OpenAI call
        with patch.object(agent, '_call_openai', new_callable=AsyncMock) as mock:
            mock.return_value = "OpenAI response"
            response = await agent._call_llm("Test prompt")
            mock.assert_called_once()


# ============================================================================
# Generation Loop Tests
# ============================================================================

class TestGenerationLoop:
    """Tests for generation loop behavior"""

    @pytest.mark.asyncio
    async def test_generation_loop_max_iterations(self):
        """Test generation loop respects max iterations"""
        config = AgentConfig(
            max_iterations=2,
            test_driven=False,
            approval_gates=[]
        )
        agent = AgentMode(config)
        agent.session = AgentSession(
            session_id="test",
            goal="Test",
            state=AgentState.GENERATING
        )

        await agent._generation_loop()
        # Should have run twice
        assert agent.session.current_iteration <= config.max_iterations

    @pytest.mark.asyncio
    async def test_execute_tests_returns_success(self):
        """Test test execution returns success"""
        config = AgentConfig()
        agent = AgentMode(config)
        agent.session = AgentSession(
            session_id="test",
            goal="Test"
        )
        agent.session.tests = {"test_main.py": "def test(): pass"}

        results = await agent._execute_tests()
        assert results["passed"] is True
        assert results["total"] == 1


# ============================================================================
# Singleton Tests
# ============================================================================

class TestGetAgent:
    """Tests for singleton agent instance"""

    def test_get_agent_creates_instance(self):
        """Test getting agent creates instance"""
        import ucts.agent.autonomous as module
        module._agent = None

        agent = get_agent()
        assert agent is not None
        assert isinstance(agent, AgentMode)

    def test_get_agent_returns_same_instance(self):
        """Test getting agent returns same instance"""
        agent1 = get_agent()
        agent2 = get_agent()
        assert agent1 is agent2

    def test_get_agent_with_config_creates_new(self):
        """Test providing config creates new instance"""
        agent1 = get_agent()

        new_config = AgentConfig(max_iterations=3)
        agent2 = get_agent(new_config)

        assert agent2.config.max_iterations == 3


# ============================================================================
# Error Handling Tests
# ============================================================================

class TestErrorHandling:
    """Tests for error handling"""

    @pytest.mark.asyncio
    async def test_start_handles_exception(self):
        """Test start handles exceptions gracefully"""
        config = AgentConfig(approval_gates=[], test_driven=False)
        agent = AgentMode(config)

        # Mock plan phase to raise
        with patch.object(agent, '_plan_phase', new_callable=AsyncMock) as mock:
            mock.side_effect = Exception("Test error")
            session = await agent.start("Test goal")

        assert session.state == AgentState.FAILED
        assert "error" in session.metadata

    @pytest.mark.asyncio
    async def test_call_llm_handles_api_error(self):
        """Test LLM call handles API errors"""
        config = AgentConfig(api_key="test-key", model="claude-3")
        agent = AgentMode(config)

        # Mock to raise exception
        with patch.object(agent, '_call_anthropic', new_callable=AsyncMock) as mock:
            mock.side_effect = Exception("API Error")
            response = await agent._call_llm("Test")

        # Should fall back to mock response
        assert response is not None


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration-style tests"""

    @pytest.mark.asyncio
    async def test_full_agent_workflow(self, tmp_path):
        """Test complete agent workflow"""
        config = AgentConfig(
            output_path=str(tmp_path),
            approval_gates=[],  # No approvals
            test_driven=False,  # Skip tests for speed
            max_iterations=1
        )
        agent = AgentMode(config)

        session = await agent.start("Create a hello world app")

        assert session is not None
        assert session.state == AgentState.COMPLETED
        assert len(session.steps) >= 1
        assert session.plan is not None

        # Verify workflow completed (files may or may not be created
        # depending on mock response matching)
        assert session.current_iteration >= 1

    @pytest.mark.asyncio
    async def test_test_driven_workflow(self, tmp_path):
        """Test test-driven workflow"""
        config = AgentConfig(
            output_path=str(tmp_path),
            approval_gates=[],
            test_driven=True,
            max_iterations=2
        )
        agent = AgentMode(config)

        session = await agent.start("Create a calculator")

        # Should have test steps
        test_steps = [s for s in session.steps if s.step_type == StepType.TEST]
        # May or may not have tests depending on mock behavior
        assert session.state in [AgentState.COMPLETED, AgentState.FAILED]
