"""Tests for analysis engine"""
import pytest

from ucts.analysis import AnalysisEngine
from ucts.core.models import Session, Message, CodeBlock, ProjectStructure


class TestAnalysisEngine:
    """Tests for the analysis engine"""

    def create_test_session(self, messages=None, code_blocks=None):
        """Helper to create test sessions"""
        return Session(
            source="test",
            messages=messages or [],
            code_blocks=code_blocks or [],
            files_created=[],
            decisions=[],
            todos=[],
        )

    def test_analyze_empty_session(self):
        """Test analyzing an empty session"""
        engine = AnalysisEngine()
        session = self.create_test_session()
        structure = engine.analyze(session)

        assert isinstance(structure, ProjectStructure)
        assert structure.name.startswith("ucts-")
        assert len(structure.languages) == 0
        assert len(structure.files) == 0

    def test_analyze_python_session(self):
        """Test analyzing session with Python code"""
        messages = [
            Message(role="user", content="Write a Python function"),
            Message(
                role="assistant",
                content="""Here's a function:
```python
def calculate(x, y):
    return x + y
```
"""
            ),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        assert "python" in structure.languages
        assert len(structure.files) > 0

    def test_extract_python_dependencies(self):
        """Test Python dependency extraction"""
        messages = [
            Message(
                role="assistant",
                content="""```python
import requests
from flask import Flask
import json
import os
```"""
            ),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        python_deps = structure.dependencies.get("python", [])
        # Should include third-party but not stdlib
        assert "requests" in python_deps
        assert "flask" in python_deps
        assert "json" not in python_deps  # stdlib
        assert "os" not in python_deps  # stdlib

    def test_extract_node_dependencies(self):
        """Test Node.js dependency extraction"""
        messages = [
            Message(
                role="assistant",
                content="""```javascript
import express from 'express';
const lodash = require('lodash');
import fs from 'fs';
```"""
            ),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        node_deps = structure.dependencies.get("node", [])
        assert "express" in node_deps
        assert "lodash" in node_deps
        assert "fs" not in node_deps  # built-in

    def test_infer_project_name(self):
        """Test project name inference"""
        # Use "building myapp" pattern which is matched by r'building\s+(?:a\s+)?(\w+)'
        messages = [
            Message(role="user", content="We're building MyApp"),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        assert "myapp" in structure.name.lower()

    def test_classify_code_purpose(self):
        """Test code block purpose classification"""
        messages = [
            Message(
                role="assistant",
                content="""
```python
def test_something():
    assert 1 == 1
```

```json
{"name": "config"}
```
"""
            ),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        # Check that files are in appropriate directories
        files = list(structure.files.keys())
        test_files = [f for f in files if "test" in f.lower()]
        assert len(test_files) > 0

    def test_generate_readme(self):
        """Test README generation"""
        messages = [
            Message(role="user", content="Create a REST API for user management"),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(messages=messages)
        structure = engine.analyze(session)

        assert len(structure.readme_content) > 0
        assert "UCTS" in structure.readme_content

    def test_infer_directories(self):
        """Test directory structure inference"""
        code_blocks = [
            CodeBlock(
                language="python",
                content="print('test')",
                filename="src/main.py",
                purpose="implementation"
            ),
            CodeBlock(
                language="python",
                content="def test(): pass",
                filename="tests/test_main.py",
                purpose="test"
            ),
        ]

        engine = AnalysisEngine()
        session = self.create_test_session(code_blocks=code_blocks)
        structure = engine.analyze(session)

        assert "src" in structure.directories or any(
            f.startswith("src/") for f in structure.files
        )
