"""Tests for SSO/Authentication module"""
import json
from datetime import datetime, timedelta
from pathlib import Path
from unittest.mock import patch, MagicMock

import pytest

from ucts.team.auth import (
    AuthProvider,
    AuthStatus,
    AuthConfig,
    AuthToken,
    UserIdentity,
    AuthSession,
    OAuth2Provider,
    SAMLProvider,
    APIKeyProvider,
    AuthManager,
    get_azure_ad_config,
    get_okta_config,
    get_google_config,
    get_github_config,
)


class TestAuthConfig:
    """Tests for AuthConfig dataclass"""

    def test_default_config(self):
        """Test default AuthConfig values"""
        config = AuthConfig()

        assert config.provider == AuthProvider.LOCAL
        assert config.enabled is False
        assert config.client_id == ""
        assert config.session_timeout == 3600
        assert config.refresh_token_lifetime == 86400 * 7
        assert "openid" in config.scopes
        assert "profile" in config.scopes
        assert "email" in config.scopes

    def test_config_to_dict(self):
        """Test AuthConfig serialization"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test-client",
            session_timeout=7200
        )

        result = config.to_dict()

        assert result["provider"] == "oauth2"
        assert result["enabled"] is True
        assert result["client_id"] == "test-client"
        assert result["session_timeout"] == 7200

    def test_config_with_custom_scopes(self):
        """Test AuthConfig with custom scopes"""
        config = AuthConfig(scopes=["read", "write", "admin"])

        assert config.scopes == ["read", "write", "admin"]


class TestAuthToken:
    """Tests for AuthToken dataclass"""

    def test_token_creation(self):
        """Test creating an AuthToken"""
        token = AuthToken(
            token_id="test-id",
            access_token="test-access-token"
        )

        assert token.token_id == "test-id"
        assert token.access_token == "test-access-token"
        assert token.token_type == "Bearer"

    def test_token_is_expired(self):
        """Test is_expired property"""
        # Expired token
        expired_token = AuthToken(
            token_id="expired",
            access_token="token",
            expires_at=datetime.now() - timedelta(hours=1)
        )
        assert expired_token.is_expired is True

        # Valid token
        valid_token = AuthToken(
            token_id="valid",
            access_token="token",
            expires_at=datetime.now() + timedelta(hours=1)
        )
        assert valid_token.is_expired is False

    def test_token_can_refresh(self):
        """Test can_refresh property"""
        # Token without refresh token
        no_refresh = AuthToken(
            token_id="test",
            access_token="token",
            refresh_token=None
        )
        assert no_refresh.can_refresh is False

        # Token with valid refresh
        with_refresh = AuthToken(
            token_id="test",
            access_token="token",
            refresh_token="refresh-token",
            refresh_expires_at=datetime.now() + timedelta(days=7)
        )
        assert with_refresh.can_refresh is True

        # Token with expired refresh
        expired_refresh = AuthToken(
            token_id="test",
            access_token="token",
            refresh_token="refresh-token",
            refresh_expires_at=datetime.now() - timedelta(hours=1)
        )
        assert expired_refresh.can_refresh is False

    def test_token_to_dict(self):
        """Test AuthToken serialization"""
        token = AuthToken(
            token_id="test-id",
            access_token="access",
            token_type="Bearer",
            expires_at=datetime(2025, 1, 1, 12, 0, 0),
            refresh_token="refresh",
            scopes=["read", "write"],
            user_id="user-1",
            provider=AuthProvider.OAUTH2
        )

        result = token.to_dict()

        assert result["token_id"] == "test-id"
        assert result["access_token"] == "access"
        assert result["token_type"] == "Bearer"
        assert result["expires_at"] == "2025-01-01T12:00:00"
        assert result["refresh_token"] == "refresh"
        assert result["scopes"] == ["read", "write"]
        assert result["user_id"] == "user-1"
        assert result["provider"] == "oauth2"

    def test_token_from_dict(self):
        """Test AuthToken deserialization"""
        data = {
            "token_id": "test-id",
            "access_token": "access",
            "token_type": "Bearer",
            "expires_at": "2025-01-01T12:00:00",
            "refresh_token": "refresh",
            "scopes": ["read"],
            "user_id": "user-1",
            "provider": "oauth2"
        }

        token = AuthToken.from_dict(data)

        assert token.token_id == "test-id"
        assert token.access_token == "access"
        assert token.user_id == "user-1"
        assert token.provider == AuthProvider.OAUTH2


class TestUserIdentity:
    """Tests for UserIdentity dataclass"""

    def test_identity_creation(self):
        """Test creating a UserIdentity"""
        identity = UserIdentity(
            user_id="user-123",
            email="user@example.com",
            name="Test User"
        )

        assert identity.user_id == "user-123"
        assert identity.email == "user@example.com"
        assert identity.name == "Test User"
        assert identity.groups == []
        assert identity.roles == []

    def test_identity_with_groups_and_roles(self):
        """Test identity with groups and roles"""
        identity = UserIdentity(
            user_id="user-123",
            email="user@example.com",
            groups=["admins", "developers"],
            roles=["admin", "contributor"]
        )

        assert "admins" in identity.groups
        assert "admin" in identity.roles

    def test_identity_to_dict(self):
        """Test UserIdentity serialization"""
        identity = UserIdentity(
            user_id="user-123",
            email="user@example.com",
            name="Test User",
            provider=AuthProvider.GOOGLE,
            groups=["team-a"],
            attributes={"department": "engineering"}
        )

        result = identity.to_dict()

        assert result["user_id"] == "user-123"
        assert result["email"] == "user@example.com"
        assert result["provider"] == "google"
        assert result["groups"] == ["team-a"]
        assert result["attributes"]["department"] == "engineering"


class TestAuthSession:
    """Tests for AuthSession dataclass"""

    def test_session_creation(self):
        """Test creating an AuthSession"""
        token = AuthToken(
            token_id="token-1",
            access_token="access",
            expires_at=datetime.now() + timedelta(hours=1)
        )
        identity = UserIdentity(
            user_id="user-1",
            email="user@example.com"
        )

        session = AuthSession(
            session_id="session-1",
            user_id="user-1",
            identity=identity,
            token=token
        )

        assert session.session_id == "session-1"
        assert session.user_id == "user-1"
        assert session.is_valid is True

    def test_session_is_valid_with_expired_token(self):
        """Test is_valid with expired token"""
        token = AuthToken(
            token_id="token-1",
            access_token="access",
            expires_at=datetime.now() - timedelta(hours=1)
        )
        identity = UserIdentity(user_id="user-1", email="user@example.com")

        session = AuthSession(
            session_id="session-1",
            user_id="user-1",
            identity=identity,
            token=token
        )

        assert session.is_valid is False

    def test_session_to_dict(self):
        """Test AuthSession serialization"""
        token = AuthToken(
            token_id="token-1",
            access_token="access",
            expires_at=datetime(2025, 1, 1, 12, 0, 0)
        )
        identity = UserIdentity(user_id="user-1", email="user@example.com")

        session = AuthSession(
            session_id="session-1",
            user_id="user-1",
            identity=identity,
            token=token,
            ip_address="192.168.1.1",
            user_agent="Mozilla/5.0"
        )

        result = session.to_dict()

        assert result["session_id"] == "session-1"
        assert result["user_id"] == "user-1"
        assert result["ip_address"] == "192.168.1.1"
        assert "identity" in result
        assert "token" in result


class TestAPIKeyProvider:
    """Tests for APIKeyProvider"""

    def test_create_api_key(self, tmp_path):
        """Test creating an API key"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="Test Key"
        )

        assert key_id is not None
        assert api_key.startswith("ucts_")
        assert len(api_key) > 20

    def test_create_api_key_with_scopes(self, tmp_path):
        """Test creating an API key with custom scopes"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="Admin Key",
            scopes=["admin", "read", "write"]
        )

        # Validate the key
        token = provider.validate_api_key(api_key)
        assert token is not None
        assert "admin" in token.scopes

    def test_create_api_key_with_expiration(self, tmp_path):
        """Test creating an API key with expiration"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="Expiring Key",
            expires_in=3600  # 1 hour
        )

        token = provider.validate_api_key(api_key)
        assert token is not None
        assert token.expires_at > datetime.now()
        assert token.expires_at < datetime.now() + timedelta(hours=2)

    def test_validate_api_key_success(self, tmp_path):
        """Test validating a valid API key"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="Test Key"
        )

        token = provider.validate_api_key(api_key)

        assert token is not None
        assert token.token_id == key_id
        assert token.user_id == "user-1"
        assert token.provider == AuthProvider.API_KEY

    def test_validate_api_key_invalid(self, tmp_path):
        """Test validating an invalid API key"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        result = provider.validate_api_key("ucts_invalid_key")

        assert result is None

    def test_validate_expired_api_key(self, tmp_path):
        """Test validating an expired API key"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="Expired Key",
            expires_in=1  # 1 second
        )

        # Manipulate the expiration
        provider._keys[key_id]["expires_at"] = (datetime.now() - timedelta(hours=1)).isoformat()
        provider._save_keys()

        result = provider.validate_api_key(api_key)
        assert result is None

    def test_revoke_api_key(self, tmp_path):
        """Test revoking an API key"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key(
            user_id="user-1",
            name="To Be Revoked"
        )

        # Revoke the key
        result = provider.revoke_api_key(key_id)
        assert result is True

        # Verify it's no longer valid
        token = provider.validate_api_key(api_key)
        assert token is None

    def test_revoke_nonexistent_key(self, tmp_path):
        """Test revoking a key that doesn't exist"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        result = provider.revoke_api_key("nonexistent")
        assert result is False

    def test_list_api_keys(self, tmp_path):
        """Test listing API keys for a user"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        # Create keys for different users
        provider.create_api_key("user-1", "Key 1")
        provider.create_api_key("user-1", "Key 2")
        provider.create_api_key("user-2", "Other Key")

        user1_keys = provider.list_api_keys("user-1")
        user2_keys = provider.list_api_keys("user-2")

        assert len(user1_keys) == 2
        assert len(user2_keys) == 1
        assert all(k["name"] in ["Key 1", "Key 2"] for k in user1_keys)

    def test_get_user_info(self, tmp_path):
        """Test getting user info from API key token"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        key_id, api_key = provider.create_api_key("user-1", "Test Key")
        token = provider.validate_api_key(api_key)

        identity = provider.get_user_info(token)

        assert identity.user_id == "user-1"
        assert identity.name == "Test Key"
        assert identity.provider == AuthProvider.API_KEY

    def test_persistence(self, tmp_path):
        """Test that keys persist across provider instances"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)

        # Create a key with first instance
        provider1 = APIKeyProvider(config, tmp_path)
        key_id, api_key = provider1.create_api_key("user-1", "Persistent Key")

        # Create new instance and verify key exists
        provider2 = APIKeyProvider(config, tmp_path)
        token = provider2.validate_api_key(api_key)

        assert token is not None
        assert token.token_id == key_id

    def test_unsupported_methods_raise(self, tmp_path):
        """Test that unsupported methods raise NotImplementedError"""
        config = AuthConfig(provider=AuthProvider.API_KEY, enabled=True)
        provider = APIKeyProvider(config, tmp_path)

        with pytest.raises(NotImplementedError):
            provider.get_authorization_url("state")

        with pytest.raises(NotImplementedError):
            provider.exchange_code("code", "state")

        with pytest.raises(NotImplementedError):
            provider.refresh_token(AuthToken(token_id="t", access_token="a"))


class TestOAuth2Provider:
    """Tests for OAuth2Provider"""

    def test_get_authorization_url(self):
        """Test generating authorization URL"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test-client",
            authorization_url="https://auth.example.com/authorize",
            scopes=["openid", "profile"]
        )
        provider = OAuth2Provider(config)

        url = provider.get_authorization_url("test-state")

        assert "https://auth.example.com/authorize" in url
        assert "client_id=test-client" in url
        assert "state=test-state" in url
        assert "response_type=code" in url
        assert "code_challenge" in url  # PKCE
        assert "code_challenge_method=S256" in url

    def test_exchange_code_stub(self):
        """Test code exchange (stub implementation)"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            session_timeout=7200
        )
        provider = OAuth2Provider(config)

        token = provider.exchange_code("auth-code", "state")

        assert token is not None
        assert token.access_token.startswith("stub_access_token_")
        assert token.refresh_token.startswith("stub_refresh_token_")
        assert token.provider == AuthProvider.OAUTH2

    def test_refresh_token_stub(self):
        """Test token refresh (stub implementation)"""
        config = AuthConfig(provider=AuthProvider.OAUTH2, enabled=True)
        provider = OAuth2Provider(config)

        original_token = AuthToken(
            token_id="orig",
            access_token="old-access",
            refresh_token="refresh-token",
            refresh_expires_at=datetime.now() + timedelta(days=7),
            scopes=["read"],
            user_id="user-1",
            provider=AuthProvider.OAUTH2
        )

        new_token = provider.refresh_token(original_token)

        assert new_token.access_token != original_token.access_token
        assert new_token.refresh_token == original_token.refresh_token  # Same refresh token
        assert new_token.user_id == original_token.user_id

    def test_refresh_token_fails_when_cannot_refresh(self):
        """Test that refresh fails when token cannot be refreshed"""
        config = AuthConfig(provider=AuthProvider.OAUTH2, enabled=True)
        provider = OAuth2Provider(config)

        token_without_refresh = AuthToken(
            token_id="t",
            access_token="a",
            refresh_token=None
        )

        with pytest.raises(ValueError, match="cannot be refreshed"):
            provider.refresh_token(token_without_refresh)

    def test_get_user_info_stub(self):
        """Test getting user info (stub implementation)"""
        config = AuthConfig(provider=AuthProvider.OAUTH2, enabled=True)
        provider = OAuth2Provider(config)

        token = AuthToken(token_id="t", access_token="a")
        identity = provider.get_user_info(token)

        assert identity is not None
        assert identity.email == "user@example.com"  # Stub value
        assert identity.provider == AuthProvider.OAUTH2

    def test_revoke_token_stub(self):
        """Test token revocation (stub implementation)"""
        config = AuthConfig(provider=AuthProvider.OAUTH2, enabled=True)
        provider = OAuth2Provider(config)

        token = AuthToken(token_id="t", access_token="a")
        result = provider.revoke_token(token)

        assert result is True


class TestSAMLProvider:
    """Tests for SAMLProvider"""

    def test_get_authorization_url(self):
        """Test generating SAML AuthnRequest URL"""
        config = AuthConfig(
            provider=AuthProvider.SAML,
            enabled=True,
            idp_sso_url="https://idp.example.com/sso"
        )
        provider = SAMLProvider(config)

        url = provider.get_authorization_url("relay-state")

        assert "https://idp.example.com/sso" in url
        assert "SAMLRequest=" in url
        assert "RelayState=relay-state" in url

    def test_exchange_code_stub(self):
        """Test SAML response processing (stub implementation)"""
        config = AuthConfig(provider=AuthProvider.SAML, enabled=True)
        provider = SAMLProvider(config)

        token = provider.exchange_code("saml-response", "relay-state")

        assert token is not None
        assert token.token_type == "SAML"
        assert token.provider == AuthProvider.SAML

    def test_refresh_token_not_supported(self):
        """Test that SAML doesn't support token refresh"""
        config = AuthConfig(provider=AuthProvider.SAML, enabled=True)
        provider = SAMLProvider(config)

        token = AuthToken(token_id="t", access_token="a")

        with pytest.raises(ValueError, match="cannot be refreshed"):
            provider.refresh_token(token)

    def test_get_user_info_stub(self):
        """Test getting user info from SAML assertion (stub)"""
        config = AuthConfig(provider=AuthProvider.SAML, enabled=True)
        provider = SAMLProvider(config)

        token = AuthToken(token_id="t", access_token="a")
        identity = provider.get_user_info(token)

        assert identity is not None
        assert identity.provider == AuthProvider.SAML


class TestAuthManager:
    """Tests for AuthManager"""

    def test_init_creates_api_key_provider(self, tmp_path):
        """Test that AuthManager always creates API key provider"""
        manager = AuthManager(str(tmp_path))

        provider = manager.get_provider(AuthProvider.API_KEY)
        assert provider is not None
        assert isinstance(provider, APIKeyProvider)

    def test_init_with_oauth_config(self, tmp_path):
        """Test AuthManager with OAuth configuration"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test",
            authorization_url="https://auth.example.com/authorize"
        )
        manager = AuthManager(str(tmp_path), config)

        oauth_provider = manager.get_provider(AuthProvider.OAUTH2)
        assert oauth_provider is not None
        assert isinstance(oauth_provider, OAuth2Provider)

    def test_init_with_saml_config(self, tmp_path):
        """Test AuthManager with SAML configuration"""
        config = AuthConfig(
            provider=AuthProvider.SAML,
            enabled=True,
            idp_sso_url="https://idp.example.com/sso"
        )
        manager = AuthManager(str(tmp_path), config)

        saml_provider = manager.get_provider(AuthProvider.SAML)
        assert saml_provider is not None
        assert isinstance(saml_provider, SAMLProvider)

    def test_start_oauth_flow(self, tmp_path):
        """Test starting OAuth flow"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test",
            authorization_url="https://auth.example.com/authorize"
        )
        manager = AuthManager(str(tmp_path), config)

        url, state = manager.start_oauth_flow()

        assert "https://auth.example.com/authorize" in url
        assert state is not None
        assert len(state) > 20

    def test_start_oauth_flow_unconfigured_provider(self, tmp_path):
        """Test starting OAuth flow with unconfigured provider"""
        manager = AuthManager(str(tmp_path))

        with pytest.raises(ValueError, match="not configured"):
            manager.start_oauth_flow(AuthProvider.GOOGLE)

    def test_complete_oauth_flow(self, tmp_path):
        """Test completing OAuth flow"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test"
        )
        manager = AuthManager(str(tmp_path), config)

        session = manager.complete_oauth_flow(
            code="auth-code",
            state="test-state",
            ip_address="192.168.1.1",
            user_agent="TestBrowser"
        )

        assert session is not None
        assert session.ip_address == "192.168.1.1"
        assert session.user_agent == "TestBrowser"
        assert session.is_valid

    def test_create_api_key(self, tmp_path):
        """Test creating API key through manager"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key(
            user_id="user-1",
            name="Manager Key"
        )

        assert key_id is not None
        assert api_key.startswith("ucts_")

    def test_authenticate_api_key(self, tmp_path):
        """Test authenticating with API key"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key("user-1", "Test Key")
        session = manager.authenticate_api_key(api_key)

        assert session is not None
        assert session.user_id == "user-1"
        assert session.is_valid

    def test_authenticate_invalid_api_key(self, tmp_path):
        """Test authenticating with invalid API key"""
        manager = AuthManager(str(tmp_path))

        session = manager.authenticate_api_key("ucts_invalid_key")
        assert session is None

    def test_list_api_keys(self, tmp_path):
        """Test listing API keys through manager"""
        manager = AuthManager(str(tmp_path))

        manager.create_api_key("user-1", "Key 1")
        manager.create_api_key("user-1", "Key 2")

        keys = manager.list_api_keys("user-1")

        assert len(keys) == 2

    def test_revoke_api_key(self, tmp_path):
        """Test revoking API key through manager"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key("user-1", "To Revoke")

        result = manager.revoke_api_key(key_id)
        assert result is True

        # Verify it's gone
        session = manager.authenticate_api_key(api_key)
        assert session is None

    def test_get_session(self, tmp_path):
        """Test getting session by ID"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key("user-1", "Test")
        session = manager.authenticate_api_key(api_key)

        retrieved = manager.get_session(session.session_id)

        assert retrieved is not None
        assert retrieved.session_id == session.session_id

    def test_get_session_nonexistent(self, tmp_path):
        """Test getting nonexistent session"""
        manager = AuthManager(str(tmp_path))

        result = manager.get_session("nonexistent")
        assert result is None

    def test_validate_token(self, tmp_path):
        """Test validating access token"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key("user-1", "Test")
        session = manager.authenticate_api_key(api_key)

        validated = manager.validate_token(session.token.access_token)

        assert validated is not None
        assert validated.session_id == session.session_id

    def test_validate_invalid_token(self, tmp_path):
        """Test validating invalid token"""
        manager = AuthManager(str(tmp_path))

        result = manager.validate_token("invalid-token")
        assert result is None

    def test_logout(self, tmp_path):
        """Test logging out a session"""
        manager = AuthManager(str(tmp_path))

        key_id, api_key = manager.create_api_key("user-1", "Test")
        session = manager.authenticate_api_key(api_key)
        session_id = session.session_id

        result = manager.logout(session_id)
        assert result is True

        # Verify session is gone
        retrieved = manager.get_session(session_id)
        assert retrieved is None

    def test_logout_nonexistent(self, tmp_path):
        """Test logging out nonexistent session"""
        manager = AuthManager(str(tmp_path))

        result = manager.logout("nonexistent")
        assert result is False

    def test_get_active_sessions(self, tmp_path):
        """Test getting active sessions for a user"""
        manager = AuthManager(str(tmp_path))

        # Create multiple sessions
        manager.create_api_key("user-1", "Key 1")
        key_id2, api_key2 = manager.create_api_key("user-1", "Key 2")
        manager.create_api_key("user-2", "Other")

        manager.authenticate_api_key(api_key2)

        sessions = manager.get_active_sessions("user-1")

        assert len(sessions) >= 1
        assert all(s.user_id == "user-1" for s in sessions)

    def test_cleanup_expired_sessions(self, tmp_path):
        """Test cleaning up expired sessions"""
        manager = AuthManager(str(tmp_path))

        # Create a session
        key_id, api_key = manager.create_api_key("user-1", "Test")
        session = manager.authenticate_api_key(api_key)

        # Manually expire the token
        session.token = AuthToken(
            token_id=session.token.token_id,
            access_token=session.token.access_token,
            expires_at=datetime.now() - timedelta(hours=1)
        )

        count = manager.cleanup_expired_sessions()

        assert count >= 1

    def test_session_persistence(self, tmp_path):
        """Test that sessions persist across manager instances"""
        config = AuthConfig(
            provider=AuthProvider.OAUTH2,
            enabled=True,
            client_id="test"
        )

        # Create session with first manager
        manager1 = AuthManager(str(tmp_path), config)
        session1 = manager1.complete_oauth_flow("code", "state")
        session_id = session1.session_id

        # Create new manager and check session exists
        manager2 = AuthManager(str(tmp_path), config)
        session2 = manager2.get_session(session_id)

        assert session2 is not None
        assert session2.session_id == session_id


class TestProviderConfigs:
    """Tests for pre-configured provider helpers"""

    def test_azure_ad_config(self):
        """Test Azure AD configuration helper"""
        config = get_azure_ad_config(
            tenant_id="test-tenant",
            client_id="test-client",
            client_secret="test-secret"
        )

        assert config.provider == AuthProvider.AZURE_AD
        assert config.enabled is True
        assert config.client_id == "test-client"
        assert config.client_secret == "test-secret"
        assert "test-tenant" in config.authorization_url
        assert "test-tenant" in config.token_url
        assert "graph.microsoft.com" in config.userinfo_url
        assert "User.Read" in config.scopes

    def test_okta_config(self):
        """Test Okta configuration helper"""
        config = get_okta_config(
            domain="dev-123.okta.com",
            client_id="okta-client",
            client_secret="okta-secret"
        )

        assert config.provider == AuthProvider.OKTA
        assert config.enabled is True
        assert "dev-123.okta.com" in config.authorization_url
        assert "dev-123.okta.com" in config.token_url
        assert "groups" in config.scopes

    def test_google_config(self):
        """Test Google configuration helper"""
        config = get_google_config(
            client_id="google-client",
            client_secret="google-secret"
        )

        assert config.provider == AuthProvider.GOOGLE
        assert config.enabled is True
        assert "accounts.google.com" in config.authorization_url
        assert "oauth2.googleapis.com" in config.token_url
        assert "openid" in config.scopes

    def test_github_config(self):
        """Test GitHub configuration helper"""
        config = get_github_config(
            client_id="github-client",
            client_secret="github-secret"
        )

        assert config.provider == AuthProvider.GITHUB
        assert config.enabled is True
        assert "github.com/login/oauth" in config.authorization_url
        assert "api.github.com/user" in config.userinfo_url
        assert "read:user" in config.scopes
        assert "user:email" in config.scopes
