"""
Tests for UCTS CLI

Tests the cli.py module including:
- Main CLI group
- ingest, forge, status, transfer, analyze commands
- docker, template commands
"""

import json
import pytest
from pathlib import Path
from unittest.mock import MagicMock, patch
from click.testing import CliRunner

from ucts.cli import cli


@pytest.fixture
def runner():
    """Create CLI test runner"""
    return CliRunner()


@pytest.fixture
def sample_session_file(tmp_path):
    """Create a sample session file"""
    session = {
        "messages": [
            {"role": "user", "content": "Create a Python app"},
            {"role": "assistant", "content": "Here's a simple app:\n```python\nprint('hello')\n```"}
        ]
    }
    file_path = tmp_path / "session.json"
    file_path.write_text(json.dumps(session))
    return str(file_path)


# ============================================================================
# CLI Group Tests
# ============================================================================

class TestCLIGroup:
    """Tests for main CLI group"""

    def test_cli_version(self, runner):
        """Test --version flag"""
        result = runner.invoke(cli, ['--version'])
        assert result.exit_code == 0
        assert '2.0.0' in result.output

    def test_cli_help(self, runner):
        """Test --help flag"""
        result = runner.invoke(cli, ['--help'])
        assert result.exit_code == 0
        assert 'UCTS' in result.output
        assert 'Universal Context Transfer System' in result.output

    def test_cli_commands_available(self, runner):
        """Test all commands are registered"""
        result = runner.invoke(cli, ['--help'])
        assert 'ingest' in result.output
        assert 'forge' in result.output
        assert 'status' in result.output
        assert 'transfer' in result.output
        assert 'analyze' in result.output


# ============================================================================
# Ingest Command Tests
# ============================================================================

class TestIngestCommand:
    """Tests for ingest command"""

    def test_ingest_help(self, runner):
        """Test ingest --help"""
        result = runner.invoke(cli, ['ingest', '--help'])
        assert result.exit_code == 0
        assert 'SOURCE' in result.output

    def test_ingest_with_file(self, runner, sample_session_file):
        """Test ingesting a file"""
        result = runner.invoke(cli, ['ingest', sample_session_file])
        assert result.exit_code == 0
        assert 'ingested' in result.output.lower() or 'messages' in result.output.lower()

    def test_ingest_with_format_option(self, runner, sample_session_file):
        """Test ingest with --format option"""
        result = runner.invoke(cli, ['ingest', sample_session_file, '--format', 'web'])
        # Should work or fail gracefully
        assert result.exit_code in [0, 1]

    def test_ingest_nonexistent_file(self, runner):
        """Test ingest with nonexistent file"""
        result = runner.invoke(cli, ['ingest', '/nonexistent/file.json'])
        assert result.exit_code != 0


# ============================================================================
# Forge Command Tests
# ============================================================================

class TestForgeCommand:
    """Tests for forge command"""

    def test_forge_help(self, runner):
        """Test forge --help"""
        result = runner.invoke(cli, ['forge', '--help'])
        assert result.exit_code == 0
        assert 'SOURCES' in result.output
        assert 'OUTPUT' in result.output
        assert '--target' in result.output

    def test_forge_basic(self, runner, sample_session_file, tmp_path):
        """Test basic forge command"""
        output_dir = str(tmp_path / "output")
        result = runner.invoke(cli, ['forge', sample_session_file, output_dir])
        assert result.exit_code == 0
        assert 'created' in result.output.lower() or 'generating' in result.output.lower()

    def test_forge_with_name(self, runner, sample_session_file, tmp_path):
        """Test forge with --name option"""
        output_dir = str(tmp_path / "output")
        result = runner.invoke(cli, ['forge', sample_session_file, output_dir, '--name', 'test-project'])
        assert result.exit_code == 0

    def test_forge_gitlab_requires_credentials(self, runner, sample_session_file, tmp_path):
        """Test forge gitlab requires credentials"""
        output_dir = str(tmp_path / "output")
        result = runner.invoke(cli, ['forge', sample_session_file, output_dir, '--target', 'gitlab'])
        # Should fail or warn about missing credentials
        assert 'token' in result.output.lower() or 'error' in result.output.lower() or result.exit_code != 0

    def test_forge_github_requires_token(self, runner, sample_session_file, tmp_path):
        """Test forge github requires token"""
        output_dir = str(tmp_path / "output")
        result = runner.invoke(cli, ['forge', sample_session_file, output_dir, '--target', 'github'])
        assert 'token' in result.output.lower() or 'error' in result.output.lower() or result.exit_code != 0


# ============================================================================
# Status Command Tests
# ============================================================================

class TestStatusCommand:
    """Tests for status command"""

    def test_status_help(self, runner):
        """Test status --help"""
        result = runner.invoke(cli, ['status', '--help'])
        assert result.exit_code == 0
        assert '--project' in result.output

    def test_status_default(self, runner):
        """Test status with default project"""
        result = runner.invoke(cli, ['status'])
        assert result.exit_code == 0
        assert 'Status' in result.output or 'Project' in result.output

    def test_status_with_project(self, runner):
        """Test status with custom project name"""
        result = runner.invoke(cli, ['status', '--project', 'test-project'])
        assert result.exit_code == 0


# ============================================================================
# Transfer Command Tests
# ============================================================================

class TestTransferCommand:
    """Tests for transfer command"""

    def test_transfer_help(self, runner):
        """Test transfer --help"""
        result = runner.invoke(cli, ['transfer', '--help'])
        assert result.exit_code == 0
        assert '--project' in result.output
        assert '--output' in result.output

    def test_transfer_default(self, runner):
        """Test transfer with default project"""
        result = runner.invoke(cli, ['transfer'])
        assert result.exit_code == 0
        assert 'Transfer' in result.output or 'package' in result.output.lower()

    def test_transfer_with_output(self, runner, tmp_path):
        """Test transfer with output file"""
        output_file = str(tmp_path / "transfer.json")
        result = runner.invoke(cli, ['transfer', '--output', output_file])
        assert result.exit_code == 0


# ============================================================================
# Analyze Command Tests
# ============================================================================

class TestAnalyzeCommand:
    """Tests for analyze command"""

    def test_analyze_help(self, runner):
        """Test analyze --help"""
        result = runner.invoke(cli, ['analyze', '--help'])
        assert result.exit_code == 0
        assert 'SOURCE' in result.output

    def test_analyze_with_file(self, runner, sample_session_file):
        """Test analyzing a file"""
        result = runner.invoke(cli, ['analyze', sample_session_file])
        assert result.exit_code == 0
        assert 'Analysis' in result.output or 'Languages' in result.output


# ============================================================================
# Docker Command Tests
# ============================================================================

class TestDockerCommand:
    """Tests for docker command"""

    def test_docker_help(self, runner):
        """Test docker --help"""
        result = runner.invoke(cli, ['docker', '--help'])
        assert result.exit_code == 0
        assert 'SOURCE' in result.output
        assert 'OUTPUT' in result.output
        assert '--port' in result.output

    def test_docker_basic(self, runner, sample_session_file, tmp_path):
        """Test docker command"""
        output_dir = str(tmp_path / "docker_output")
        result = runner.invoke(cli, ['docker', sample_session_file, output_dir])
        assert result.exit_code == 0

    def test_docker_with_options(self, runner, sample_session_file, tmp_path):
        """Test docker with options"""
        output_dir = str(tmp_path / "docker_output")
        result = runner.invoke(cli, [
            'docker', sample_session_file, output_dir,
            '--port', '3000',
            '--python-version', '3.12',
            '--alpine'
        ])
        assert result.exit_code == 0


# ============================================================================
# Template Command Tests
# ============================================================================

class TestTemplateCommand:
    """Tests for template command"""

    def test_template_help(self, runner):
        """Test template --help"""
        result = runner.invoke(cli, ['template', '--help'])
        assert result.exit_code == 0
        assert 'TEMPLATE_NAME' in result.output
        assert 'OUTPUT' in result.output


# ============================================================================
# Integration Tests
# ============================================================================

class TestCLIIntegration:
    """Integration tests for CLI"""

    def test_full_workflow(self, runner, sample_session_file, tmp_path):
        """Test full ingest -> analyze -> forge workflow"""
        # Ingest
        result = runner.invoke(cli, ['ingest', sample_session_file])
        assert result.exit_code == 0

        # Analyze
        result = runner.invoke(cli, ['analyze', sample_session_file])
        assert result.exit_code == 0

        # Forge
        output_dir = str(tmp_path / "project")
        result = runner.invoke(cli, ['forge', sample_session_file, output_dir])
        assert result.exit_code == 0
