"""
Tests for Cloud Deployment Generators

Tests the generators/cloud.py module including:
- CloudConfig dataclass
- AWSGenerator, GCPGenerator, AzureGenerator, VercelGenerator, RailwayGenerator
- CloudGenerator unified class
"""

import json
import pytest
from pathlib import Path
from dataclasses import dataclass, field
from typing import List, Dict, Any

from ucts.generators.cloud import (
    CloudConfig,
    AWSGenerator,
    GCPGenerator,
    AzureGenerator,
    VercelGenerator,
    RailwayGenerator,
    CloudGenerator,
)


# ============================================================================
# Mock Structure for Testing
# ============================================================================

@dataclass
class MockStructure:
    """Mock project structure for testing"""
    languages: List[str] = field(default_factory=lambda: ["python"])
    name: str = "test-app"
    dependencies: Dict[str, Any] = field(default_factory=dict)


# ============================================================================
# CloudConfig Tests
# ============================================================================

class TestCloudConfig:
    """Tests for CloudConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = CloudConfig()
        assert config.project_name == "app"
        assert config.region == "us-east-1"
        assert config.environment == "production"
        assert config.memory_mb == 512
        assert config.cpu == 0.5
        assert config.min_instances == 1
        assert config.max_instances == 10
        assert config.port == 8000
        assert config.health_check_path == "/health"
        assert config.include_database is False
        assert config.database_type == "postgres"
        assert config.env_vars == {}

    def test_custom_values(self):
        """Test custom configuration values"""
        config = CloudConfig(
            project_name="my-app",
            region="eu-west-1",
            environment="staging",
            memory_mb=1024,
            cpu=1.0,
            min_instances=2,
            max_instances=20,
            port=3000,
            include_database=True,
            database_type="mysql",
            env_vars={"API_KEY": "secret"}
        )
        assert config.project_name == "my-app"
        assert config.region == "eu-west-1"
        assert config.environment == "staging"
        assert config.memory_mb == 1024
        assert config.include_database is True
        assert config.env_vars["API_KEY"] == "secret"


# ============================================================================
# AWSGenerator Tests
# ============================================================================

class TestAWSGenerator:
    """Tests for AWS CloudFormation/SAM generator"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = AWSGenerator()
        assert gen.config is not None
        assert gen.config.project_name == "app"

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = CloudConfig(project_name="my-aws-app")
        gen = AWSGenerator(config)
        assert gen.config.project_name == "my-aws-app"

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = AWSGenerator()
        structure = MockStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "template.yaml" in files
        assert "samconfig.toml" in files
        assert "buildspec.yml" in files

    def test_generate_writes_to_disk(self, tmp_path):
        """Test files are written to disk"""
        gen = AWSGenerator()
        structure = MockStructure()

        gen.generate(structure, str(tmp_path))

        assert (tmp_path / "template.yaml").exists()
        assert (tmp_path / "samconfig.toml").exists()
        assert (tmp_path / "buildspec.yml").exists()

    def test_sam_template_python(self):
        """Test SAM template for Python"""
        gen = AWSGenerator()
        template = gen._generate_sam_template(is_python=True, is_node=False)

        assert "python3.11" in template
        assert "AWS::Serverless::Function" in template
        assert "AWS::Serverless::Api" in template

    def test_sam_template_node(self):
        """Test SAM template for Node.js"""
        gen = AWSGenerator()
        template = gen._generate_sam_template(is_python=False, is_node=True)

        assert "nodejs20.x" in template
        assert "index.handler" in template

    def test_sam_template_with_database(self):
        """Test SAM template includes database when configured"""
        config = CloudConfig(include_database=True, database_type="postgres")
        gen = AWSGenerator(config)
        template = gen._generate_sam_template(is_python=True, is_node=False)

        assert "AWS::RDS::DBInstance" in template
        assert "postgres" in template

    def test_samconfig(self):
        """Test samconfig.toml generation"""
        config = CloudConfig(project_name="test-app", region="us-west-2")
        gen = AWSGenerator(config)
        samconfig = gen._generate_samconfig()

        assert "test-app-stack" in samconfig
        assert "us-west-2" in samconfig

    def test_buildspec_python(self):
        """Test buildspec for Python"""
        gen = AWSGenerator()
        buildspec = gen._generate_buildspec(is_python=True, is_node=False)

        assert "python: 3.11" in buildspec
        assert "pip install" in buildspec

    def test_buildspec_node(self):
        """Test buildspec for Node.js"""
        gen = AWSGenerator()
        buildspec = gen._generate_buildspec(is_python=False, is_node=True)

        assert "nodejs: 20" in buildspec
        assert "npm ci" in buildspec


# ============================================================================
# GCPGenerator Tests
# ============================================================================

class TestGCPGenerator:
    """Tests for GCP Terraform + Cloud Run generator"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = GCPGenerator()
        assert gen.config is not None

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = GCPGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "main.tf" in files
        assert "variables.tf" in files
        assert "outputs.tf" in files
        assert "cloudbuild.yaml" in files

    def test_main_tf_content(self):
        """Test main.tf contains required resources"""
        config = CloudConfig(project_name="gcp-app")
        gen = GCPGenerator(config)
        main_tf = gen._generate_main_tf()

        assert 'terraform {' in main_tf
        assert 'google_cloud_run_v2_service' in main_tf
        assert 'google_artifact_registry_repository' in main_tf
        assert 'gcp-app' in main_tf

    def test_main_tf_with_database(self):
        """Test main.tf includes Cloud SQL when configured"""
        config = CloudConfig(include_database=True)
        gen = GCPGenerator(config)
        main_tf = gen._generate_main_tf()

        assert 'google_sql_database_instance' in main_tf
        assert 'google_sql_database' in main_tf

    def test_variables_tf(self):
        """Test variables.tf content"""
        config = CloudConfig(environment="staging")
        gen = GCPGenerator(config)
        variables = gen._generate_variables_tf()

        assert 'variable "project_id"' in variables
        assert 'variable "region"' in variables
        assert 'staging' in variables

    def test_outputs_tf(self):
        """Test outputs.tf content"""
        gen = GCPGenerator()
        outputs = gen._generate_outputs_tf()

        assert 'output "service_url"' in outputs
        assert 'output "service_name"' in outputs

    def test_cloudbuild_yaml(self):
        """Test cloudbuild.yaml content"""
        gen = GCPGenerator()
        cloudbuild = gen._generate_cloudbuild(is_python=True)

        assert 'gcr.io/cloud-builders/docker' in cloudbuild
        assert 'gcr.io/cloud-builders/gcloud' in cloudbuild
        assert 'run' in cloudbuild
        assert 'deploy' in cloudbuild


# ============================================================================
# AzureGenerator Tests
# ============================================================================

class TestAzureGenerator:
    """Tests for Azure ARM templates generator"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = AzureGenerator()
        assert gen.config is not None

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = AzureGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "azuredeploy.json" in files
        assert "azuredeploy.parameters.json" in files
        assert "azure-pipelines.yml" in files

    def test_arm_template_valid_json(self):
        """Test ARM template is valid JSON"""
        gen = AzureGenerator()
        template_str = gen._generate_arm_template()
        template = json.loads(template_str)

        assert "$schema" in template
        assert "contentVersion" in template
        assert "resources" in template

    def test_arm_template_resources(self):
        """Test ARM template contains required resources"""
        config = CloudConfig(project_name="azure-app")
        gen = AzureGenerator(config)
        template_str = gen._generate_arm_template()
        template = json.loads(template_str)

        resource_types = [r["type"] for r in template["resources"]]
        assert "Microsoft.Web/serverfarms" in resource_types
        assert "Microsoft.Web/sites" in resource_types

    def test_parameters_valid_json(self):
        """Test parameters file is valid JSON"""
        gen = AzureGenerator()
        params_str = gen._generate_parameters()
        params = json.loads(params_str)

        assert "parameters" in params
        assert "appName" in params["parameters"]

    def test_azure_pipeline(self):
        """Test Azure Pipeline YAML content"""
        config = CloudConfig(project_name="test-app", environment="prod")
        gen = AzureGenerator(config)
        pipeline = gen._generate_pipeline()

        assert "trigger:" in pipeline
        assert "test-app" in pipeline
        assert "AzureRmWebAppDeployment" in pipeline


# ============================================================================
# VercelGenerator Tests
# ============================================================================

class TestVercelGenerator:
    """Tests for Vercel configuration generator"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = VercelGenerator()
        assert gen.config is not None

    def test_generate_creates_files_python(self, tmp_path):
        """Test generate creates files for Python"""
        gen = VercelGenerator()
        structure = MockStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "vercel.json" in files
        assert "api/index.py" in files

    def test_generate_creates_files_node(self, tmp_path):
        """Test generate creates files for Node.js"""
        gen = VercelGenerator()
        structure = MockStructure(languages=["javascript"])

        files = gen.generate(structure, str(tmp_path))

        assert "vercel.json" in files
        assert "api/index.ts" in files

    def test_vercel_json_valid(self):
        """Test vercel.json is valid JSON"""
        gen = VercelGenerator()
        vercel_str = gen._generate_vercel_json(is_python=True)
        vercel = json.loads(vercel_str)

        assert vercel["version"] == 2
        assert "builds" in vercel
        assert "routes" in vercel

    def test_vercel_json_python_config(self):
        """Test vercel.json configured for Python"""
        gen = VercelGenerator()
        vercel_str = gen._generate_vercel_json(is_python=True)
        vercel = json.loads(vercel_str)

        assert vercel["builds"][0]["use"] == "@vercel/python"
        assert "api/**/*.py" in vercel["builds"][0]["src"]

    def test_vercel_json_node_config(self):
        """Test vercel.json configured for Node.js"""
        gen = VercelGenerator()
        vercel_str = gen._generate_vercel_json(is_python=False)
        vercel = json.loads(vercel_str)

        assert vercel["builds"][0]["use"] == "@vercel/node"

    def test_python_api_handler(self):
        """Test Python API handler content"""
        gen = VercelGenerator()
        handler = gen._generate_python_api()

        assert "class handler" in handler
        assert "def do_GET" in handler
        assert "def do_POST" in handler

    def test_node_api_handler(self):
        """Test Node.js API handler content"""
        gen = VercelGenerator()
        handler = gen._generate_node_api()

        assert "VercelRequest" in handler
        assert "VercelResponse" in handler
        assert "export default function handler" in handler


# ============================================================================
# RailwayGenerator Tests
# ============================================================================

class TestRailwayGenerator:
    """Tests for Railway configuration generator"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = RailwayGenerator()
        assert gen.config is not None

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = RailwayGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "railway.json" in files
        assert "Procfile" in files
        assert "nixpacks.toml" in files

    def test_railway_json_valid(self):
        """Test railway.json is valid JSON"""
        gen = RailwayGenerator()
        railway_str = gen._generate_railway_json(is_python=True)
        railway = json.loads(railway_str)

        assert "$schema" in railway
        assert "build" in railway
        assert "deploy" in railway

    def test_railway_json_config(self):
        """Test railway.json configuration"""
        config = CloudConfig(min_instances=2, health_check_path="/api/health")
        gen = RailwayGenerator(config)
        railway_str = gen._generate_railway_json(is_python=True)
        railway = json.loads(railway_str)

        assert railway["deploy"]["numReplicas"] == 2
        assert railway["deploy"]["healthcheckPath"] == "/api/health"

    def test_procfile_python(self):
        """Test Procfile for Python"""
        gen = RailwayGenerator()
        procfile = gen._generate_procfile(is_python=True)

        assert "web:" in procfile
        assert "uvicorn" in procfile

    def test_procfile_node(self):
        """Test Procfile for Node.js"""
        gen = RailwayGenerator()
        procfile = gen._generate_procfile(is_python=False)

        assert "web:" in procfile
        assert "npm start" in procfile

    def test_nixpacks_python(self):
        """Test nixpacks.toml for Python"""
        gen = RailwayGenerator()
        nixpacks = gen._generate_nixpacks(is_python=True)

        assert "python311" in nixpacks
        assert "pip install" in nixpacks

    def test_nixpacks_node(self):
        """Test nixpacks.toml for Node.js"""
        gen = RailwayGenerator()
        nixpacks = gen._generate_nixpacks(is_python=False)

        assert "nodejs" in nixpacks
        assert "npm ci" in nixpacks


# ============================================================================
# CloudGenerator Unified Tests
# ============================================================================

class TestCloudGenerator:
    """Tests for unified CloudGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = CloudGenerator()
        assert gen.config is not None

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = CloudConfig(project_name="unified-app")
        gen = CloudGenerator(config)
        assert gen.config.project_name == "unified-app"

    def test_platforms_mapping(self):
        """Test all platforms are mapped"""
        assert "aws" in CloudGenerator.PLATFORMS
        assert "gcp" in CloudGenerator.PLATFORMS
        assert "azure" in CloudGenerator.PLATFORMS
        assert "vercel" in CloudGenerator.PLATFORMS
        assert "railway" in CloudGenerator.PLATFORMS

    def test_generate_aws(self, tmp_path):
        """Test generating AWS files"""
        gen = CloudGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path), "aws")

        assert "template.yaml" in files
        assert "samconfig.toml" in files

    def test_generate_gcp(self, tmp_path):
        """Test generating GCP files"""
        gen = CloudGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path), "gcp")

        assert "main.tf" in files
        assert "variables.tf" in files

    def test_generate_azure(self, tmp_path):
        """Test generating Azure files"""
        gen = CloudGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path), "azure")

        assert "azuredeploy.json" in files

    def test_generate_vercel(self, tmp_path):
        """Test generating Vercel files"""
        gen = CloudGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path), "vercel")

        assert "vercel.json" in files

    def test_generate_railway(self, tmp_path):
        """Test generating Railway files"""
        gen = CloudGenerator()
        structure = MockStructure()

        files = gen.generate(structure, str(tmp_path), "railway")

        assert "railway.json" in files

    def test_generate_unknown_platform_raises(self, tmp_path):
        """Test unknown platform raises error"""
        gen = CloudGenerator()
        structure = MockStructure()

        with pytest.raises(ValueError, match="Unknown platform"):
            gen.generate(structure, str(tmp_path), "unknown")

    def test_generate_all(self, tmp_path):
        """Test generating for all platforms"""
        gen = CloudGenerator()
        structure = MockStructure()

        results = gen.generate_all(structure, str(tmp_path))

        assert "aws" in results
        assert "gcp" in results
        assert "azure" in results
        assert "vercel" in results
        assert "railway" in results

        # Check subdirectories created
        assert (tmp_path / "aws").exists()
        assert (tmp_path / "gcp").exists()

    def test_list_platforms(self):
        """Test listing available platforms"""
        platforms = CloudGenerator.list_platforms()

        assert isinstance(platforms, list)
        assert len(platforms) == 5

        names = [p["name"] for p in platforms]
        assert "aws" in names
        assert "gcp" in names
        assert "azure" in names
        assert "vercel" in names
        assert "railway" in names


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration tests for cloud generators"""

    def test_full_workflow_python_project(self, tmp_path):
        """Test generating cloud configs for Python project"""
        config = CloudConfig(
            project_name="python-api",
            environment="staging",
            include_database=True
        )
        gen = CloudGenerator(config)
        structure = MockStructure(languages=["python"])

        # Generate for AWS
        aws_files = gen.generate(structure, str(tmp_path / "aws"), "aws")
        assert "template.yaml" in aws_files
        # Verify database is included
        assert "RDS" in aws_files["template.yaml"]

    def test_full_workflow_node_project(self, tmp_path):
        """Test generating cloud configs for Node.js project"""
        config = CloudConfig(project_name="node-api")
        gen = CloudGenerator(config)
        structure = MockStructure(languages=["javascript", "typescript"])

        # Generate for Vercel
        vercel_files = gen.generate(structure, str(tmp_path / "vercel"), "vercel")
        vercel_config = json.loads(vercel_files["vercel.json"])

        assert "@vercel/node" in vercel_config["builds"][0]["use"]

    def test_config_propagates_to_all_platforms(self, tmp_path):
        """Test configuration values propagate correctly"""
        config = CloudConfig(
            project_name="test-project",
            port=3000,
            health_check_path="/api/health"
        )
        gen = CloudGenerator(config)
        structure = MockStructure()

        # Check GCP
        gcp_files = gen.generate(structure, str(tmp_path / "gcp"), "gcp")
        assert "3000" in gcp_files["main.tf"]
        assert "/api/health" in gcp_files["main.tf"]

        # Check Railway
        railway_files = gen.generate(structure, str(tmp_path / "railway"), "railway")
        railway_config = json.loads(railway_files["railway.json"])
        assert railway_config["deploy"]["healthcheckPath"] == "/api/health"
