"""Tests for Coherence integration"""
import json
from pathlib import Path
from unittest.mock import patch, mock_open, MagicMock

import pytest

from ucts.integrations.coherence import (
    CoherenceIntegration,
    CoherenceError,
    CoherenceConnectionError,
    CoherenceStateError,
)


class TestCoherenceIntegration:
    """Tests for Coherence integration"""

    def test_init_with_custom_path(self, tmp_path):
        """Test initialization with custom path"""
        custom_path = tmp_path / "custom" / "state.json"
        integration = CoherenceIntegration(str(custom_path))

        assert integration.coherence_path == custom_path

    def test_init_finds_existing_state_file(self, tmp_path, monkeypatch):
        """Test that init finds existing state file"""
        state_file = tmp_path / "coherence_state.json"
        state_file.write_text("{}")

        monkeypatch.chdir(tmp_path)

        integration = CoherenceIntegration()
        assert integration.coherence_path == state_file

    def test_init_defaults_to_first_path(self, tmp_path, monkeypatch):
        """Test that init uses default path when none exists"""
        monkeypatch.chdir(tmp_path)

        integration = CoherenceIntegration()
        assert "coherence_state.json" in str(integration.coherence_path)

    def test_export_state_creates_file(self, tmp_path):
        """Test that export_state creates the state file"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.export_state(
            project_id="test-project",
            status="operational",
            completed_work=["task1", "task2"],
            available_actions=["forge", "transfer"]
        )

        assert state_path.exists()
        assert result["project_id"] == "test-project"
        assert result["status"] == "operational"
        assert result["system"] == "ucts"
        assert "exported_at" in result
        assert len(result["completed_work"]) == 2
        assert len(result["available_actions"]) == 2

    def test_export_state_includes_capabilities(self, tmp_path):
        """Test that export_state includes UCTS capabilities"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.export_state(
            project_id="test",
            status="operational",
            completed_work=[],
            available_actions=[]
        )

        assert "session_ingestion" in result["capabilities"]
        assert "project_generation" in result["capabilities"]
        assert "vscode_workspace" in result["capabilities"]

    def test_export_state_requires_project_id(self, tmp_path):
        """Test that export_state requires project_id"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="project_id must be a non-empty string"):
            integration.export_state(
                project_id="",
                status="operational",
                completed_work=[],
                available_actions=[]
            )

    def test_export_state_requires_string_project_id(self, tmp_path):
        """Test that export_state requires project_id to be a string"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="project_id must be a non-empty string"):
            integration.export_state(
                project_id=123,  # type: ignore
                status="operational",
                completed_work=[],
                available_actions=[]
            )

    def test_export_state_converts_non_list_completed_work(self, tmp_path):
        """Test that non-list completed_work is converted"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.export_state(
            project_id="test",
            status="operational",
            completed_work=("task1", "task2"),  # tuple instead of list
            available_actions=[]
        )

        assert isinstance(result["completed_work"], list)
        assert len(result["completed_work"]) == 2

    def test_export_state_handles_permission_error(self, tmp_path):
        """Test handling of permission error during export"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with patch("builtins.open", side_effect=PermissionError("Access denied")):
            with pytest.raises(CoherenceStateError, match="Permission denied"):
                integration.export_state(
                    project_id="test",
                    status="operational",
                    completed_work=[],
                    available_actions=[]
                )

    def test_import_state_returns_none_when_missing(self, tmp_path):
        """Test that import_state returns None when file doesn't exist"""
        state_path = tmp_path / "nonexistent.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.import_state()
        assert result is None

    def test_import_state_reads_file(self, tmp_path):
        """Test that import_state reads existing file"""
        state_path = tmp_path / "coherence_state.json"
        state_data = {"project_id": "test", "status": "operational"}
        state_path.write_text(json.dumps(state_data))

        integration = CoherenceIntegration(str(state_path))
        result = integration.import_state()

        assert result["project_id"] == "test"
        assert result["status"] == "operational"

    def test_import_state_handles_invalid_json(self, tmp_path):
        """Test handling of invalid JSON during import"""
        state_path = tmp_path / "coherence_state.json"
        state_path.write_text("not valid json")

        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(CoherenceStateError, match="Invalid JSON"):
            integration.import_state()

    def test_import_state_handles_permission_error(self, tmp_path):
        """Test handling of permission error during import"""
        state_path = tmp_path / "coherence_state.json"
        state_path.write_text("{}")

        integration = CoherenceIntegration(str(state_path))

        with patch("builtins.open", side_effect=PermissionError("Access denied")):
            with pytest.raises(CoherenceStateError, match="Permission denied"):
                integration.import_state()

    def test_register_workflow_creates_workflow(self, tmp_path):
        """Test that register_workflow creates workflow entry"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.register_workflow(
            workflow_id="wf-001",
            workflow_type="forge",
            source_project="project-a",
            target_project="project-b"
        )

        assert result["workflow_id"] == "wf-001"
        assert result["type"] == "forge"
        assert result["source"] == "project-a"
        assert result["target"] == "project-b"
        assert result["status"] == "registered"
        assert "registered_at" in result

        # Verify file was created
        workflows_path = tmp_path / "workflows.json"
        assert workflows_path.exists()

    def test_register_workflow_appends_to_existing(self, tmp_path):
        """Test that register_workflow appends to existing workflows"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text(json.dumps([{"workflow_id": "wf-000"}]))

        integration = CoherenceIntegration(str(state_path))
        integration.register_workflow(
            workflow_id="wf-001",
            workflow_type="transfer",
            source_project="a",
            target_project="b"
        )

        workflows = json.loads(workflows_path.read_text())
        assert len(workflows) == 2

    def test_register_workflow_requires_workflow_id(self, tmp_path):
        """Test that register_workflow requires workflow_id"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="workflow_id must be non-empty"):
            integration.register_workflow(
                workflow_id="",
                workflow_type="forge",
                source_project="a",
                target_project="b"
            )

    def test_register_workflow_requires_workflow_type(self, tmp_path):
        """Test that register_workflow requires workflow_type"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="workflow_type must be non-empty"):
            integration.register_workflow(
                workflow_id="wf-001",
                workflow_type="",
                source_project="a",
                target_project="b"
            )

    def test_register_workflow_requires_source_project(self, tmp_path):
        """Test that register_workflow requires source_project"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="source_project must be non-empty"):
            integration.register_workflow(
                workflow_id="wf-001",
                workflow_type="forge",
                source_project="",
                target_project="b"
            )

    def test_register_workflow_requires_target_project(self, tmp_path):
        """Test that register_workflow requires target_project"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="target_project must be non-empty"):
            integration.register_workflow(
                workflow_id="wf-001",
                workflow_type="forge",
                source_project="a",
                target_project=""
            )

    def test_register_workflow_handles_corrupted_file(self, tmp_path):
        """Test handling of corrupted workflows file"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text("not valid json")

        integration = CoherenceIntegration(str(state_path))
        result = integration.register_workflow(
            workflow_id="wf-001",
            workflow_type="forge",
            source_project="a",
            target_project="b"
        )

        # Should still succeed, resetting the file
        assert result["workflow_id"] == "wf-001"
        workflows = json.loads(workflows_path.read_text())
        assert len(workflows) == 1

    def test_register_workflow_handles_non_list_file(self, tmp_path):
        """Test handling of non-list workflows file"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text('{"not": "a list"}')

        integration = CoherenceIntegration(str(state_path))
        result = integration.register_workflow(
            workflow_id="wf-001",
            workflow_type="forge",
            source_project="a",
            target_project="b"
        )

        # Should reset to list
        workflows = json.loads(workflows_path.read_text())
        assert isinstance(workflows, list)

    def test_notify_completion_updates_workflow(self, tmp_path):
        """Test that notify_completion updates workflow status"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text(json.dumps([
            {"workflow_id": "wf-001", "status": "registered"}
        ]))

        integration = CoherenceIntegration(str(state_path))
        result = integration.notify_completion(
            workflow_id="wf-001",
            result={"output": "success"}
        )

        assert result is True

        workflows = json.loads(workflows_path.read_text())
        assert workflows[0]["status"] == "completed"
        assert workflows[0]["result"]["output"] == "success"
        assert "completed_at" in workflows[0]

    def test_notify_completion_returns_false_for_missing_workflow(self, tmp_path):
        """Test that notify_completion returns False for missing workflow"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text(json.dumps([
            {"workflow_id": "other-workflow", "status": "registered"}
        ]))

        integration = CoherenceIntegration(str(state_path))
        result = integration.notify_completion(
            workflow_id="wf-001",
            result={"output": "success"}
        )

        assert result is False

    def test_notify_completion_returns_false_for_missing_file(self, tmp_path):
        """Test that notify_completion returns False when file doesn't exist"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        result = integration.notify_completion(
            workflow_id="wf-001",
            result={"output": "success"}
        )

        assert result is False

    def test_notify_completion_requires_workflow_id(self, tmp_path):
        """Test that notify_completion requires workflow_id"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(ValueError, match="workflow_id must be non-empty"):
            integration.notify_completion(workflow_id="", result={})

    def test_notify_completion_converts_non_dict_result(self, tmp_path):
        """Test that non-dict result is converted"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text(json.dumps([
            {"workflow_id": "wf-001", "status": "registered"}
        ]))

        integration = CoherenceIntegration(str(state_path))
        result = integration.notify_completion(
            workflow_id="wf-001",
            result="simple string"  # type: ignore
        )

        assert result is True
        workflows = json.loads(workflows_path.read_text())
        assert workflows[0]["result"]["value"] == "simple string"

    def test_notify_completion_handles_invalid_json(self, tmp_path):
        """Test handling of invalid JSON in workflows file"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text("not valid json")

        integration = CoherenceIntegration(str(state_path))

        with pytest.raises(CoherenceStateError, match="Invalid JSON"):
            integration.notify_completion(workflow_id="wf-001", result={})

    def test_get_pending_workflows_returns_pending(self, tmp_path):
        """Test that get_pending_workflows returns non-completed workflows"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text(json.dumps([
            {"workflow_id": "wf-001", "status": "registered"},
            {"workflow_id": "wf-002", "status": "completed"},
            {"workflow_id": "wf-003", "status": "pending"},
        ]))

        integration = CoherenceIntegration(str(state_path))
        pending = integration.get_pending_workflows()

        assert len(pending) == 2
        assert all(w["status"] != "completed" for w in pending)

    def test_get_pending_workflows_returns_empty_for_missing_file(self, tmp_path):
        """Test that get_pending_workflows returns empty list when file missing"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        pending = integration.get_pending_workflows()
        assert pending == []

    def test_get_pending_workflows_handles_corrupted_file(self, tmp_path):
        """Test that get_pending_workflows handles corrupted file"""
        state_path = tmp_path / "coherence_state.json"
        workflows_path = tmp_path / "workflows.json"
        workflows_path.write_text("not valid json")

        integration = CoherenceIntegration(str(state_path))
        pending = integration.get_pending_workflows()

        assert pending == []

    def test_roundtrip_export_import(self, tmp_path):
        """Test exporting and importing state"""
        state_path = tmp_path / "coherence_state.json"
        integration = CoherenceIntegration(str(state_path))

        # Export
        exported = integration.export_state(
            project_id="roundtrip-test",
            status="operational",
            completed_work=["task1"],
            available_actions=["action1"]
        )

        # Import
        imported = integration.import_state()

        assert imported["project_id"] == exported["project_id"]
        assert imported["status"] == exported["status"]
        assert imported["completed_work"] == exported["completed_work"]

