"""
Tests for Context Manager

Tests the core/context_manager.py module including:
- get_default_storage_path function
- UniversalContextManager class
- get_ucts singleton
"""

import json
import os
import platform
import pytest
from pathlib import Path
from unittest.mock import patch, MagicMock
from datetime import datetime

from ucts.core.context_manager import (
    get_default_storage_path,
    UniversalContextManager,
    get_ucts,
)


# ============================================================================
# get_default_storage_path Tests
# ============================================================================

class TestGetDefaultStoragePath:
    """Tests for get_default_storage_path function"""

    def test_returns_path(self):
        """Test returns a Path object"""
        result = get_default_storage_path()
        assert isinstance(result, Path)

    @patch('platform.system', return_value='Windows')
    @patch.dict(os.environ, {'APPDATA': 'C:\\Users\\Test\\AppData\\Roaming'})
    def test_windows_appdata(self, mock_system):
        """Test Windows uses APPDATA"""
        result = get_default_storage_path()
        assert 'UCTS' in str(result)

    @patch('platform.system', return_value='Windows')
    @patch.dict(os.environ, {'LOCALAPPDATA': 'C:\\Users\\Test\\AppData\\Local'}, clear=True)
    def test_windows_localappdata_fallback(self, mock_system):
        """Test Windows falls back to LOCALAPPDATA"""
        with patch.dict(os.environ, {'APPDATA': ''}, clear=False):
            result = get_default_storage_path()
            assert 'UCTS' in str(result) or '.ucts' in str(result)

    @patch('platform.system', return_value='Darwin')
    def test_macos_path(self, mock_system):
        """Test macOS uses Application Support"""
        result = get_default_storage_path()
        assert 'UCTS' in str(result) or 'ucts' in str(result).lower()

    @patch('platform.system', return_value='Linux')
    def test_linux_path(self, mock_system):
        """Test Linux uses .ucts or XDG_DATA_HOME"""
        result = get_default_storage_path()
        assert 'ucts' in str(result).lower()

    @patch('platform.system', return_value='Linux')
    @patch.dict(os.environ, {'XDG_DATA_HOME': '/home/test/.local/share'})
    def test_linux_xdg(self, mock_system):
        """Test Linux uses XDG_DATA_HOME when set"""
        result = get_default_storage_path()
        assert 'ucts' in str(result).lower()


# ============================================================================
# UniversalContextManager Tests
# ============================================================================

class TestUniversalContextManager:
    """Tests for UniversalContextManager class"""

    def test_init_default(self, tmp_path):
        """Test initialization with defaults"""
        manager = UniversalContextManager("test", str(tmp_path))
        assert manager.project_name == "test"
        assert manager.current_messages == []

    def test_init_creates_directory(self, tmp_path):
        """Test initialization creates project directory"""
        manager = UniversalContextManager("test-project", str(tmp_path))
        assert manager.project_dir.exists()

    def test_sanitize_filename(self, tmp_path):
        """Test filename sanitization"""
        manager = UniversalContextManager("test", str(tmp_path))
        result = manager._sanitize_filename('file<>:"/\\|?*name')
        assert '<' not in result
        assert '>' not in result
        assert ':' not in result

    def test_sanitize_filename_length(self, tmp_path):
        """Test filename truncation"""
        manager = UniversalContextManager("test", str(tmp_path))
        long_name = "a" * 200
        result = manager._sanitize_filename(long_name)
        assert len(result) <= 100

    def test_load_config_creates_default(self, tmp_path):
        """Test config creation when missing"""
        manager = UniversalContextManager("test", str(tmp_path))
        config = manager._load_config()
        assert 'version' in config
        assert 'token_warning' in config
        assert 'token_critical' in config

    def test_load_config_existing(self, tmp_path):
        """Test loading existing config"""
        # Create config first
        config_file = tmp_path / "config.yaml"
        config_file.write_text("version: '1.0'\ncustom_key: 'value'\n")

        manager = UniversalContextManager("test", str(tmp_path))
        config = manager.config
        assert config.get('custom_key') == 'value'

    def test_add_message(self, tmp_path):
        """Test adding a message"""
        manager = UniversalContextManager("test", str(tmp_path))
        message = manager.add_message("user", "Hello world")

        assert message.role == "user"
        assert message.content == "Hello world"
        assert len(manager.current_messages) == 1

    def test_add_message_with_metadata(self, tmp_path):
        """Test adding message with metadata"""
        manager = UniversalContextManager("test", str(tmp_path))
        message = manager.add_message("assistant", "Response", {"tokens": 50})

        assert message.metadata == {"tokens": 50}

    def test_estimate_tokens_empty(self, tmp_path):
        """Test token estimation with no messages"""
        manager = UniversalContextManager("test", str(tmp_path))
        assert manager.estimate_tokens() == 0

    def test_estimate_tokens(self, tmp_path):
        """Test token estimation"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "A" * 100)  # 100 chars = ~25 tokens
        tokens = manager.estimate_tokens()
        assert tokens == 25

    def test_save_chunk(self, tmp_path):
        """Test saving a chunk"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Test message")

        chunk_file = manager.save_chunk()

        assert chunk_file is not None
        assert chunk_file.exists()
        assert manager.current_messages == []

    def test_save_chunk_empty(self, tmp_path):
        """Test saving empty chunk returns None"""
        manager = UniversalContextManager("test", str(tmp_path))
        result = manager.save_chunk()
        assert result is None

    def test_save_chunk_content(self, tmp_path):
        """Test chunk file content"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Test message")

        chunk_file = manager.save_chunk()
        with open(chunk_file) as f:
            data = json.load(f)

        assert 'messages' in data
        assert len(data['messages']) == 1
        assert data['message_count'] == 1

    def test_get_recent_chunks(self, tmp_path):
        """Test getting recent chunks"""
        import time
        manager = UniversalContextManager("test", str(tmp_path))

        # Create some chunks with slight delay to ensure different timestamps
        for i in range(3):
            manager.add_message("user", f"Message {i}")
            manager.save_chunk()
            time.sleep(0.01)  # Small delay to ensure unique timestamps

        chunks = manager.get_recent_chunks(limit=2)
        # Should get at least 1 chunk (implementation may vary)
        assert len(chunks) >= 1

    def test_get_recent_chunks_empty(self, tmp_path):
        """Test getting chunks when none exist"""
        manager = UniversalContextManager("test", str(tmp_path))
        chunks = manager.get_recent_chunks()
        assert chunks == []

    def test_analyze_conversation_empty(self, tmp_path):
        """Test analyzing empty conversation"""
        manager = UniversalContextManager("test", str(tmp_path))
        analysis = manager.analyze_conversation()

        assert analysis['total_messages'] == 0
        assert analysis['status'] == 'no_chunks'

    def test_analyze_conversation(self, tmp_path):
        """Test analyzing conversation with messages"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Write Python code ```python\nprint('hello')```")
        manager.add_message("assistant", "Here's the code for Python")
        manager.save_chunk()

        analysis = manager.analyze_conversation()

        assert analysis['total_messages'] == 2
        assert analysis['has_code'] is True
        assert 'python' in analysis['topics']

    def test_analyze_detects_errors(self, tmp_path):
        """Test analysis detects error-related content"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "I have a bug in my code")
        manager.add_message("assistant", "Let me fix that error")
        manager.save_chunk()

        analysis = manager.analyze_conversation()
        assert analysis['has_errors'] is True

    def test_analyze_detects_decisions(self, tmp_path):
        """Test analysis detects decisions"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "We decided to use PostgreSQL")
        manager.save_chunk()

        analysis = manager.analyze_conversation()
        assert analysis['has_decisions'] is True

    def test_create_transfer_package(self, tmp_path):
        """Test creating transfer package"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Test message")

        package = manager.create_transfer_package()

        assert 'transfer_id' in package
        assert 'project' in package
        assert 'markdown' in package

    def test_transfer_package_saves_files(self, tmp_path):
        """Test transfer package saves JSON and MD files"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Test")

        package = manager.create_transfer_package()

        json_files = list(manager.project_dir.glob("transfer_*.json"))
        md_files = list(manager.project_dir.glob("transfer_*.md"))

        assert len(json_files) >= 1
        assert len(md_files) >= 1

    def test_suggest_next_actions(self, tmp_path):
        """Test next action suggestions"""
        manager = UniversalContextManager("test", str(tmp_path))

        # With errors
        analysis = {'has_errors': True, 'has_decisions': False, 'has_code': False}
        actions = manager._suggest_next_actions(analysis)
        assert any('error' in a.lower() for a in actions)

        # With decisions
        analysis = {'has_errors': False, 'has_decisions': True, 'has_code': False}
        actions = manager._suggest_next_actions(analysis)
        assert any('decision' in a.lower() for a in actions)

    def test_get_status(self, tmp_path):
        """Test getting status"""
        manager = UniversalContextManager("test", str(tmp_path))
        manager.add_message("user", "Test")

        status = manager.get_status()

        assert status['project'] == "test"
        assert status['message_count'] == 1
        assert 'estimated_tokens' in status
        assert 'storage_path' in status


# ============================================================================
# Singleton Tests
# ============================================================================

class TestGetUcts:
    """Tests for get_ucts singleton"""

    def test_get_ucts_creates_instance(self):
        """Test getting manager creates instance"""
        import ucts.core.context_manager as module
        module._global_ucts = None

        manager = get_ucts("test-singleton")
        assert manager is not None
        assert isinstance(manager, UniversalContextManager)

    def test_get_ucts_returns_same_instance(self):
        """Test getting manager returns same instance"""
        manager1 = get_ucts()
        manager2 = get_ucts()
        assert manager1 is manager2
