"""
Tests for Docker Generator

Tests the generators/docker.py module including:
- DockerConfig dataclass
- DockerGenerator class
- add_docker_to_project function
"""

import pytest
from pathlib import Path
from dataclasses import dataclass, field
from typing import Dict, List

from ucts.generators.docker import (
    DockerConfig,
    DockerGenerator,
    add_docker_to_project,
)


# ============================================================================
# Mock ProjectStructure
# ============================================================================

@dataclass
class MockProjectStructure:
    """Mock project structure for testing"""
    name: str = "test-app"
    languages: List[str] = field(default_factory=lambda: ["python"])
    dependencies: Dict[str, List[str]] = field(default_factory=dict)
    files: Dict[str, str] = field(default_factory=dict)


# ============================================================================
# DockerConfig Tests
# ============================================================================

class TestDockerConfig:
    """Tests for DockerConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = DockerConfig()
        assert config.python_version == "3.11"
        assert config.node_version == "20"
        assert config.expose_port == 8000
        assert config.include_dev is True
        assert config.use_alpine is False
        assert config.multi_stage is True

    def test_custom_values(self):
        """Test custom configuration values"""
        config = DockerConfig(
            python_version="3.12",
            node_version="22",
            expose_port=3000,
            use_alpine=True,
            multi_stage=False
        )
        assert config.python_version == "3.12"
        assert config.node_version == "22"
        assert config.expose_port == 3000
        assert config.use_alpine is True
        assert config.multi_stage is False


# ============================================================================
# DockerGenerator Tests
# ============================================================================

class TestDockerGenerator:
    """Tests for DockerGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = DockerGenerator()
        assert gen.config is not None
        assert gen.config.python_version == "3.11"

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = DockerConfig(expose_port=5000)
        gen = DockerGenerator(config)
        assert gen.config.expose_port == 5000

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = DockerGenerator()
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "Dockerfile" in files
        assert "docker-compose.yml" in files
        assert ".dockerignore" in files

    def test_generate_writes_to_disk(self, tmp_path):
        """Test files are written to disk"""
        gen = DockerGenerator()
        structure = MockProjectStructure()

        gen.generate(structure, str(tmp_path))

        assert (tmp_path / "Dockerfile").exists()
        assert (tmp_path / "docker-compose.yml").exists()
        assert (tmp_path / ".dockerignore").exists()

    def test_generate_python_dockerfile(self, tmp_path):
        """Test generating Dockerfile for Python"""
        gen = DockerGenerator()
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "python" in files["Dockerfile"].lower()
        assert "pip" in files["Dockerfile"]

    def test_generate_node_dockerfile(self, tmp_path):
        """Test generating Dockerfile for Node.js"""
        gen = DockerGenerator()
        structure = MockProjectStructure(languages=["javascript"])

        files = gen.generate(structure, str(tmp_path))

        assert "node" in files["Dockerfile"].lower()
        assert "npm" in files["Dockerfile"]


# ============================================================================
# Python Dockerfile Tests
# ============================================================================

class TestPythonDockerfile:
    """Tests for Python Dockerfile generation"""

    def test_multi_stage_build(self, tmp_path):
        """Test multi-stage build"""
        config = DockerConfig(multi_stage=True)
        gen = DockerGenerator(config)
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "AS builder" in files["Dockerfile"]
        assert "FROM" in files["Dockerfile"]
        # Should have two FROM statements
        assert files["Dockerfile"].count("FROM") >= 2

    def test_single_stage_build(self, tmp_path):
        """Test single-stage build"""
        config = DockerConfig(multi_stage=False)
        gen = DockerGenerator(config)
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        # Should have only one FROM
        assert files["Dockerfile"].count("FROM") == 1

    def test_alpine_image(self, tmp_path):
        """Test Alpine-based image"""
        config = DockerConfig(use_alpine=True)
        gen = DockerGenerator(config)
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "alpine" in files["Dockerfile"]

    def test_slim_image(self, tmp_path):
        """Test slim-based image"""
        config = DockerConfig(use_alpine=False)
        gen = DockerGenerator(config)
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "slim" in files["Dockerfile"]

    def test_fastapi_command(self, tmp_path):
        """Test FastAPI CMD"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["fastapi", "uvicorn"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "uvicorn" in files["Dockerfile"]

    def test_flask_command(self, tmp_path):
        """Test Flask CMD"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["flask"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "flask" in files["Dockerfile"]

    def test_django_command(self, tmp_path):
        """Test Django CMD"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["django"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "manage.py" in files["Dockerfile"]

    def test_expose_port(self, tmp_path):
        """Test expose port is in Dockerfile"""
        config = DockerConfig(expose_port=5000)
        gen = DockerGenerator(config)
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "EXPOSE 5000" in files["Dockerfile"]


# ============================================================================
# Node.js Dockerfile Tests
# ============================================================================

class TestNodeDockerfile:
    """Tests for Node.js Dockerfile generation"""

    def test_next_command(self, tmp_path):
        """Test Next.js CMD"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            dependencies={"node": ["next"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "npm" in files["Dockerfile"]
        assert "build" in files["Dockerfile"] or "start" in files["Dockerfile"]

    def test_vite_command(self, tmp_path):
        """Test Vite CMD"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            dependencies={"node": ["vite"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "npm" in files["Dockerfile"]


# ============================================================================
# Docker Compose Tests
# ============================================================================

class TestDockerCompose:
    """Tests for docker-compose.yml generation"""

    def test_app_service(self, tmp_path):
        """Test app service is generated"""
        gen = DockerGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "app:" in files["docker-compose.yml"]
        assert "build:" in files["docker-compose.yml"]
        assert "ports:" in files["docker-compose.yml"]

    def test_postgres_service(self, tmp_path):
        """Test PostgreSQL service when dependency detected"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["psycopg2"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "postgres:" in files["docker-compose.yml"]
        assert "5432" in files["docker-compose.yml"]

    def test_redis_service(self, tmp_path):
        """Test Redis service when dependency detected"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["redis"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "redis:" in files["docker-compose.yml"]
        assert "6379" in files["docker-compose.yml"]

    def test_mongodb_service(self, tmp_path):
        """Test MongoDB service when dependency detected"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["pymongo"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "mongodb:" in files["docker-compose.yml"]
        assert "27017" in files["docker-compose.yml"]

    def test_mysql_service(self, tmp_path):
        """Test MySQL service when dependency detected"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["mysql-connector"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "mysql:" in files["docker-compose.yml"]
        assert "3306" in files["docker-compose.yml"]

    def test_volumes(self, tmp_path):
        """Test volumes are defined when services use them"""
        gen = DockerGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["psycopg2", "redis"]}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "volumes:" in files["docker-compose.yml"]
        assert "postgres_data" in files["docker-compose.yml"]
        assert "redis_data" in files["docker-compose.yml"]


# ============================================================================
# Dockerignore Tests
# ============================================================================

class TestDockerignore:
    """Tests for .dockerignore generation"""

    def test_common_patterns(self, tmp_path):
        """Test common ignore patterns"""
        gen = DockerGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert ".git" in files[".dockerignore"]
        assert ".vscode/" in files[".dockerignore"]
        assert "Dockerfile" in files[".dockerignore"]

    def test_python_patterns(self, tmp_path):
        """Test Python-specific patterns"""
        gen = DockerGenerator()
        structure = MockProjectStructure(languages=["python"])

        files = gen.generate(structure, str(tmp_path))

        assert "__pycache__/" in files[".dockerignore"]
        assert ".venv/" in files[".dockerignore"]
        # Pattern uses *.py[cod] to match .pyc, .pyo, .pyd
        assert "*.py[cod]" in files[".dockerignore"]

    def test_node_patterns(self, tmp_path):
        """Test Node.js-specific patterns"""
        gen = DockerGenerator()
        structure = MockProjectStructure(languages=["javascript"])

        files = gen.generate(structure, str(tmp_path))

        assert "node_modules/" in files[".dockerignore"]


# ============================================================================
# Convenience Function Tests
# ============================================================================

class TestAddDockerToProject:
    """Tests for add_docker_to_project function"""

    def test_basic_usage(self, tmp_path):
        """Test basic function usage"""
        structure = MockProjectStructure()
        files = add_docker_to_project(structure, str(tmp_path))

        assert "Dockerfile" in files
        assert (tmp_path / "Dockerfile").exists()

    def test_with_config(self, tmp_path):
        """Test function with custom config"""
        structure = MockProjectStructure()
        config = DockerConfig(expose_port=9000)

        files = add_docker_to_project(structure, str(tmp_path), config)

        assert "EXPOSE 9000" in files["Dockerfile"]
