"""Tests for project generators"""
import json
from pathlib import Path

import pytest

from ucts.generators import VSCodeGenerator
from ucts.core.models import ProjectStructure


class TestVSCodeGenerator:
    """Tests for VS Code workspace generator"""

    def create_test_structure(self):
        """Create a test ProjectStructure"""
        return ProjectStructure(
            name="test-project",
            description="A test project",
            languages=["python"],
            dependencies={"python": ["requests", "flask"]},
            files={
                "src/main.py": "print('Hello')",
                "tests/test_main.py": "def test(): pass",
            },
            directories=["src", "tests"],
            readme_content="# Test Project\n\nA test.",
            todos=["Add more tests"],
        )

    def test_generate_creates_directory(self, tmp_path):
        """Test that generator creates output directory"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        result = generator.generate(structure, str(output))

        assert result.exists()
        assert result.is_dir()

    def test_generate_creates_files(self, tmp_path):
        """Test that generator creates project files"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        assert (output / "src" / "main.py").exists()
        assert (output / "tests" / "test_main.py").exists()
        assert (output / "README.md").exists()

    def test_generate_creates_vscode_settings(self, tmp_path):
        """Test that generator creates .vscode directory"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        vscode_dir = output / ".vscode"
        assert vscode_dir.exists()
        assert (vscode_dir / "settings.json").exists()
        assert (vscode_dir / "extensions.json").exists()
        assert (vscode_dir / "tasks.json").exists()

    def test_generate_python_specific_files(self, tmp_path):
        """Test Python-specific file generation"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        assert (output / "requirements.txt").exists()
        assert (output / "pyproject.toml").exists()

        # Check requirements.txt content
        reqs = (output / "requirements.txt").read_text()
        assert "requests" in reqs
        assert "flask" in reqs

    def test_generate_gitignore(self, tmp_path):
        """Test .gitignore generation"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        gitignore = output / ".gitignore"
        assert gitignore.exists()

        content = gitignore.read_text()
        assert "__pycache__" in content
        assert ".venv" in content

    def test_generate_workspace_file(self, tmp_path):
        """Test workspace file generation"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        workspace_file = output / "test-project.code-workspace"
        assert workspace_file.exists()

        content = json.loads(workspace_file.read_text())
        assert "folders" in content
        assert "settings" in content

    def test_generate_typescript_project(self, tmp_path):
        """Test TypeScript project generation"""
        output = tmp_path / "output"
        structure = ProjectStructure(
            name="ts-project",
            description="A TypeScript project",
            languages=["typescript"],
            dependencies={"node": ["express"]},
            files={"src/index.ts": "console.log('Hello');"},
            directories=["src"],
            readme_content="# TS Project",
            todos=[],
        )

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        assert (output / "package.json").exists()
        assert (output / "tsconfig.json").exists()

        # Check package.json
        pkg = json.loads((output / "package.json").read_text())
        assert "typescript" in pkg.get("devDependencies", {})

    def test_settings_for_language(self, tmp_path):
        """Test language-specific settings"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        settings = json.loads((output / ".vscode" / "settings.json").read_text())

        # Python-specific settings
        assert "python.defaultInterpreterPath" in settings

    def test_recommended_extensions(self, tmp_path):
        """Test recommended extensions based on languages"""
        output = tmp_path / "output"
        structure = self.create_test_structure()

        generator = VSCodeGenerator()
        generator.generate(structure, str(output))

        extensions = json.loads((output / ".vscode" / "extensions.json").read_text())

        recommendations = extensions.get("recommendations", [])
        assert "ms-python.python" in recommendations
