"""Tests for GitHub repository generator"""
import json
from pathlib import Path
from unittest.mock import Mock, patch, MagicMock

import pytest

from ucts.generators.github import GitHubGenerator, GitHubAPIError, GitCommandError
from ucts.core.models import ProjectStructure


class TestGitHubGenerator:
    """Tests for GitHub repository generator"""

    def create_test_structure(self, languages=None):
        """Create a test ProjectStructure"""
        return ProjectStructure(
            name="test-project",
            description="A test project",
            languages=languages or ["python"],
            dependencies={"python": ["requests", "flask"]},
            files={
                "src/main.py": "print('Hello')",
                "tests/test_main.py": "def test(): pass",
            },
            directories=["src", "tests"],
            readme_content="# Test Project\n\nA test.",
            todos=["Add more tests"],
        )

    def test_init_requires_token(self):
        """Test that initialization requires a token"""
        with pytest.raises(ValueError, match="token cannot be empty"):
            GitHubGenerator("")

        with pytest.raises(ValueError, match="token cannot be empty"):
            GitHubGenerator(None)

    def test_init_sets_headers(self):
        """Test that initialization sets correct headers"""
        gen = GitHubGenerator("test-token")

        assert gen.headers["Authorization"] == "Bearer test-token"
        assert "application/vnd.github+json" in gen.headers["Accept"]

    def test_generate_requires_structure(self):
        """Test that generate requires a ProjectStructure"""
        gen = GitHubGenerator("test-token")

        with pytest.raises(ValueError, match="structure cannot be None"):
            gen.generate(None)

    @patch('ucts.generators.github.requests.post')
    @patch('ucts.generators.github.subprocess.run')
    def test_generate_creates_repo(self, mock_run, mock_post):
        """Test that generate creates a GitHub repository"""
        # Mock successful API response
        mock_post.return_value = Mock(
            ok=True,
            json=lambda: {
                'html_url': 'https://github.com/user/test-project',
                'clone_url': 'https://github.com/user/test-project.git'
            }
        )

        # Mock successful git commands
        mock_run.return_value = Mock(returncode=0, stdout='', stderr='')

        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure()

        result = gen.generate(structure)

        assert result == 'https://github.com/user/test-project'
        assert mock_post.called

    @patch('ucts.generators.github.requests.post')
    def test_generate_handles_api_error(self, mock_post):
        """Test that generate handles API errors"""
        mock_post.return_value = Mock(
            ok=False,
            text='{"message": "Bad credentials"}'
        )

        gen = GitHubGenerator("bad-token")
        structure = self.create_test_structure()

        with pytest.raises(GitHubAPIError, match="Failed to create GitHub repo"):
            gen.generate(structure)

    @patch('ucts.generators.github.requests.post')
    def test_generate_handles_connection_error(self, mock_post):
        """Test that generate handles connection errors"""
        import requests
        mock_post.side_effect = requests.RequestException("Connection failed")

        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure()

        with pytest.raises(GitHubAPIError, match="Failed to connect"):
            gen.generate(structure)

    def test_add_github_actions_python(self, tmp_path):
        """Test GitHub Actions workflow for Python projects"""
        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_github_actions(tmp_path, structure)

        workflow_file = tmp_path / ".github" / "workflows" / "ci.yml"
        assert workflow_file.exists()

        content = workflow_file.read_text()

        # Check for Python-specific features
        assert "lint-python" in content
        assert "test-python" in content
        assert "build-python" in content
        assert "python-version:" in content
        assert "black" in content
        assert "pytest" in content

    def test_add_github_actions_node(self, tmp_path):
        """Test GitHub Actions workflow for Node.js projects"""
        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure(languages=["typescript"])

        gen._add_github_actions(tmp_path, structure)

        workflow_file = tmp_path / ".github" / "workflows" / "ci.yml"
        content = workflow_file.read_text()

        # Check for Node-specific features
        assert "lint-node" in content
        assert "test-node" in content
        assert "build-node" in content
        assert "node-version:" in content

    def test_add_github_actions_release_workflow(self, tmp_path):
        """Test GitHub Actions release workflow"""
        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_github_actions(tmp_path, structure)

        release_file = tmp_path / ".github" / "workflows" / "release.yml"
        assert release_file.exists()

        content = release_file.read_text()

        # Check for release features
        assert "Release" in content
        assert "publish-pypi" in content
        assert "create-release" in content
        assert "tags:" in content

    def test_add_github_actions_matrix_testing(self, tmp_path):
        """Test that GitHub Actions includes matrix testing"""
        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_github_actions(tmp_path, structure)

        workflow_file = tmp_path / ".github" / "workflows" / "ci.yml"
        content = workflow_file.read_text()

        # Check for matrix strategy
        assert "matrix:" in content
        assert "3.9" in content
        assert "3.10" in content
        assert "3.11" in content
        assert "3.12" in content

    def test_add_github_actions_caching(self, tmp_path):
        """Test that GitHub Actions includes caching"""
        gen = GitHubGenerator("test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_github_actions(tmp_path, structure)

        workflow_file = tmp_path / ".github" / "workflows" / "ci.yml"
        content = workflow_file.read_text()

        # Check for caching
        assert "cache:" in content

    def test_run_git_timeout(self, tmp_path):
        """Test that git commands have timeout handling"""
        gen = GitHubGenerator("test-token")

        with patch('ucts.generators.github.subprocess.run') as mock_run:
            import subprocess
            mock_run.side_effect = subprocess.TimeoutExpired('git', 120)

            with pytest.raises(GitCommandError, match="timed out"):
                gen._run_git(['status'], tmp_path)

    def test_run_git_not_found(self, tmp_path):
        """Test handling when git is not found"""
        gen = GitHubGenerator("test-token")

        with patch('ucts.generators.github.subprocess.run') as mock_run:
            mock_run.side_effect = FileNotFoundError()

            with pytest.raises(GitCommandError, match="not found in PATH"):
                gen._run_git(['status'], tmp_path)

    def test_run_git_failure(self, tmp_path):
        """Test handling of git command failures"""
        gen = GitHubGenerator("test-token")

        with patch('ucts.generators.github.subprocess.run') as mock_run:
            mock_run.return_value = Mock(returncode=1, stderr='fatal: error')

            with pytest.raises(GitCommandError, match="Git command failed"):
                gen._run_git(['status'], tmp_path)
