"""Tests for GitLab repository generator"""
import json
from pathlib import Path
from unittest.mock import Mock, patch, MagicMock

import pytest

from ucts.generators.gitlab import GitLabGenerator, GitLabAPIError, GitCommandError
from ucts.core.models import ProjectStructure


class TestGitLabGenerator:
    """Tests for GitLab repository generator"""

    def create_test_structure(self, languages=None):
        """Create a test ProjectStructure"""
        return ProjectStructure(
            name="test-project",
            description="A test project",
            languages=languages or ["python"],
            dependencies={"python": ["requests", "flask"]},
            files={
                "src/main.py": "print('Hello')",
                "tests/test_main.py": "def test(): pass",
            },
            directories=["src", "tests"],
            readme_content="# Test Project\n\nA test.",
            todos=["Add more tests"],
        )

    def test_init_requires_url(self):
        """Test that initialization requires a GitLab URL"""
        with pytest.raises(ValueError, match="gitlab_url cannot be empty"):
            GitLabGenerator("", "token")

    def test_init_requires_token(self):
        """Test that initialization requires a token"""
        with pytest.raises(ValueError, match="token cannot be empty"):
            GitLabGenerator("https://gitlab.com", "")

    def test_init_sets_api_url(self):
        """Test that initialization sets correct API URL"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        assert gen.api_url == "https://gitlab.com/api/v4"
        assert gen.headers["PRIVATE-TOKEN"] == "test-token"

    def test_init_strips_trailing_slash(self):
        """Test that trailing slashes are stripped from URL"""
        gen = GitLabGenerator("https://gitlab.com/", "test-token")

        assert gen.gitlab_url == "https://gitlab.com"

    def test_generate_requires_structure(self):
        """Test that generate requires a ProjectStructure"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        with pytest.raises(ValueError, match="structure cannot be None"):
            gen.generate(None)

    def test_generate_validates_visibility(self):
        """Test that generate validates visibility parameter"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure()

        with pytest.raises(ValueError, match="Invalid visibility"):
            gen.generate(structure, visibility="invalid")

    @patch('ucts.generators.gitlab.requests.post')
    @patch('ucts.generators.gitlab.subprocess.run')
    def test_generate_creates_repo(self, mock_run, mock_post):
        """Test that generate creates a GitLab repository"""
        # Mock successful API response
        mock_post.return_value = Mock(
            ok=True,
            json=lambda: {
                'web_url': 'https://gitlab.com/user/test-project',
                'ssh_url_to_repo': 'git@gitlab.com:user/test-project.git',
                'http_url_to_repo': 'https://gitlab.com/user/test-project.git'
            }
        )

        # Mock successful git commands
        mock_run.return_value = Mock(returncode=0, stdout='', stderr='')

        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure()

        with patch.object(gen, '_ssh_available', return_value=False):
            result = gen.generate(structure)

        assert result == 'https://gitlab.com/user/test-project'
        assert mock_post.called

    @patch('ucts.generators.gitlab.requests.post')
    def test_generate_handles_api_error(self, mock_post):
        """Test that generate handles API errors"""
        mock_post.return_value = Mock(
            ok=False,
            text='{"message": "401 Unauthorized"}'
        )

        gen = GitLabGenerator("https://gitlab.com", "bad-token")
        structure = self.create_test_structure()

        with pytest.raises(GitLabAPIError, match="Failed to create GitLab project"):
            gen.generate(structure)

    @patch('ucts.generators.gitlab.requests.post')
    def test_generate_handles_connection_error(self, mock_post):
        """Test that generate handles connection errors"""
        import requests
        mock_post.side_effect = requests.RequestException("Connection failed")

        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure()

        with pytest.raises(GitLabAPIError, match="Failed to connect"):
            gen.generate(structure)

    def test_add_gitlab_ci_python(self, tmp_path):
        """Test GitLab CI configuration for Python projects"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        assert ci_file.exists()

        content = ci_file.read_text()

        # Check for Python-specific features
        assert "lint-python" in content
        assert "test-python" in content
        assert "build-python" in content
        assert "publish-pypi" in content
        assert "black" in content
        assert "pytest" in content

    def test_add_gitlab_ci_node(self, tmp_path):
        """Test GitLab CI configuration for Node.js projects"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["javascript"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for Node-specific features
        assert "lint-node" in content
        assert "test-node" in content
        assert "build-node" in content
        assert "publish-npm" in content

    def test_add_gitlab_ci_stages(self, tmp_path):
        """Test that GitLab CI includes all stages"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for all stages
        assert "stages:" in content
        assert "- lint" in content
        assert "- test" in content
        assert "- build" in content
        assert "- publish" in content

    def test_add_gitlab_ci_matrix_testing(self, tmp_path):
        """Test that GitLab CI includes matrix testing"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for matrix strategy
        assert "parallel:" in content
        assert "matrix:" in content
        assert "'3.9'" in content
        assert "'3.12'" in content

    def test_add_gitlab_ci_caching(self, tmp_path):
        """Test that GitLab CI includes caching"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for caching
        assert "cache:" in content
        assert ".pip-cache/" in content

    def test_add_gitlab_ci_coverage_report(self, tmp_path):
        """Test that GitLab CI includes coverage reporting"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        structure = self.create_test_structure(languages=["python"])

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for coverage
        assert "coverage:" in content
        assert "cobertura" in content
        assert "artifacts:" in content

    def test_add_gitlab_ci_generic(self, tmp_path):
        """Test GitLab CI for projects with no detected language"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")
        # Create structure with empty languages list for generic config
        structure = ProjectStructure(
            name="generic-project",
            description="A generic project",
            languages=[],  # No languages detected
            dependencies={},
            files={},
            directories=[],
            readme_content="# Generic",
            todos=[],
        )

        gen._add_gitlab_ci(tmp_path, structure)

        ci_file = tmp_path / ".gitlab-ci.yml"
        content = ci_file.read_text()

        # Check for generic jobs (when no Python or Node detected)
        assert "Generic Jobs" in content

    def test_run_git_timeout(self, tmp_path):
        """Test that git commands have timeout handling"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        with patch('ucts.generators.gitlab.subprocess.run') as mock_run:
            import subprocess
            mock_run.side_effect = subprocess.TimeoutExpired('git', 120)

            with pytest.raises(GitCommandError, match="timed out"):
                gen._run_git(['status'], tmp_path)

    def test_run_git_not_found(self, tmp_path):
        """Test handling when git is not found"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        with patch('ucts.generators.gitlab.subprocess.run') as mock_run:
            mock_run.side_effect = FileNotFoundError()

            with pytest.raises(GitCommandError, match="not found in PATH"):
                gen._run_git(['status'], tmp_path)

    def test_ssh_available_timeout(self):
        """Test SSH availability check handles timeout"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        with patch('ucts.generators.gitlab.subprocess.run') as mock_run:
            import subprocess
            mock_run.side_effect = subprocess.TimeoutExpired('ssh', 10)

            assert gen._ssh_available() is False

    def test_ssh_available_not_found(self):
        """Test SSH availability check handles missing ssh"""
        gen = GitLabGenerator("https://gitlab.com", "test-token")

        with patch('ucts.generators.gitlab.subprocess.run') as mock_run:
            mock_run.side_effect = FileNotFoundError()

            assert gen._ssh_available() is False
