"""
Tests for Forge Pipeline Hooks

Tests the core/hooks.py module including:
- HookPhase and HookResult enums
- HookContext and HookResponse dataclasses
- ForgeHooks class
- Singleton functions
"""

import time
import pytest
from unittest.mock import MagicMock, patch
from dataclasses import dataclass

from ucts.core.hooks import (
    HookPhase,
    HookResult,
    HookContext,
    HookResponse,
    ForgeHooks,
    get_forge_hooks,
    reset_hooks,
)


# ============================================================================
# HookPhase Enum Tests
# ============================================================================

class TestHookPhase:
    """Tests for HookPhase enum"""

    def test_pre_ingest_value(self):
        """Test pre_ingest phase value"""
        assert HookPhase.PRE_INGEST.value == "pre_ingest"

    def test_post_ingest_value(self):
        """Test post_ingest phase value"""
        assert HookPhase.POST_INGEST.value == "post_ingest"

    def test_pre_analyze_value(self):
        """Test pre_analyze phase value"""
        assert HookPhase.PRE_ANALYZE.value == "pre_analyze"

    def test_post_analyze_value(self):
        """Test post_analyze phase value"""
        assert HookPhase.POST_ANALYZE.value == "post_analyze"

    def test_pre_generate_value(self):
        """Test pre_generate phase value"""
        assert HookPhase.PRE_GENERATE.value == "pre_generate"

    def test_post_generate_value(self):
        """Test post_generate phase value"""
        assert HookPhase.POST_GENERATE.value == "post_generate"

    def test_pre_merge_value(self):
        """Test pre_merge phase value"""
        assert HookPhase.PRE_MERGE.value == "pre_merge"

    def test_post_merge_value(self):
        """Test post_merge phase value"""
        assert HookPhase.POST_MERGE.value == "post_merge"


# ============================================================================
# HookResult Enum Tests
# ============================================================================

class TestHookResult:
    """Tests for HookResult enum"""

    def test_continue_value(self):
        """Test continue result value"""
        assert HookResult.CONTINUE.value == "continue"

    def test_skip_value(self):
        """Test skip result value"""
        assert HookResult.SKIP.value == "skip"

    def test_abort_value(self):
        """Test abort result value"""
        assert HookResult.ABORT.value == "abort"

    def test_warn_value(self):
        """Test warn result value"""
        assert HookResult.WARN.value == "warn"


# ============================================================================
# HookContext Dataclass Tests
# ============================================================================

class TestHookContext:
    """Tests for HookContext dataclass"""

    def test_creation(self):
        """Test basic context creation"""
        context = HookContext(phase=HookPhase.PRE_INGEST)
        assert context.phase == HookPhase.PRE_INGEST
        assert context.source_paths == []
        assert context.output_path is None
        assert context.target_type == "vscode"

    def test_with_all_fields(self):
        """Test context with all fields"""
        context = HookContext(
            phase=HookPhase.POST_GENERATE,
            source_paths=["file1.json", "file2.json"],
            output_path="/output",
            target_type="gitlab",
            metadata={"key": "value"}
        )
        assert context.source_paths == ["file1.json", "file2.json"]
        assert context.output_path == "/output"
        assert context.target_type == "gitlab"
        assert context.metadata["key"] == "value"

    def test_default_lists(self):
        """Test default list initialization"""
        context = HookContext(phase=HookPhase.PRE_INGEST)
        assert context.warnings == []
        assert context.errors == []

    def test_start_time_default(self):
        """Test start_time has default"""
        before = time.time()
        context = HookContext(phase=HookPhase.PRE_INGEST)
        after = time.time()
        assert before <= context.start_time <= after


# ============================================================================
# HookResponse Dataclass Tests
# ============================================================================

class TestHookResponse:
    """Tests for HookResponse dataclass"""

    def test_creation(self):
        """Test basic response creation"""
        response = HookResponse(result=HookResult.CONTINUE)
        assert response.result == HookResult.CONTINUE
        assert response.message is None
        assert response.data == {}
        assert response.duration_ms == 0

    def test_with_all_fields(self):
        """Test response with all fields"""
        response = HookResponse(
            result=HookResult.WARN,
            message="Warning message",
            data={"key": "value"},
            duration_ms=150
        )
        assert response.result == HookResult.WARN
        assert response.message == "Warning message"
        assert response.data["key"] == "value"
        assert response.duration_ms == 150


# ============================================================================
# ForgeHooks Tests
# ============================================================================

class TestForgeHooks:
    """Tests for ForgeHooks class"""

    def test_init_default(self):
        """Test initialization with defaults"""
        hooks = ForgeHooks(ecosystem_enabled=False)
        assert hooks.ecosystem_enabled is False
        assert all(phase in hooks._hooks for phase in HookPhase)

    def test_init_ecosystem_disabled(self):
        """Test initialization with ecosystem disabled"""
        hooks = ForgeHooks(ecosystem_enabled=False)
        # Should not register ecosystem hooks
        assert len(hooks._hooks[HookPhase.PRE_ANALYZE]) == 0

    def test_register_hook(self):
        """Test registering a hook"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def my_hook(context):
            return HookResponse(result=HookResult.CONTINUE)

        hooks.register(HookPhase.PRE_INGEST, my_hook)
        assert my_hook in hooks._hooks[HookPhase.PRE_INGEST]

    def test_unregister_hook(self):
        """Test unregistering a hook"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def my_hook(context):
            return HookResponse(result=HookResult.CONTINUE)

        hooks.register(HookPhase.PRE_INGEST, my_hook)
        hooks.unregister(HookPhase.PRE_INGEST, my_hook)
        assert my_hook not in hooks._hooks[HookPhase.PRE_INGEST]

    def test_execute_no_hooks(self):
        """Test executing with no hooks"""
        hooks = ForgeHooks(ecosystem_enabled=False)
        context = HookContext(phase=HookPhase.PRE_INGEST)

        response = hooks.execute(HookPhase.PRE_INGEST, context)
        assert response.result == HookResult.CONTINUE

    def test_execute_single_hook(self):
        """Test executing single hook"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def my_hook(context):
            return HookResponse(result=HookResult.CONTINUE, data={"processed": True})

        hooks.register(HookPhase.PRE_INGEST, my_hook)
        context = HookContext(phase=HookPhase.PRE_INGEST)

        # Enable ecosystem to actually execute hooks
        hooks.ecosystem_enabled = True
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        assert response.data.get("processed") is True

    def test_execute_multiple_hooks(self):
        """Test executing multiple hooks"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def hook1(context):
            return HookResponse(result=HookResult.CONTINUE, data={"hook1": True})

        def hook2(context):
            return HookResponse(result=HookResult.CONTINUE, data={"hook2": True})

        hooks.register(HookPhase.PRE_INGEST, hook1)
        hooks.register(HookPhase.PRE_INGEST, hook2)

        hooks.ecosystem_enabled = True
        context = HookContext(phase=HookPhase.PRE_INGEST)
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        assert response.data.get("hook1") is True
        assert response.data.get("hook2") is True

    def test_execute_abort_stops_pipeline(self):
        """Test abort result stops execution"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def abort_hook(context):
            return HookResponse(result=HookResult.ABORT, message="Stopped")

        def never_called(context):
            return HookResponse(result=HookResult.CONTINUE, data={"called": True})

        hooks.register(HookPhase.PRE_INGEST, abort_hook)
        hooks.register(HookPhase.PRE_INGEST, never_called)

        hooks.ecosystem_enabled = True
        context = HookContext(phase=HookPhase.PRE_INGEST)
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        assert response.result == HookResult.ABORT
        assert response.message == "Stopped"
        assert "called" not in response.data

    def test_execute_warn_continues(self):
        """Test warn result continues execution"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def warn_hook(context):
            return HookResponse(result=HookResult.WARN, message="Warning")

        def next_hook(context):
            return HookResponse(result=HookResult.CONTINUE, data={"next": True})

        hooks.register(HookPhase.PRE_INGEST, warn_hook)
        hooks.register(HookPhase.PRE_INGEST, next_hook)

        hooks.ecosystem_enabled = True
        context = HookContext(phase=HookPhase.PRE_INGEST)
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        assert response.result == HookResult.WARN
        assert response.data.get("next") is True
        assert "Warning" in context.warnings

    def test_execute_handles_exception(self):
        """Test hook exceptions are handled"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def bad_hook(context):
            raise Exception("Hook error")

        hooks.register(HookPhase.PRE_INGEST, bad_hook)

        hooks.ecosystem_enabled = True
        context = HookContext(phase=HookPhase.PRE_INGEST)
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        # Should not raise, should continue
        assert response.result == HookResult.CONTINUE
        assert len(context.errors) >= 1

    def test_execute_measures_duration(self):
        """Test execution measures duration"""
        hooks = ForgeHooks(ecosystem_enabled=False)

        def slow_hook(context):
            time.sleep(0.01)
            return HookResponse(result=HookResult.CONTINUE)

        hooks.register(HookPhase.PRE_INGEST, slow_hook)

        hooks.ecosystem_enabled = True
        context = HookContext(phase=HookPhase.PRE_INGEST)
        response = hooks.execute(HookPhase.PRE_INGEST, context)

        assert response.duration_ms >= 10


# ============================================================================
# Ecosystem Hook Tests
# ============================================================================

class TestEcosystemHooks:
    """Tests for ecosystem integration hooks"""

    def test_search_patterns_no_ecosystem(self):
        """Test search patterns when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.PRE_ANALYZE)
        response = hooks._hook_search_patterns(context)

        assert response.result == HookResult.CONTINUE

    def test_validate_policy_no_ecosystem(self):
        """Test validate policy when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.POST_ANALYZE)
        response = hooks._hook_validate_policy(context)

        assert response.result == HookResult.CONTINUE

    def test_trust_check_no_ecosystem(self):
        """Test trust check when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.PRE_GENERATE)
        response = hooks._hook_trust_check(context)

        assert response.result == HookResult.CONTINUE

    def test_security_scan_no_ecosystem(self):
        """Test security scan when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.POST_GENERATE)
        response = hooks._hook_security_scan(context)

        assert response.result == HookResult.CONTINUE

    def test_store_session_no_ecosystem(self):
        """Test store session when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.POST_GENERATE)
        response = hooks._hook_store_session(context)

        assert response.result == HookResult.CONTINUE

    def test_record_metrics_no_ecosystem(self):
        """Test record metrics when ecosystem not available"""
        hooks = ForgeHooks(ecosystem_enabled=True)
        hooks._ecosystem = None
        hooks.ecosystem_enabled = False

        context = HookContext(phase=HookPhase.POST_GENERATE)
        response = hooks._hook_record_metrics(context)

        assert response.result == HookResult.CONTINUE


# ============================================================================
# Singleton Tests
# ============================================================================

class TestSingleton:
    """Tests for singleton functions"""

    def test_get_forge_hooks(self):
        """Test getting forge hooks singleton"""
        reset_hooks()
        hooks1 = get_forge_hooks(ecosystem_enabled=False)
        hooks2 = get_forge_hooks(ecosystem_enabled=False)
        assert hooks1 is hooks2

    def test_reset_hooks(self):
        """Test resetting hooks singleton"""
        hooks1 = get_forge_hooks(ecosystem_enabled=False)
        reset_hooks()
        hooks2 = get_forge_hooks(ecosystem_enabled=False)
        assert hooks1 is not hooks2
