"""Tests for session ingestion module"""
import json
import tempfile
from pathlib import Path

import pytest

from ucts.ingestion import get_ingester, ClaudeDesktopIngester, TranscriptIngester
from ucts.core.models import Session, Message


class TestClaudeDesktopIngester:
    """Tests for Claude Desktop ingester"""

    def test_ingest_simple_conversation(self, tmp_path):
        """Test ingesting a simple conversation"""
        conversation = [
            {"role": "user", "content": "Hello, can you help me write Python?"},
            {"role": "assistant", "content": "Of course! Here's a simple example:\n```python\nprint('Hello')\n```"},
        ]

        conv_file = tmp_path / "conversation.json"
        conv_file.write_text(json.dumps(conversation))

        ingester = ClaudeDesktopIngester()
        session = ingester.ingest(str(conv_file))

        assert isinstance(session, Session)
        assert len(session.messages) == 2
        assert session.messages[0].role == "user"
        assert session.messages[1].role == "assistant"
        assert len(session.code_blocks) == 1
        assert session.code_blocks[0].language == "python"

    def test_ingest_with_metadata(self, tmp_path):
        """Test ingesting conversation with metadata"""
        data = {
            "conversation": [
                {"role": "user", "content": "Test message"},
            ],
            "metadata": {"model": "claude-3"}
        }

        conv_file = tmp_path / "conversation.json"
        conv_file.write_text(json.dumps(data))

        ingester = ClaudeDesktopIngester()
        session = ingester.ingest(str(conv_file))

        assert len(session.messages) == 1


class TestTranscriptIngester:
    """Tests for transcript ingester"""

    def test_ingest_markdown_transcript(self, tmp_path):
        """Test ingesting markdown transcript"""
        transcript = """**User**: Can you help me?

**Assistant**: Yes, I can help you with that.

**User**: Thanks!
"""
        trans_file = tmp_path / "transcript.md"
        trans_file.write_text(transcript)

        ingester = TranscriptIngester()
        session = ingester.ingest(str(trans_file))

        assert len(session.messages) == 3
        assert session.messages[0].role == "user"
        assert session.messages[1].role == "assistant"
        assert session.messages[2].role == "user"

    def test_ingest_unstructured_text(self, tmp_path):
        """Test ingesting unstructured text"""
        text = "This is just some text without clear role markers."

        text_file = tmp_path / "text.txt"
        text_file.write_text(text)

        ingester = TranscriptIngester()
        session = ingester.ingest(str(text_file))

        # Should create a single assistant message
        assert len(session.messages) == 1


class TestGetIngester:
    """Tests for ingester factory function"""

    def test_get_auto_ingester_json(self, tmp_path):
        """Test auto-detection for JSON files"""
        data = {"conversation": [{"role": "user", "content": "test"}]}
        json_file = tmp_path / "test.json"
        json_file.write_text(json.dumps(data))

        ingester = get_ingester("auto", str(json_file))
        assert ingester is not None

    def test_get_explicit_ingester(self):
        """Test explicit format selection"""
        ingester = get_ingester("transcript")
        assert isinstance(ingester, TranscriptIngester)

        ingester = get_ingester("claude")
        assert isinstance(ingester, ClaudeDesktopIngester)


class TestCodeBlockExtraction:
    """Tests for code block extraction"""

    def test_extract_multiple_languages(self, tmp_path):
        """Test extracting code blocks in multiple languages"""
        conversation = [
            {
                "role": "assistant",
                "content": """Here's Python:
```python
def hello():
    print("Hi")
```

And JavaScript:
```javascript
function hello() {
    console.log("Hi");
}
```
"""
            }
        ]

        conv_file = tmp_path / "conversation.json"
        conv_file.write_text(json.dumps(conversation))

        ingester = ClaudeDesktopIngester()
        session = ingester.ingest(str(conv_file))

        assert len(session.code_blocks) == 2
        languages = {cb.language for cb in session.code_blocks}
        assert "python" in languages
        assert "javascript" in languages
