"""
Tests for Kubernetes Generator

Tests the generators/kubernetes.py module including:
- K8sConfig dataclass
- KubernetesGenerator class
- HelmGenerator class
"""

import yaml
import pytest
from pathlib import Path
from dataclasses import dataclass, field
from typing import Dict, List

from ucts.generators.kubernetes import (
    K8sConfig,
    KubernetesGenerator,
)


# ============================================================================
# Mock ProjectStructure
# ============================================================================

@dataclass
class MockProjectStructure:
    """Mock project structure for testing"""
    name: str = "test-app"
    languages: List[str] = field(default_factory=lambda: ["python"])
    dependencies: Dict[str, List[str]] = field(default_factory=dict)
    files: Dict[str, str] = field(default_factory=dict)


# ============================================================================
# K8sConfig Tests
# ============================================================================

class TestK8sConfig:
    """Tests for K8sConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = K8sConfig()
        assert config.name == "app"
        assert config.namespace == "default"
        assert config.port == 8000
        assert config.replicas == 2
        assert config.enable_ingress is True
        assert config.enable_hpa is True

    def test_resource_defaults(self):
        """Test resource defaults"""
        config = K8sConfig()
        assert config.cpu_request == "100m"
        assert config.cpu_limit == "500m"
        assert config.memory_request == "128Mi"
        assert config.memory_limit == "512Mi"

    def test_custom_values(self):
        """Test custom configuration values"""
        config = K8sConfig(
            name="my-app",
            namespace="production",
            port=3000,
            replicas=5,
            cpu_limit="1000m",
            memory_limit="1Gi"
        )
        assert config.name == "my-app"
        assert config.namespace == "production"
        assert config.port == 3000
        assert config.replicas == 5

    def test_autoscaling_config(self):
        """Test autoscaling configuration"""
        config = K8sConfig(
            min_replicas=3,
            max_replicas=20,
            target_cpu=70
        )
        assert config.min_replicas == 3
        assert config.max_replicas == 20
        assert config.target_cpu == 70

    def test_env_vars(self):
        """Test environment variables"""
        config = K8sConfig(
            env_vars={"DEBUG": "true", "LOG_LEVEL": "debug"}
        )
        assert config.env_vars["DEBUG"] == "true"
        assert config.env_vars["LOG_LEVEL"] == "debug"

    def test_secrets(self):
        """Test secrets"""
        config = K8sConfig(
            secrets={"API_KEY": "secret-value"}
        )
        assert config.secrets["API_KEY"] == "secret-value"


# ============================================================================
# KubernetesGenerator Tests
# ============================================================================

class TestKubernetesGenerator:
    """Tests for KubernetesGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = KubernetesGenerator()
        assert gen.config is not None
        assert gen.config.name == "app"

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = K8sConfig(name="custom-app")
        gen = KubernetesGenerator(config)
        assert gen.config.name == "custom-app"

    def test_generate_creates_files(self, tmp_path):
        """Test generate creates expected files"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "namespace.yaml" in files
        assert "configmap.yaml" in files
        assert "secret.yaml" in files
        assert "deployment.yaml" in files
        assert "service.yaml" in files
        assert "kustomization.yaml" in files

    def test_generate_writes_to_disk(self, tmp_path):
        """Test files are written to disk"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        gen.generate(structure, str(tmp_path))

        assert (tmp_path / "deployment.yaml").exists()
        assert (tmp_path / "service.yaml").exists()

    def test_generate_ingress_when_enabled(self, tmp_path):
        """Test ingress is generated when enabled"""
        config = K8sConfig(enable_ingress=True)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "ingress.yaml" in files

    def test_no_ingress_when_disabled(self, tmp_path):
        """Test ingress is not generated when disabled"""
        config = K8sConfig(enable_ingress=False)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "ingress.yaml" not in files

    def test_generate_hpa_when_enabled(self, tmp_path):
        """Test HPA is generated when enabled"""
        config = K8sConfig(enable_hpa=True)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "hpa.yaml" in files

    def test_no_hpa_when_disabled(self, tmp_path):
        """Test HPA is not generated when disabled"""
        config = K8sConfig(enable_hpa=False)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        assert "hpa.yaml" not in files


# ============================================================================
# Namespace Tests
# ============================================================================

class TestNamespaceGeneration:
    """Tests for namespace.yaml generation"""

    def test_namespace_valid_yaml(self, tmp_path):
        """Test namespace is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        ns = yaml.safe_load(files["namespace.yaml"])

        assert ns["kind"] == "Namespace"
        assert ns["apiVersion"] == "v1"

    def test_namespace_name(self, tmp_path):
        """Test namespace uses config name"""
        config = K8sConfig(namespace="production")
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        ns = yaml.safe_load(files["namespace.yaml"])

        assert ns["metadata"]["name"] == "production"


# ============================================================================
# ConfigMap Tests
# ============================================================================

class TestConfigMapGeneration:
    """Tests for configmap.yaml generation"""

    def test_configmap_valid_yaml(self, tmp_path):
        """Test configmap is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        cm = yaml.safe_load(files["configmap.yaml"])

        assert cm["kind"] == "ConfigMap"
        assert "data" in cm

    def test_configmap_includes_env_vars(self, tmp_path):
        """Test configmap includes custom env vars"""
        config = K8sConfig(env_vars={"MY_VAR": "my_value"})
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        cm = yaml.safe_load(files["configmap.yaml"])

        assert cm["data"]["MY_VAR"] == "my_value"


# ============================================================================
# Secret Tests
# ============================================================================

class TestSecretGeneration:
    """Tests for secret.yaml generation"""

    def test_secret_valid_yaml(self, tmp_path):
        """Test secret is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        secret = yaml.safe_load(files["secret.yaml"])

        assert secret["kind"] == "Secret"
        assert secret["type"] == "Opaque"

    def test_secret_includes_secrets(self, tmp_path):
        """Test secret includes custom secrets"""
        config = K8sConfig(secrets={"API_KEY": "secret123"})
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        secret = yaml.safe_load(files["secret.yaml"])

        assert secret["stringData"]["API_KEY"] == "secret123"


# ============================================================================
# Deployment Tests
# ============================================================================

class TestDeploymentGeneration:
    """Tests for deployment.yaml generation"""

    def test_deployment_valid_yaml(self, tmp_path):
        """Test deployment is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        deploy = yaml.safe_load(files["deployment.yaml"])

        assert deploy["kind"] == "Deployment"
        assert deploy["apiVersion"] == "apps/v1"

    def test_deployment_replicas(self, tmp_path):
        """Test deployment uses config replicas"""
        config = K8sConfig(replicas=5)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        deploy = yaml.safe_load(files["deployment.yaml"])

        assert deploy["spec"]["replicas"] == 5

    def test_deployment_resources(self, tmp_path):
        """Test deployment includes resource limits"""
        config = K8sConfig(cpu_limit="1", memory_limit="1Gi")
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        deploy = yaml.safe_load(files["deployment.yaml"])

        container = deploy["spec"]["template"]["spec"]["containers"][0]
        assert container["resources"]["limits"]["cpu"] == "1"
        assert container["resources"]["limits"]["memory"] == "1Gi"

    def test_deployment_port(self, tmp_path):
        """Test deployment uses config port"""
        config = K8sConfig(port=3000)
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        deploy = yaml.safe_load(files["deployment.yaml"])

        container = deploy["spec"]["template"]["spec"]["containers"][0]
        assert container["ports"][0]["containerPort"] == 3000


# ============================================================================
# Service Tests
# ============================================================================

class TestServiceGeneration:
    """Tests for service.yaml generation"""

    def test_service_valid_yaml(self, tmp_path):
        """Test service is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        svc = yaml.safe_load(files["service.yaml"])

        assert svc["kind"] == "Service"
        assert "spec" in svc


# ============================================================================
# Kustomization Tests
# ============================================================================

class TestKustomizationGeneration:
    """Tests for kustomization.yaml generation"""

    def test_kustomization_valid_yaml(self, tmp_path):
        """Test kustomization is valid YAML"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        kust = yaml.safe_load(files["kustomization.yaml"])

        assert "resources" in kust

    def test_kustomization_lists_resources(self, tmp_path):
        """Test kustomization lists all resources"""
        gen = KubernetesGenerator()
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))
        kust = yaml.safe_load(files["kustomization.yaml"])

        assert "deployment.yaml" in kust["resources"]
        assert "service.yaml" in kust["resources"]


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration tests"""

    def test_full_manifest_generation(self, tmp_path):
        """Test generating complete manifest set"""
        config = K8sConfig(
            name="my-api",
            namespace="staging",
            replicas=3,
            port=8080,
            enable_ingress=True,
            enable_hpa=True,
            env_vars={"ENV": "staging"},
            secrets={"DB_PASS": "secret"}
        )
        gen = KubernetesGenerator(config)
        structure = MockProjectStructure()

        files = gen.generate(structure, str(tmp_path))

        # All files generated
        assert len(files) >= 7

        # All files valid YAML
        for filename, content in files.items():
            manifest = yaml.safe_load(content)
            assert manifest is not None

        # Config values propagated
        deploy = yaml.safe_load(files["deployment.yaml"])
        assert deploy["metadata"]["name"] == "my-api"
        assert deploy["metadata"]["namespace"] == "staging"
