"""
Tests for Project Merger

Tests the generators/merger.py module including:
- MergeStrategy enum
- MergeResult dataclass
- ProjectMerger class with various merge strategies
"""

import json
import pytest
from pathlib import Path
from dataclasses import dataclass, field
from typing import Dict, List, Any

from ucts.generators.merger import (
    MergeStrategy,
    MergeResult,
    ProjectMerger,
)


# ============================================================================
# Mock ProjectStructure
# ============================================================================

@dataclass
class MockProjectStructure:
    """Mock project structure for testing"""
    name: str = "test-project"
    languages: List[str] = field(default_factory=lambda: ["python"])
    files: Dict[str, str] = field(default_factory=dict)
    directories: List[str] = field(default_factory=list)
    dependencies: Dict[str, List[str]] = field(default_factory=dict)


# ============================================================================
# MergeStrategy Tests
# ============================================================================

class TestMergeStrategy:
    """Tests for MergeStrategy enum"""

    def test_skip_value(self):
        """Test skip strategy value"""
        assert MergeStrategy.SKIP.value == "skip"

    def test_overwrite_value(self):
        """Test overwrite strategy value"""
        assert MergeStrategy.OVERWRITE.value == "overwrite"

    def test_append_value(self):
        """Test append strategy value"""
        assert MergeStrategy.APPEND.value == "append"

    def test_smart_value(self):
        """Test smart strategy value"""
        assert MergeStrategy.SMART.value == "smart"


# ============================================================================
# MergeResult Tests
# ============================================================================

class TestMergeResult:
    """Tests for MergeResult dataclass"""

    def test_creation(self):
        """Test result creation"""
        result = MergeResult(
            files_created=["new.py"],
            files_updated=["existing.py"],
            files_skipped=["skip.py"],
            conflicts=["conflict.py"],
            dependencies_added={"python": ["requests"]}
        )
        assert result.files_created == ["new.py"]
        assert result.files_updated == ["existing.py"]
        assert result.files_skipped == ["skip.py"]
        assert result.conflicts == ["conflict.py"]
        assert result.dependencies_added["python"] == ["requests"]


# ============================================================================
# ProjectMerger Tests
# ============================================================================

class TestProjectMerger:
    """Tests for ProjectMerger class"""

    def test_init_default_strategy(self):
        """Test initialization with default strategy"""
        merger = ProjectMerger()
        assert merger.strategy == MergeStrategy.SMART

    def test_init_custom_strategy(self):
        """Test initialization with custom strategy"""
        merger = ProjectMerger(MergeStrategy.SKIP)
        assert merger.strategy == MergeStrategy.SKIP

    def test_merge_nonexistent_target_raises(self, tmp_path):
        """Test merge raises for nonexistent target"""
        merger = ProjectMerger()
        structure = MockProjectStructure()

        with pytest.raises(ValueError, match="does not exist"):
            merger.merge(structure, str(tmp_path / "nonexistent"))

    def test_merge_creates_new_files(self, tmp_path):
        """Test merge creates new files"""
        merger = ProjectMerger()
        structure = MockProjectStructure(
            files={"new_file.py": "print('hello')"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "new_file.py" in result.files_created
        assert (tmp_path / "new_file.py").exists()
        assert (tmp_path / "new_file.py").read_text() == "print('hello')"

    def test_merge_creates_directories(self, tmp_path):
        """Test merge creates directories"""
        merger = ProjectMerger()
        structure = MockProjectStructure(
            directories=["src", "src/utils", "tests"]
        )

        merger.merge(structure, str(tmp_path))

        assert (tmp_path / "src").exists()
        assert (tmp_path / "src" / "utils").exists()
        assert (tmp_path / "tests").exists()

    def test_merge_skip_strategy(self, tmp_path):
        """Test skip strategy skips existing files"""
        # Create existing file
        (tmp_path / "existing.py").write_text("original content")

        merger = ProjectMerger(MergeStrategy.SKIP)
        structure = MockProjectStructure(
            files={"existing.py": "new content"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "existing.py" in result.files_skipped
        assert (tmp_path / "existing.py").read_text() == "original content"

    def test_merge_overwrite_strategy(self, tmp_path):
        """Test overwrite strategy replaces files"""
        # Create existing file
        (tmp_path / "existing.py").write_text("original content")

        merger = ProjectMerger(MergeStrategy.OVERWRITE)
        structure = MockProjectStructure(
            files={"existing.py": "new content"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "existing.py" in result.files_updated
        assert (tmp_path / "existing.py").read_text() == "new content"

    def test_merge_append_strategy(self, tmp_path):
        """Test append strategy adds to files"""
        # Create existing file
        (tmp_path / "existing.py").write_text("original content")

        merger = ProjectMerger(MergeStrategy.APPEND)
        structure = MockProjectStructure(
            files={"existing.py": "new content"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "existing.py" in result.files_updated
        content = (tmp_path / "existing.py").read_text()
        assert "original content" in content
        assert "UCTS APPENDED" in content
        assert "new content" in content


# ============================================================================
# Smart Merge Tests
# ============================================================================

class TestSmartMerge:
    """Tests for smart merge functionality"""

    def test_smart_merge_json(self, tmp_path):
        """Test smart merge for JSON files"""
        # Create existing JSON
        (tmp_path / "config.json").write_text(json.dumps({
            "name": "test",
            "settings": {"a": 1}
        }))

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"config.json": json.dumps({
                "version": "1.0",
                "settings": {"b": 2}
            })}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "config.json" in result.files_updated
        merged = json.loads((tmp_path / "config.json").read_text())
        assert merged["name"] == "test"
        assert merged["version"] == "1.0"
        assert merged["settings"]["a"] == 1
        assert merged["settings"]["b"] == 2

    def test_smart_merge_invalid_json(self, tmp_path):
        """Test smart merge marks invalid JSON as conflict"""
        # Create existing invalid JSON
        (tmp_path / "bad.json").write_text("not valid json")

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"bad.json": json.dumps({"key": "value"})}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "bad.json" in result.conflicts

    def test_smart_merge_python_imports(self, tmp_path):
        """Test smart merge deduplicates Python imports"""
        existing = """import os
import json

def existing():
    pass
"""
        (tmp_path / "module.py").write_text(existing)

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"module.py": """import json
import requests

def new_function():
    pass
"""}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "module.py" in result.files_updated
        content = (tmp_path / "module.py").read_text()
        # Should have requests import added
        assert "import requests" in content
        # Should not duplicate json import
        assert content.count("import json") == 1

    def test_smart_merge_markdown(self, tmp_path):
        """Test smart merge appends markdown sections"""
        existing = """# Title

## Existing Section

Content here.
"""
        (tmp_path / "README.md").write_text(existing)

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"README.md": """## New Section

New content here.

## Existing Section

Should not duplicate.
"""}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "README.md" in result.files_updated
        content = (tmp_path / "README.md").read_text()
        assert "## New Section" in content
        # Should not duplicate existing section
        assert content.count("## Existing Section") == 1

    def test_smart_merge_requirements(self, tmp_path):
        """Test smart merge for requirements.txt"""
        (tmp_path / "requirements.txt").write_text("flask==2.0.0\npytest>=7.0")

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"requirements.txt": "requests\njinja2"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "requirements.txt" in result.files_updated
        content = (tmp_path / "requirements.txt").read_text()
        # Should add new deps
        assert "requests" in content
        assert "jinja2" in content
        # Original deps preserved
        assert "flask==2.0.0" in content

    def test_smart_merge_gitignore(self, tmp_path):
        """Test smart merge for .gitignore"""
        (tmp_path / ".gitignore").write_text("*.pyc\n__pycache__/")

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={".gitignore": "*.pyc\n.venv/\ndist/"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert ".gitignore" in result.files_updated
        content = (tmp_path / ".gitignore").read_text()
        assert ".venv/" in content
        assert "dist/" in content
        # Should not duplicate
        assert content.count("*.pyc") == 1

    def test_smart_merge_unknown_type_conflicts(self, tmp_path):
        """Test unknown file types marked as conflicts"""
        (tmp_path / "config.yaml").write_text("key: value")

        merger = ProjectMerger(MergeStrategy.SMART)
        structure = MockProjectStructure(
            files={"config.yaml": "other: data"}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "config.yaml" in result.conflicts


# ============================================================================
# Dependency Merge Tests
# ============================================================================

class TestDependencyMerge:
    """Tests for dependency merging"""

    def test_merge_python_deps_new_file(self, tmp_path):
        """Test creating new requirements.txt"""
        merger = ProjectMerger()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["flask", "requests"]}
        )

        result = merger.merge(structure, str(tmp_path))

        assert "python" in result.dependencies_added
        assert (tmp_path / "requirements.txt").exists()
        content = (tmp_path / "requirements.txt").read_text()
        assert "flask" in content
        assert "requests" in content

    def test_merge_python_deps_existing_file(self, tmp_path):
        """Test merging into existing requirements.txt"""
        (tmp_path / "requirements.txt").write_text("flask==2.0.0\n")

        merger = ProjectMerger()
        structure = MockProjectStructure(
            languages=["python"],
            dependencies={"python": ["flask", "pytest"]}
        )

        result = merger.merge(structure, str(tmp_path))

        # Only pytest should be added (flask exists)
        assert result.dependencies_added.get("python") == ["pytest"]

    def test_merge_node_deps_new_file(self, tmp_path):
        """Test doesn't create package.json if doesn't exist"""
        merger = ProjectMerger()
        structure = MockProjectStructure(
            languages=["javascript"],
            dependencies={"node": ["express"]}
        )

        result = merger.merge(structure, str(tmp_path))

        # No package.json should be created automatically
        assert "node" not in result.dependencies_added or not result.dependencies_added.get("node")

    def test_merge_node_deps_existing_package(self, tmp_path):
        """Test merging into existing package.json"""
        (tmp_path / "package.json").write_text(json.dumps({
            "name": "test",
            "dependencies": {"express": "^4.17.0"}
        }))

        merger = ProjectMerger()
        structure = MockProjectStructure(
            languages=["javascript"],
            dependencies={"node": ["express", "lodash"]}
        )

        result = merger.merge(structure, str(tmp_path))

        assert result.dependencies_added.get("node") == ["lodash"]
        pkg = json.loads((tmp_path / "package.json").read_text())
        assert "lodash" in pkg["dependencies"]


# ============================================================================
# Gitignore Merge Tests
# ============================================================================

class TestGitignoreMerge:
    """Tests for .gitignore merging"""

    def test_create_gitignore_python(self, tmp_path):
        """Test creating .gitignore for Python project"""
        merger = ProjectMerger()
        structure = MockProjectStructure(languages=["python"])

        merger.merge(structure, str(tmp_path))

        assert (tmp_path / ".gitignore").exists()
        content = (tmp_path / ".gitignore").read_text()
        assert "__pycache__/" in content
        assert "*.pyc" in content

    def test_create_gitignore_node(self, tmp_path):
        """Test creating .gitignore for Node.js project"""
        merger = ProjectMerger()
        structure = MockProjectStructure(languages=["javascript"])

        merger.merge(structure, str(tmp_path))

        assert (tmp_path / ".gitignore").exists()
        content = (tmp_path / ".gitignore").read_text()
        assert "node_modules/" in content

    def test_merge_gitignore_existing(self, tmp_path):
        """Test merging with existing .gitignore"""
        (tmp_path / ".gitignore").write_text("*.log\n")

        merger = ProjectMerger()
        structure = MockProjectStructure(languages=["python"])

        merger.merge(structure, str(tmp_path))

        content = (tmp_path / ".gitignore").read_text()
        assert "*.log" in content
        assert "__pycache__/" in content


# ============================================================================
# Deep Merge Tests
# ============================================================================

class TestDeepMerge:
    """Tests for deep dictionary merging"""

    def test_deep_merge_simple(self):
        """Test simple dictionary merge"""
        merger = ProjectMerger()
        base = {"a": 1, "b": 2}
        override = {"b": 3, "c": 4}

        result = merger._deep_merge_dict(base, override)

        assert result["a"] == 1
        assert result["b"] == 3  # Override
        assert result["c"] == 4

    def test_deep_merge_nested(self):
        """Test nested dictionary merge"""
        merger = ProjectMerger()
        base = {"config": {"debug": True, "port": 8000}}
        override = {"config": {"port": 3000, "host": "localhost"}}

        result = merger._deep_merge_dict(base, override)

        assert result["config"]["debug"] is True
        assert result["config"]["port"] == 3000
        assert result["config"]["host"] == "localhost"

    def test_deep_merge_lists(self):
        """Test list merging without duplicates"""
        merger = ProjectMerger()
        base = {"plugins": ["a", "b"]}
        override = {"plugins": ["b", "c"]}

        result = merger._deep_merge_dict(base, override)

        # Should have a, b, c (no duplicate b)
        assert "a" in result["plugins"]
        assert "b" in result["plugins"]
        assert "c" in result["plugins"]


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration tests for merger"""

    def test_full_project_merge(self, tmp_path):
        """Test merging a complete project structure"""
        # Create existing project
        (tmp_path / "src").mkdir()
        (tmp_path / "src" / "main.py").write_text("# Existing main\n")
        (tmp_path / "requirements.txt").write_text("flask\n")
        (tmp_path / "README.md").write_text("# Existing Project\n")

        structure = MockProjectStructure(
            languages=["python"],
            directories=["src", "tests"],
            files={
                "src/utils.py": "# New utils\n",
                "tests/__init__.py": "",
                "README.md": "## Installation\n\nNew section."
            },
            dependencies={"python": ["flask", "pytest"]}
        )

        merger = ProjectMerger(MergeStrategy.SMART)
        result = merger.merge(structure, str(tmp_path))

        # Check new files created
        assert "src/utils.py" in result.files_created
        assert "tests/__init__.py" in result.files_created

        # Check README merged
        assert "README.md" in result.files_updated

        # Check dependencies
        assert result.dependencies_added.get("python") == ["pytest"]

        # Verify file contents
        assert (tmp_path / "src" / "utils.py").exists()
        assert (tmp_path / "tests" / "__init__.py").exists()
