"""Tests for Plugin System"""
import json
import os
from datetime import datetime
from pathlib import Path
from typing import Any, Dict
from unittest.mock import patch, MagicMock

import pytest

from ucts.plugins.system import (
    PluginType,
    HookType,
    PluginConfig,
    Plugin,
    IngesterPlugin,
    GeneratorPlugin,
    AnalyzerPlugin,
    HookPlugin,
    PluginMetadata,
    PluginManager,
    get_plugin_manager,
)


class TestEnums:
    """Tests for plugin enums"""

    def test_plugin_types(self):
        """Test PluginType enum values"""
        assert PluginType.INGESTER.value == "ingester"
        assert PluginType.GENERATOR.value == "generator"
        assert PluginType.ANALYZER.value == "analyzer"
        assert PluginType.HOOK.value == "hook"
        assert PluginType.FORMATTER.value == "formatter"
        assert PluginType.TRANSFORMER.value == "transformer"

    def test_hook_types(self):
        """Test HookType enum values"""
        assert HookType.PRE_INGEST.value == "pre_ingest"
        assert HookType.POST_INGEST.value == "post_ingest"
        assert HookType.PRE_ANALYZE.value == "pre_analyze"
        assert HookType.POST_ANALYZE.value == "post_analyze"
        assert HookType.PRE_GENERATE.value == "pre_generate"
        assert HookType.POST_GENERATE.value == "post_generate"
        assert HookType.PRE_FORGE.value == "pre_forge"
        assert HookType.POST_FORGE.value == "post_forge"


class TestPluginConfig:
    """Tests for PluginConfig dataclass"""

    def test_default_values(self):
        """Test PluginConfig with required fields only"""
        config = PluginConfig(name="test", version="1.0.0")

        assert config.name == "test"
        assert config.version == "1.0.0"
        assert config.description == ""
        assert config.author == ""
        assert config.plugin_type == PluginType.HOOK
        assert config.dependencies == []
        assert config.config == {}
        assert config.enabled is True

    def test_full_config(self):
        """Test PluginConfig with all fields"""
        config = PluginConfig(
            name="full-plugin",
            version="2.0.0",
            description="A full plugin",
            author="Test Author",
            plugin_type=PluginType.ANALYZER,
            dependencies=["dep1", "dep2"],
            config={"key": "value"},
            enabled=False
        )

        assert config.name == "full-plugin"
        assert config.plugin_type == PluginType.ANALYZER
        assert len(config.dependencies) == 2
        assert config.config["key"] == "value"
        assert config.enabled is False


class TestPluginMetadata:
    """Tests for PluginMetadata dataclass"""

    def test_metadata_creation(self):
        """Test creating PluginMetadata"""
        metadata = PluginMetadata(
            name="test-plugin",
            version="1.0.0",
            plugin_type=PluginType.INGESTER,
            path="/path/to/plugin",
            class_name="TestPlugin"
        )

        assert metadata.name == "test-plugin"
        assert metadata.version == "1.0.0"
        assert metadata.plugin_type == PluginType.INGESTER
        assert metadata.path == "/path/to/plugin"
        assert metadata.class_name == "TestPlugin"
        assert metadata.enabled is True
        assert metadata.instance is None

    def test_metadata_with_instance(self):
        """Test metadata with plugin instance"""
        mock_instance = MagicMock()
        metadata = PluginMetadata(
            name="test",
            version="1.0",
            plugin_type=PluginType.HOOK,
            path="/path",
            class_name="Plugin",
            instance=mock_instance
        )

        assert metadata.instance is mock_instance


# Sample plugin implementations for testing
class SampleHookPlugin(HookPlugin):
    """Sample hook plugin for testing"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="sample-hook",
            version="1.0.0",
            plugin_type=PluginType.HOOK
        )

    def initialize(self) -> bool:
        return True

    @property
    def hook_type(self) -> HookType:
        return HookType.POST_FORGE

    def execute(self, context: Dict[str, Any]) -> Dict[str, Any]:
        context["hook_executed"] = True
        return context


class SampleIngesterPlugin(IngesterPlugin):
    """Sample ingester plugin for testing"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="sample-ingester",
            version="1.0.0",
            plugin_type=PluginType.INGESTER
        )

    def initialize(self) -> bool:
        return True

    def can_ingest(self, source: str) -> bool:
        return source.endswith(".sample")

    def ingest(self, source: str):
        from ucts.core.models import Session
        return Session(source=source, messages=[], code_blocks=[])


class SampleGeneratorPlugin(GeneratorPlugin):
    """Sample generator plugin for testing"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="sample-generator",
            version="1.0.0",
            plugin_type=PluginType.GENERATOR
        )

    def initialize(self) -> bool:
        return True

    def can_generate(self, structure, target: str) -> bool:
        return target == "sample"

    def generate(self, structure, output: str) -> Dict[str, str]:
        return {"sample.txt": "Generated content"}


class SampleAnalyzerPlugin(AnalyzerPlugin):
    """Sample analyzer plugin for testing"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="sample-analyzer",
            version="1.0.0",
            plugin_type=PluginType.ANALYZER
        )

    def initialize(self) -> bool:
        return True

    def analyze(self, session) -> Dict[str, Any]:
        return {"analyzed": True}


class FailingPlugin(HookPlugin):
    """Plugin that fails to initialize"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(name="failing", version="1.0.0")

    def initialize(self) -> bool:
        return False

    @property
    def hook_type(self) -> HookType:
        return HookType.PRE_FORGE

    def execute(self, context: Dict[str, Any]) -> Dict[str, Any]:
        return context


class TestPluginBaseClasses:
    """Tests for Plugin base classes"""

    def test_hook_plugin(self):
        """Test HookPlugin"""
        plugin = SampleHookPlugin()
        assert plugin.initialize()
        assert plugin.name == "sample-hook"
        assert plugin.version == "1.0.0"
        assert plugin.plugin_type == PluginType.HOOK
        assert plugin.hook_type == HookType.POST_FORGE

        context = {"data": "test"}
        result = plugin.execute(context)
        assert result["hook_executed"] is True

    def test_ingester_plugin(self):
        """Test IngesterPlugin"""
        plugin = SampleIngesterPlugin()
        assert plugin.initialize()
        assert plugin.can_ingest("test.sample")
        assert not plugin.can_ingest("test.json")

    def test_generator_plugin(self):
        """Test GeneratorPlugin"""
        plugin = SampleGeneratorPlugin()
        assert plugin.initialize()
        assert plugin.can_generate(None, "sample")
        assert not plugin.can_generate(None, "other")

    def test_analyzer_plugin(self):
        """Test AnalyzerPlugin"""
        plugin = SampleAnalyzerPlugin()
        assert plugin.initialize()
        result = plugin.analyze(None)
        assert result["analyzed"] is True

    def test_plugin_shutdown(self):
        """Test plugin shutdown"""
        plugin = SampleHookPlugin()
        plugin.initialize()
        # Should not raise
        plugin.shutdown()


class TestPluginManager:
    """Tests for PluginManager"""

    def test_init_default_dirs(self):
        """Test PluginManager initialization with default dirs"""
        manager = PluginManager()

        assert len(manager.plugin_dirs) >= 1
        assert any(".ucts" in d for d in manager.plugin_dirs)

    def test_init_custom_dirs(self, tmp_path):
        """Test PluginManager initialization with custom dirs"""
        plugin_dir = str(tmp_path / "plugins")
        manager = PluginManager(plugin_dirs=[plugin_dir])

        assert manager.plugin_dirs == [plugin_dir]

    def test_discover_plugins_empty(self, tmp_path):
        """Test discovering plugins in empty directory"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        discovered = manager.discover_plugins()

        assert discovered == []

    def test_discover_plugins_with_manifest(self, tmp_path):
        """Test discovering plugins with manifest"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        # Create plugin with manifest
        test_plugin = plugin_dir / "test-plugin"
        test_plugin.mkdir()

        manifest = {
            "name": "test-plugin",
            "version": "1.0.0",
            "type": "hook",
            "class": "TestPlugin"
        }
        (test_plugin / "plugin.json").write_text(json.dumps(manifest))

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        discovered = manager.discover_plugins()

        assert len(discovered) == 1
        assert discovered[0].name == "test-plugin"
        assert discovered[0].version == "1.0.0"

    def test_load_manifest(self, tmp_path):
        """Test loading plugin manifest"""
        manifest_data = {
            "name": "manifest-test",
            "version": "2.0.0",
            "type": "analyzer",
            "class": "ManifestPlugin",
            "enabled": True
        }
        manifest_path = tmp_path / "plugin.json"
        manifest_path.write_text(json.dumps(manifest_data))

        manager = PluginManager()
        metadata = manager._load_manifest(str(manifest_path))

        assert metadata.name == "manifest-test"
        assert metadata.version == "2.0.0"
        assert metadata.plugin_type == PluginType.ANALYZER
        assert metadata.class_name == "ManifestPlugin"

    def test_register_hook_plugin(self):
        """Test registering hook plugin"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleHookPlugin()

        manager._register_plugin(plugin)

        assert len(manager._hooks[HookType.POST_FORGE]) == 1
        assert manager._hooks[HookType.POST_FORGE][0] is plugin

    def test_register_ingester_plugin(self):
        """Test registering ingester plugin"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleIngesterPlugin()

        manager._register_plugin(plugin)

        assert len(manager._ingesters) == 1
        assert manager._ingesters[0] is plugin

    def test_register_generator_plugin(self):
        """Test registering generator plugin"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleGeneratorPlugin()

        manager._register_plugin(plugin)

        assert len(manager._generators) == 1

    def test_register_analyzer_plugin(self):
        """Test registering analyzer plugin"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleAnalyzerPlugin()

        manager._register_plugin(plugin)

        assert len(manager._analyzers) == 1

    def test_execute_hooks(self):
        """Test executing hooks"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleHookPlugin()
        plugin.initialize()
        manager._register_plugin(plugin)

        context = {"initial": "data"}
        result = manager.execute_hooks(HookType.POST_FORGE, context)

        assert result["initial"] == "data"
        assert result["hook_executed"] is True

    def test_execute_hooks_empty(self):
        """Test executing hooks with no registered hooks"""
        manager = PluginManager(plugin_dirs=[])

        context = {"data": "test"}
        result = manager.execute_hooks(HookType.PRE_FORGE, context)

        assert result == context

    def test_execute_hooks_error_handling(self):
        """Test that hook errors don't stop execution"""
        manager = PluginManager(plugin_dirs=[])

        # Create a hook that raises an exception
        class ErrorHook(HookPlugin):
            @classmethod
            def default_config(cls):
                return PluginConfig(name="error", version="1.0")

            def initialize(self):
                return True

            @property
            def hook_type(self):
                return HookType.PRE_FORGE

            def execute(self, context):
                raise ValueError("Hook error")

        error_hook = ErrorHook()
        error_hook.initialize()
        manager._register_plugin(error_hook)

        # Should not raise
        result = manager.execute_hooks(HookType.PRE_FORGE, {"data": "test"})
        assert result == {"data": "test"}

    def test_get_ingester(self):
        """Test getting ingester for source"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleIngesterPlugin()
        plugin.initialize()
        manager._register_plugin(plugin)

        ingester = manager.get_ingester("test.sample")
        assert ingester is plugin

        no_ingester = manager.get_ingester("test.json")
        assert no_ingester is None

    def test_get_generator(self):
        """Test getting generator for target"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleGeneratorPlugin()
        plugin.initialize()
        manager._register_plugin(plugin)

        generator = manager.get_generator(None, "sample")
        assert generator is plugin

        no_generator = manager.get_generator(None, "other")
        assert no_generator is None

    def test_get_analyzers(self):
        """Test getting all analyzers"""
        manager = PluginManager(plugin_dirs=[])
        plugin = SampleAnalyzerPlugin()
        plugin.initialize()
        manager._register_plugin(plugin)

        analyzers = manager.get_analyzers()

        assert len(analyzers) == 1
        # Should return a copy
        analyzers.append(None)
        assert len(manager._analyzers) == 1

    def test_list_plugins(self):
        """Test listing loaded plugins"""
        manager = PluginManager(plugin_dirs=[])

        # Initially empty
        assert manager.list_plugins() == []

    def test_get_plugin_not_found(self):
        """Test getting non-existent plugin"""
        manager = PluginManager(plugin_dirs=[])

        result = manager.get_plugin("nonexistent")
        assert result is None

    def test_unload_plugin_not_found(self):
        """Test unloading non-existent plugin"""
        manager = PluginManager(plugin_dirs=[])

        result = manager.unload_plugin("nonexistent")
        assert result is False


class TestPluginTemplates:
    """Tests for plugin template generation"""

    def test_create_ingester_template(self, tmp_path):
        """Test creating ingester plugin template"""
        manager = PluginManager(plugin_dirs=[])

        files = manager.create_plugin_template(
            "my-ingester",
            PluginType.INGESTER,
            str(tmp_path)
        )

        assert "plugin.json" in files
        assert "__init__.py" in files
        assert "README.md" in files

        # Check manifest content
        manifest_path = tmp_path / "my-ingester" / "plugin.json"
        manifest = json.loads(manifest_path.read_text())
        assert manifest["name"] == "my-ingester"
        assert manifest["type"] == "ingester"

        # Check code content
        init_path = tmp_path / "my-ingester" / "__init__.py"
        code = init_path.read_text()
        assert "IngesterPlugin" in code
        assert "can_ingest" in code
        assert "ingest" in code

    def test_create_generator_template(self, tmp_path):
        """Test creating generator plugin template"""
        manager = PluginManager(plugin_dirs=[])

        files = manager.create_plugin_template(
            "my-generator",
            PluginType.GENERATOR,
            str(tmp_path)
        )

        init_path = tmp_path / "my-generator" / "__init__.py"
        code = init_path.read_text()
        assert "GeneratorPlugin" in code
        assert "can_generate" in code
        assert "generate" in code

    def test_create_analyzer_template(self, tmp_path):
        """Test creating analyzer plugin template"""
        manager = PluginManager(plugin_dirs=[])

        files = manager.create_plugin_template(
            "my-analyzer",
            PluginType.ANALYZER,
            str(tmp_path)
        )

        init_path = tmp_path / "my-analyzer" / "__init__.py"
        code = init_path.read_text()
        assert "AnalyzerPlugin" in code
        assert "analyze" in code

    def test_create_hook_template(self, tmp_path):
        """Test creating hook plugin template"""
        manager = PluginManager(plugin_dirs=[])

        files = manager.create_plugin_template(
            "my-hook",
            PluginType.HOOK,
            str(tmp_path)
        )

        init_path = tmp_path / "my-hook" / "__init__.py"
        code = init_path.read_text()
        assert "HookPlugin" in code
        assert "hook_type" in code
        assert "execute" in code

    def test_create_template_readme(self, tmp_path):
        """Test that template includes README"""
        manager = PluginManager(plugin_dirs=[])

        manager.create_plugin_template(
            "test-plugin",
            PluginType.HOOK,
            str(tmp_path)
        )

        readme_path = tmp_path / "test-plugin" / "README.md"
        readme = readme_path.read_text()
        assert "test-plugin" in readme
        assert "Installation" in readme
        assert ".ucts/plugins" in readme


class TestGetPluginManager:
    """Tests for get_plugin_manager singleton"""

    def test_returns_instance(self):
        """Test that get_plugin_manager returns an instance"""
        # Reset singleton
        import ucts.plugins.system as plugin_module
        plugin_module._plugin_manager = None

        manager = get_plugin_manager()

        assert manager is not None
        assert isinstance(manager, PluginManager)

    def test_singleton_behavior(self):
        """Test singleton returns same instance"""
        import ucts.plugins.system as plugin_module
        plugin_module._plugin_manager = None

        manager1 = get_plugin_manager()
        manager2 = get_plugin_manager()

        assert manager1 is manager2


class TestPluginLoading:
    """Tests for plugin loading functionality"""

    def test_load_plugin_single_file(self, tmp_path):
        """Test loading a single-file plugin"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        # Create a simple plugin file
        plugin_code = '''
from ucts.plugins.system import Plugin, PluginConfig, PluginType, HookPlugin, HookType

class SimplePlugin(HookPlugin):
    @classmethod
    def default_config(cls):
        return PluginConfig(name="simple", version="1.0.0", plugin_type=PluginType.HOOK)

    def initialize(self):
        return True

    @property
    def hook_type(self):
        return HookType.POST_FORGE

    def execute(self, context):
        return context
'''
        plugin_file = plugin_dir / "simple_plugin.py"
        plugin_file.write_text(plugin_code)

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        discovered = manager.discover_plugins()

        # Should discover the plugin
        assert len(discovered) == 1
        assert discovered[0].name == "simple"

    def test_load_plugin_directory(self, tmp_path):
        """Test loading a directory plugin"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        # Create plugin directory
        test_plugin = plugin_dir / "dir-plugin"
        test_plugin.mkdir()

        manifest = {
            "name": "dir-plugin",
            "version": "1.0.0",
            "type": "hook",
            "class": "DirPlugin"
        }
        (test_plugin / "plugin.json").write_text(json.dumps(manifest))

        init_code = '''
from ucts.plugins.system import Plugin, PluginConfig, PluginType, HookPlugin, HookType

class DirPlugin(HookPlugin):
    @classmethod
    def default_config(cls):
        return PluginConfig(name="dir-plugin", version="1.0.0")

    def initialize(self):
        return True

    @property
    def hook_type(self):
        return HookType.PRE_FORGE

    def execute(self, context):
        return context
'''
        (test_plugin / "__init__.py").write_text(init_code)

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        discovered = manager.discover_plugins()

        assert len(discovered) == 1
        assert discovered[0].name == "dir-plugin"

        # Load the plugin
        result = manager.load_plugin(discovered[0])
        assert result is True

        # Check it's registered
        plugin = manager.get_plugin("dir-plugin")
        assert plugin is not None

    def test_load_all_plugins(self, tmp_path):
        """Test loading all discovered plugins"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        # Create two plugins
        for i in range(2):
            test_plugin = plugin_dir / f"plugin-{i}"
            test_plugin.mkdir()

            manifest = {
                "name": f"plugin-{i}",
                "version": "1.0.0",
                "type": "hook",
                "class": f"Plugin{i}"
            }
            (test_plugin / "plugin.json").write_text(json.dumps(manifest))

            init_code = f'''
from ucts.plugins.system import Plugin, PluginConfig, PluginType, HookPlugin, HookType

class Plugin{i}(HookPlugin):
    @classmethod
    def default_config(cls):
        return PluginConfig(name="plugin-{i}", version="1.0.0")

    def initialize(self):
        return True

    @property
    def hook_type(self):
        return HookType.POST_FORGE

    def execute(self, context):
        return context
'''
            (test_plugin / "__init__.py").write_text(init_code)

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        loaded = manager.load_all_plugins()

        assert loaded == 2
        assert len(manager.list_plugins()) == 2

    def test_unload_plugin(self, tmp_path):
        """Test unloading a plugin"""
        plugin_dir = tmp_path / "plugins"
        plugin_dir.mkdir()

        test_plugin = plugin_dir / "unload-test"
        test_plugin.mkdir()

        manifest = {
            "name": "unload-test",
            "version": "1.0.0",
            "type": "hook",
            "class": "UnloadPlugin"
        }
        (test_plugin / "plugin.json").write_text(json.dumps(manifest))

        init_code = '''
from ucts.plugins.system import Plugin, PluginConfig, PluginType, HookPlugin, HookType

class UnloadPlugin(HookPlugin):
    @classmethod
    def default_config(cls):
        return PluginConfig(name="unload-test", version="1.0.0")

    def initialize(self):
        return True

    def shutdown(self):
        pass

    @property
    def hook_type(self):
        return HookType.POST_FORGE

    def execute(self, context):
        return context
'''
        (test_plugin / "__init__.py").write_text(init_code)

        manager = PluginManager(plugin_dirs=[str(plugin_dir)])
        manager.load_all_plugins()

        assert manager.get_plugin("unload-test") is not None

        result = manager.unload_plugin("unload-test")
        assert result is True
        assert manager.get_plugin("unload-test") is None
