"""Tests for Sentinel integration"""
import json
from pathlib import Path
from unittest.mock import patch, MagicMock, Mock

import pytest

from ucts.integrations.sentinel import (
    SentinelIntegration,
    SentinelError,
    SentinelNotFoundError,
    SentinelStateError,
    SentinelModuleError,
)


class TestSentinelIntegration:
    """Tests for Sentinel integration"""

    def test_init_with_custom_path(self, tmp_path):
        """Test initialization with custom path"""
        integration = SentinelIntegration(str(tmp_path))

        assert integration.sentinel_path == tmp_path

    def test_init_finds_execution_dir(self, tmp_path, monkeypatch):
        """Test that init finds directory with execution folder"""
        (tmp_path / "execution").mkdir()
        monkeypatch.chdir(tmp_path)

        integration = SentinelIntegration()
        assert integration.sentinel_path == tmp_path

    def test_init_defaults_to_cwd(self, tmp_path, monkeypatch):
        """Test that init defaults to cwd when no execution dir found"""
        monkeypatch.chdir(tmp_path)

        integration = SentinelIntegration()
        # When no execution dir is found in any of the search paths,
        # defaults to cwd (or the first location that has execution dir)
        # Just verify it's a valid Path
        assert isinstance(integration.sentinel_path, Path)

    def test_validate_sentinel_path_true(self, tmp_path):
        """Test validation returns True when execution dir exists"""
        (tmp_path / "execution").mkdir()
        integration = SentinelIntegration(str(tmp_path))

        assert integration._validate_sentinel_path() is True

    def test_validate_sentinel_path_false(self, tmp_path):
        """Test validation returns False when execution dir missing"""
        integration = SentinelIntegration(str(tmp_path))

        assert integration._validate_sentinel_path() is False

    def test_capture_execution_state_not_found(self, tmp_path):
        """Test capturing state when file doesn't exist"""
        integration = SentinelIntegration(str(tmp_path))

        result = integration.capture_execution_state()

        assert result["status"] == "not_found"
        assert "not found" in result["message"].lower()

    def test_capture_execution_state_success(self, tmp_path):
        """Test capturing existing execution state"""
        (tmp_path / "execution").mkdir()
        state_file = tmp_path / "execution" / "execution_state.json"
        state_file.write_text(json.dumps({
            "current_phase": "development",
            "tasks": {
                "task1": {"status": "completed"},
                "task2": {"status": "pending"},
                "task3": {"status": "completed"},
            },
            "agent_registry": {
                "agent1": {"type": "executor"},
                "agent2": {"type": "analyzer"},
            },
            "ready": ["item1", "item2"]
        }))

        integration = SentinelIntegration(str(tmp_path))
        result = integration.capture_execution_state()

        assert result["status"] == "captured"
        assert result["phase"] == "development"
        assert result["tasks"]["total"] == 3
        assert result["tasks"]["completed"] == 2
        assert "agent1" in result["agents"]
        assert "agent2" in result["agents"]
        assert result["ready_for_work"] is True

    def test_capture_execution_state_handles_invalid_json(self, tmp_path):
        """Test handling of invalid JSON in execution state"""
        (tmp_path / "execution").mkdir()
        state_file = tmp_path / "execution" / "execution_state.json"
        state_file.write_text("not valid json")

        integration = SentinelIntegration(str(tmp_path))

        with pytest.raises(SentinelStateError, match="Invalid JSON"):
            integration.capture_execution_state()

    def test_capture_execution_state_handles_permission_error(self, tmp_path):
        """Test handling of permission error"""
        (tmp_path / "execution").mkdir()
        state_file = tmp_path / "execution" / "execution_state.json"
        state_file.write_text("{}")

        integration = SentinelIntegration(str(tmp_path))

        with patch("builtins.open", side_effect=PermissionError("Access denied")):
            with pytest.raises(SentinelStateError, match="Permission denied"):
                integration.capture_execution_state()

    def test_capture_execution_state_handles_non_dict_values(self, tmp_path):
        """Test handling of non-dict values in state"""
        (tmp_path / "execution").mkdir()
        state_file = tmp_path / "execution" / "execution_state.json"
        state_file.write_text(json.dumps({
            "current_phase": "test",
            "tasks": "not a dict",
            "agent_registry": "also not a dict",
            "ready": "not a list"
        }))

        integration = SentinelIntegration(str(tmp_path))
        result = integration.capture_execution_state()

        assert result["tasks"]["total"] == 0
        assert result["agents"] == []
        assert result["ready_for_work"] is False

    def test_get_agent_registry_returns_empty_when_missing(self, tmp_path):
        """Test get_agent_registry returns empty list when file missing"""
        integration = SentinelIntegration(str(tmp_path))

        result = integration.get_agent_registry()
        assert result == []

    def test_get_agent_registry_returns_agents(self, tmp_path):
        """Test get_agent_registry returns agent list"""
        (tmp_path / "agents" / "registry").mkdir(parents=True)
        agents_file = tmp_path / "agents" / "registry" / "agents.json"
        agents_file.write_text(json.dumps({
            "agents": [
                {"id": "agent1", "type": "executor", "status": "active", "capabilities": ["build"]},
                {"id": "agent2", "type": "analyzer", "status": "idle", "capabilities": ["scan", "report"]},
            ]
        }))

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_agent_registry()

        assert len(result) == 2
        assert result[0]["id"] == "agent1"
        assert result[0]["type"] == "executor"
        assert "build" in result[0]["capabilities"]

    def test_get_agent_registry_handles_invalid_json(self, tmp_path):
        """Test handling of invalid JSON in agent registry"""
        (tmp_path / "agents" / "registry").mkdir(parents=True)
        agents_file = tmp_path / "agents" / "registry" / "agents.json"
        agents_file.write_text("not valid json")

        integration = SentinelIntegration(str(tmp_path))

        with pytest.raises(SentinelStateError, match="Invalid JSON"):
            integration.get_agent_registry()

    def test_get_agent_registry_handles_non_dict(self, tmp_path):
        """Test handling of non-dict agent registry"""
        (tmp_path / "agents" / "registry").mkdir(parents=True)
        agents_file = tmp_path / "agents" / "registry" / "agents.json"
        agents_file.write_text('["not", "a", "dict"]')

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_agent_registry()

        assert result == []

    def test_get_agent_registry_handles_non_list_agents(self, tmp_path):
        """Test handling when agents key is not a list"""
        (tmp_path / "agents" / "registry").mkdir(parents=True)
        agents_file = tmp_path / "agents" / "registry" / "agents.json"
        agents_file.write_text('{"agents": "not a list"}')

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_agent_registry()

        assert result == []

    def test_get_agent_registry_skips_non_dict_agents(self, tmp_path):
        """Test that non-dict agent entries are skipped"""
        (tmp_path / "agents" / "registry").mkdir(parents=True)
        agents_file = tmp_path / "agents" / "registry" / "agents.json"
        agents_file.write_text(json.dumps({
            "agents": [
                {"id": "agent1", "type": "executor"},
                "not a dict",
                {"id": "agent2", "type": "analyzer"},
            ]
        }))

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_agent_registry()

        assert len(result) == 2

    def test_forge_sentinel_module_requires_session(self, tmp_path):
        """Test that forge_sentinel_module requires session"""
        integration = SentinelIntegration(str(tmp_path))

        with pytest.raises(ValueError, match="session cannot be None"):
            integration.forge_sentinel_module(None, "agent")

    def test_forge_sentinel_module_creates_module(self, tmp_path):
        """Test that forge_sentinel_module creates a module"""
        integration = SentinelIntegration(str(tmp_path))

        # Mock the session object
        mock_session = MagicMock()
        mock_session.messages = []
        mock_session.code_blocks = []
        mock_session.todos = []

        # Mock the analysis and generation - patch where they're imported
        with patch("ucts.analysis.AnalysisEngine") as mock_engine_cls:
            with patch("ucts.generators.VSCodeGenerator") as mock_gen_cls:
                mock_engine = MagicMock()
                mock_structure = MagicMock()
                mock_structure.name = "test-module"
                mock_structure.files = {}
                mock_engine.analyze.return_value = mock_structure
                mock_engine_cls.return_value = mock_engine

                mock_gen = MagicMock()
                mock_gen.generate.return_value = str(tmp_path / "modules" / "sentinel-test-module")
                mock_gen_cls.return_value = mock_gen

                result = integration.forge_sentinel_module(mock_session, "agent")

                assert result["status"] == "created"
                assert "sentinel-" in result["module_name"]
                assert result["module_type"] == "agent"

    def test_forge_sentinel_module_handles_import_error(self, tmp_path):
        """Test handling of import errors"""
        integration = SentinelIntegration(str(tmp_path))
        mock_session = MagicMock()

        # Temporarily break the import by patching sys.modules
        import sys
        original_analysis = sys.modules.get('ucts.analysis')
        try:
            sys.modules['ucts.analysis'] = None  # type: ignore
            with pytest.raises(SentinelModuleError, match="Failed to import"):
                integration.forge_sentinel_module(mock_session, "agent")
        finally:
            if original_analysis is not None:
                sys.modules['ucts.analysis'] = original_analysis
            elif 'ucts.analysis' in sys.modules:
                del sys.modules['ucts.analysis']

    def test_forge_sentinel_module_handles_analysis_error(self, tmp_path):
        """Test handling of analysis errors"""
        integration = SentinelIntegration(str(tmp_path))
        mock_session = MagicMock()

        with patch("ucts.analysis.AnalysisEngine") as mock_engine_cls:
            mock_engine = MagicMock()
            mock_engine.analyze.side_effect = Exception("Analysis failed")
            mock_engine_cls.return_value = mock_engine

            with pytest.raises(SentinelModuleError, match="Failed to analyze"):
                integration.forge_sentinel_module(mock_session, "agent")

    def test_forge_sentinel_module_handles_invalid_structure(self, tmp_path):
        """Test handling of invalid analysis result"""
        integration = SentinelIntegration(str(tmp_path))
        mock_session = MagicMock()

        with patch("ucts.analysis.AnalysisEngine") as mock_engine_cls:
            mock_engine = MagicMock()
            mock_engine.analyze.return_value = None
            mock_engine_cls.return_value = mock_engine

            with pytest.raises(SentinelModuleError, match="invalid structure"):
                integration.forge_sentinel_module(mock_session, "agent")

    def test_sync_with_coherence_requires_dict(self, tmp_path):
        """Test that sync_with_coherence requires dict"""
        integration = SentinelIntegration(str(tmp_path))

        with pytest.raises(ValueError, match="must be a dictionary"):
            integration.sync_with_coherence("not a dict")  # type: ignore

    def test_sync_with_coherence_creates_file(self, tmp_path):
        """Test that sync_with_coherence creates coherence state"""
        integration = SentinelIntegration(str(tmp_path))

        result = integration.sync_with_coherence({
            "status": "operational",
            "available_actions": ["forge", "transfer"]
        })

        assert result["status"] == "synced"
        assert "synced_at" in result

        coherence_file = tmp_path / "execution" / "coherence_state.json"
        assert coherence_file.exists()

        content = json.loads(coherence_file.read_text())
        assert "ucts" in content
        assert content["ucts"]["status"] == "operational"

    def test_sync_with_coherence_preserves_existing(self, tmp_path):
        """Test that sync preserves existing coherence data"""
        (tmp_path / "execution").mkdir()
        coherence_file = tmp_path / "execution" / "coherence_state.json"
        coherence_file.write_text(json.dumps({
            "existing_data": "should be preserved",
            "ucts": {"old": "data"}
        }))

        integration = SentinelIntegration(str(tmp_path))
        integration.sync_with_coherence({"status": "new"})

        content = json.loads(coherence_file.read_text())
        assert content["existing_data"] == "should be preserved"
        assert content["ucts"]["status"] == "new"

    def test_sync_with_coherence_handles_invalid_existing_json(self, tmp_path):
        """Test handling of invalid existing coherence state"""
        (tmp_path / "execution").mkdir()
        coherence_file = tmp_path / "execution" / "coherence_state.json"
        coherence_file.write_text("not valid json")

        integration = SentinelIntegration(str(tmp_path))
        result = integration.sync_with_coherence({"status": "new"})

        # Should still succeed, resetting the state
        assert result["status"] == "synced"

    def test_sync_with_coherence_handles_non_dict_existing(self, tmp_path):
        """Test handling of non-dict existing coherence state"""
        (tmp_path / "execution").mkdir()
        coherence_file = tmp_path / "execution" / "coherence_state.json"
        coherence_file.write_text('["not", "a", "dict"]')

        integration = SentinelIntegration(str(tmp_path))
        result = integration.sync_with_coherence({"status": "new"})

        assert result["status"] == "synced"

    def test_sync_with_coherence_handles_permission_error(self, tmp_path):
        """Test handling of permission error during sync"""
        integration = SentinelIntegration(str(tmp_path))

        with patch("builtins.open", side_effect=PermissionError("Access denied")):
            with pytest.raises(SentinelStateError, match="Permission denied"):
                integration.sync_with_coherence({"status": "test"})

    def test_get_available_modules_returns_empty_when_missing(self, tmp_path):
        """Test get_available_modules returns empty when dir missing"""
        integration = SentinelIntegration(str(tmp_path))

        result = integration.get_available_modules()
        assert result == []

    def test_get_available_modules_returns_modules(self, tmp_path):
        """Test get_available_modules returns module list"""
        modules_dir = tmp_path / "modules"
        modules_dir.mkdir()

        # Module with config
        (modules_dir / "sentinel-module1").mkdir()
        (modules_dir / "sentinel-module1" / "sentinel_config.json").write_text(
            json.dumps({"module_type": "agent", "version": "1.0.0"})
        )

        # Module without config
        (modules_dir / "sentinel-module2").mkdir()

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_available_modules()

        assert len(result) == 2

        module1 = next(m for m in result if m["name"] == "sentinel-module1")
        assert module1["type"] == "agent"
        assert module1["version"] == "1.0.0"

        module2 = next(m for m in result if m["name"] == "sentinel-module2")
        assert module2["type"] == "unknown"
        assert module2["version"] == "unknown"

    def test_get_available_modules_handles_invalid_config(self, tmp_path):
        """Test handling of invalid config in module"""
        modules_dir = tmp_path / "modules"
        modules_dir.mkdir()

        (modules_dir / "bad-module").mkdir()
        (modules_dir / "bad-module" / "sentinel_config.json").write_text("invalid json")

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_available_modules()

        assert len(result) == 1
        assert result[0]["type"] == "unknown"

    def test_get_available_modules_skips_files(self, tmp_path):
        """Test that non-directories are skipped"""
        modules_dir = tmp_path / "modules"
        modules_dir.mkdir()

        (modules_dir / "actual-module").mkdir()
        (modules_dir / "not-a-module.txt").write_text("file")

        integration = SentinelIntegration(str(tmp_path))
        result = integration.get_available_modules()

        assert len(result) == 1
        assert result[0]["name"] == "actual-module"

