"""Tests for Bidirectional Sync module"""
import json
from datetime import datetime, timedelta
from pathlib import Path
from unittest.mock import patch, MagicMock

import pytest

from ucts.sync.bidirectional import (
    SyncConfig,
    FileState,
    ProjectState,
    FileDiff,
    ProjectDiff,
    ContextSummary,
    HandoffDocument,
    BidirectionalSync,
    _detect_language,
)


class TestSyncConfig:
    """Tests for SyncConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = SyncConfig()

        assert config.max_context_tokens == 8000
        assert config.include_file_contents is True
        assert config.include_git_history is True
        assert config.compression_level == "medium"
        assert config.diff_context_lines == 3
        assert config.storage_path == ".ucts/sync"

    def test_custom_values(self):
        """Test custom configuration"""
        config = SyncConfig(
            max_context_tokens=4000,
            include_file_contents=False,
            compression_level="high"
        )

        assert config.max_context_tokens == 4000
        assert config.include_file_contents is False
        assert config.compression_level == "high"


class TestFileState:
    """Tests for FileState dataclass"""

    def test_creation(self):
        """Test creating FileState"""
        state = FileState(
            path="/path/to/file.py",
            content_hash="abc123",
            size=1024,
            modified=datetime.now(),
            language="python"
        )

        assert state.path == "/path/to/file.py"
        assert state.content_hash == "abc123"
        assert state.size == 1024
        assert state.language == "python"

    def test_from_file(self, tmp_path):
        """Test creating FileState from file"""
        test_file = tmp_path / "test.py"
        test_file.write_text("print('hello')")

        state = FileState.from_file(test_file)

        assert state.path == str(test_file)
        assert len(state.content_hash) == 16
        assert state.size == len("print('hello')")
        assert state.language == "python"

    def test_from_file_different_types(self, tmp_path):
        """Test FileState for different file types"""
        # JavaScript
        js_file = tmp_path / "test.js"
        js_file.write_text("console.log('hi');")
        assert FileState.from_file(js_file).language == "javascript"

        # TypeScript
        ts_file = tmp_path / "test.ts"
        ts_file.write_text("const x: number = 1;")
        assert FileState.from_file(ts_file).language == "typescript"

        # Unknown
        txt_file = tmp_path / "test.txt"
        txt_file.write_text("plain text")
        assert FileState.from_file(txt_file).language is None


class TestProjectState:
    """Tests for ProjectState dataclass"""

    def test_creation(self):
        """Test creating ProjectState"""
        state = ProjectState(timestamp=datetime.now())

        assert len(state.files) == 0
        assert len(state.dependencies) == 0
        assert state.git_commit is None
        assert state.git_branch is None

    def test_to_dict(self):
        """Test ProjectState serialization"""
        now = datetime(2025, 1, 1, 12, 0, 0)
        state = ProjectState(
            timestamp=now,
            files={
                "test.py": FileState(
                    path="test.py",
                    content_hash="abc123",
                    size=100,
                    modified=now,
                    language="python"
                )
            },
            dependencies={"python": ["requests"]},
            git_commit="abc123def",
            git_branch="main"
        )

        result = state.to_dict()

        assert result["timestamp"] == "2025-01-01T12:00:00"
        assert "test.py" in result["files"]
        assert result["files"]["test.py"]["hash"] == "abc123"
        assert result["git_commit"] == "abc123def"
        assert result["git_branch"] == "main"


class TestFileDiff:
    """Tests for FileDiff dataclass"""

    def test_creation(self):
        """Test creating FileDiff"""
        diff = FileDiff(
            path="test.py",
            change_type="modified",
            old_hash="abc123",
            new_hash="def456",
            diff_lines=["+added line", "-removed line"],
            additions=1,
            deletions=1
        )

        assert diff.path == "test.py"
        assert diff.change_type == "modified"
        assert diff.additions == 1
        assert diff.deletions == 1


class TestProjectDiff:
    """Tests for ProjectDiff dataclass"""

    def test_has_changes_true(self):
        """Test has_changes with changes"""
        diff = ProjectDiff(
            from_state=datetime.now(),
            to_state=datetime.now(),
            files_added=["new.py"]
        )

        assert diff.has_changes is True

    def test_has_changes_false(self):
        """Test has_changes without changes"""
        diff = ProjectDiff(
            from_state=datetime.now(),
            to_state=datetime.now()
        )

        assert diff.has_changes is False

    def test_to_markdown(self):
        """Test ProjectDiff markdown generation"""
        diff = ProjectDiff(
            from_state=datetime(2025, 1, 1),
            to_state=datetime(2025, 1, 2),
            files_added=["new.py"],
            files_modified=["changed.py"],
            files_deleted=["old.py"],
            summary="Test changes",
            dependencies_added={"python": ["new-dep"]},
            dependencies_removed={"python": ["old-dep"]}
        )

        markdown = diff.to_markdown()

        assert "# Project Changes" in markdown
        assert "new.py" in markdown
        assert "changed.py" in markdown
        assert "old.py" in markdown
        assert "Test changes" in markdown
        assert "new-dep" in markdown
        assert "old-dep" in markdown

    def test_to_markdown_with_file_diffs(self):
        """Test markdown with file diffs"""
        diff = ProjectDiff(
            from_state=datetime.now(),
            to_state=datetime.now(),
            files_modified=["test.py"],
            file_diffs=[
                FileDiff(
                    path="test.py",
                    change_type="modified",
                    diff_lines=["+new line", "-old line"],
                    additions=1,
                    deletions=1
                )
            ]
        )

        markdown = diff.to_markdown()

        assert "```diff" in markdown
        assert "+new line" in markdown
        assert "-old line" in markdown


class TestContextSummary:
    """Tests for ContextSummary dataclass"""

    def test_creation(self):
        """Test creating ContextSummary"""
        summary = ContextSummary(
            project_name="test-project",
            description="A test project",
            languages=["python", "javascript"],
            key_files=["main.py", "index.js"],
            architecture_overview="Simple MVC",
            recent_changes="Added auth",
            current_state="Development",
            dependencies={"python": ["flask"]},
            todos=["Add tests"]
        )

        assert summary.project_name == "test-project"
        assert len(summary.languages) == 2

    def test_to_markdown(self):
        """Test ContextSummary markdown generation"""
        summary = ContextSummary(
            project_name="test-project",
            description="A test project",
            languages=["python"],
            key_files=["main.py"],
            architecture_overview="Simple",
            recent_changes="Initial commit",
            current_state="Started",
            dependencies={"python": ["flask", "requests"]},
            todos=["Add tests", "Add docs"]
        )

        markdown = summary.to_markdown()

        assert "# test-project" in markdown
        assert "A test project" in markdown
        assert "python" in markdown
        assert "main.py" in markdown
        assert "flask" in markdown
        assert "Add tests" in markdown

    def test_to_compressed(self):
        """Test ContextSummary compression"""
        summary = ContextSummary(
            project_name="test",
            description="A" * 1000,  # Long description
            languages=["python"],
            key_files=["main.py"] * 20,  # Many files
            architecture_overview="arch",
            recent_changes="changes",
            current_state="state",
            dependencies={},
            todos=["todo"] * 50  # Many todos
        )

        # With high token limit, should return full
        full = summary.to_compressed(max_tokens=100000)
        assert "# test" in full

        # With low limit, should compress
        compressed = summary.to_compressed(max_tokens=100)
        assert len(compressed) < len(summary.to_markdown())


class TestHandoffDocument:
    """Tests for HandoffDocument dataclass"""

    def test_creation(self):
        """Test creating HandoffDocument"""
        context = ContextSummary(
            project_name="test",
            description="desc",
            languages=["python"],
            key_files=["main.py"],
            architecture_overview="arch",
            recent_changes="",
            current_state="state",
            dependencies={},
            todos=[]
        )

        handoff = HandoffDocument(
            session_id="session-123",
            created=datetime.now(),
            context_summary=context,
            conversation_summary="We discussed X",
            key_decisions=["Use Python", "Add tests"],
            pending_tasks=["Write tests"],
            important_context=["Auth system uses JWT"],
            files_discussed=["auth.py"],
            code_snippets=[{"language": "python", "code": "def test(): pass"}]
        )

        assert handoff.session_id == "session-123"
        assert len(handoff.key_decisions) == 2

    def test_to_markdown(self):
        """Test HandoffDocument markdown generation"""
        context = ContextSummary(
            project_name="test",
            description="desc",
            languages=["python"],
            key_files=[],
            architecture_overview="",
            recent_changes="",
            current_state="",
            dependencies={},
            todos=[]
        )

        handoff = HandoffDocument(
            session_id="session-123",
            created=datetime(2025, 1, 1),
            context_summary=context,
            conversation_summary="Discussed features",
            key_decisions=["Decision 1"],
            pending_tasks=["Task 1"],
            important_context=["Context 1"],
            files_discussed=["file.py"],
            code_snippets=[{
                "description": "Helper function",
                "language": "python",
                "code": "def helper(): pass"
            }]
        )

        markdown = handoff.to_markdown()

        assert "# Session Handoff Document" in markdown
        assert "session-123" in markdown
        assert "Discussed features" in markdown
        assert "Decision 1" in markdown
        assert "Task 1" in markdown
        assert "Context 1" in markdown
        assert "file.py" in markdown
        assert "def helper()" in markdown

    def test_to_resume_prompt(self):
        """Test generating resume prompt"""
        context = ContextSummary(
            project_name="test",
            description="desc",
            languages=["python"],
            key_files=[],
            architecture_overview="",
            recent_changes="",
            current_state="",
            dependencies={},
            todos=[]
        )

        handoff = HandoffDocument(
            session_id="session-123",
            created=datetime.now(),
            context_summary=context,
            conversation_summary="Previous work",
            key_decisions=["Used Flask"],
            pending_tasks=["Add auth"],
            important_context=[],
            files_discussed=[],
            code_snippets=[]
        )

        prompt = handoff.to_resume_prompt()

        assert "Continue from a previous session" in prompt
        assert "Previous work" in prompt
        assert "Used Flask" in prompt
        assert "Add auth" in prompt


class TestBidirectionalSync:
    """Tests for BidirectionalSync class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        sync = BidirectionalSync()

        assert sync.config.max_context_tokens == 8000
        assert sync._states == []

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = SyncConfig(max_context_tokens=4000)
        sync = BidirectionalSync(config)

        assert sync.config.max_context_tokens == 4000

    def test_capture_state(self, tmp_path):
        """Test capturing project state"""
        # Create test project
        (tmp_path / "src").mkdir()
        (tmp_path / "src" / "main.py").write_text("print('hello')")
        (tmp_path / "README.md").write_text("# Test")

        sync = BidirectionalSync()
        state = sync.capture_state(str(tmp_path))

        assert len(state.files) >= 2
        assert "src/main.py" in state.files or "src\\main.py" in state.files

    def test_capture_state_ignores_patterns(self, tmp_path):
        """Test that capture ignores common patterns"""
        # Create test files
        (tmp_path / "main.py").write_text("print('hello')")
        (tmp_path / "node_modules").mkdir()
        (tmp_path / "node_modules" / "pkg.js").write_text("module")
        (tmp_path / "__pycache__").mkdir()
        (tmp_path / "__pycache__" / "main.cpython.pyc").write_bytes(b"compiled")

        sync = BidirectionalSync()
        state = sync.capture_state(str(tmp_path))

        # Should only have main.py
        assert any("main.py" in f for f in state.files)
        assert not any("node_modules" in f for f in state.files)
        assert not any("__pycache__" in f for f in state.files)

    def test_compute_diff_no_changes(self, tmp_path):
        """Test computing diff with no changes"""
        (tmp_path / "test.py").write_text("print('hello')")

        sync = BidirectionalSync()
        state1 = sync.capture_state(str(tmp_path))
        state2 = sync.capture_state(str(tmp_path))

        diff = sync.compute_diff(state1, state2)

        assert not diff.has_changes

    def test_compute_diff_with_added_file(self, tmp_path):
        """Test computing diff with added file"""
        (tmp_path / "test.py").write_text("print('hello')")

        sync = BidirectionalSync()
        state1 = sync.capture_state(str(tmp_path))

        # Add a new file
        (tmp_path / "new.py").write_text("print('new')")
        state2 = sync.capture_state(str(tmp_path))

        diff = sync.compute_diff(state1, state2)

        assert diff.has_changes
        assert any("new.py" in f for f in diff.files_added)

    def test_compute_diff_with_modified_file(self, tmp_path):
        """Test computing diff with modified file"""
        test_file = tmp_path / "test.py"
        test_file.write_text("print('hello')")

        sync = BidirectionalSync()
        state1 = sync.capture_state(str(tmp_path))

        # Modify the file
        test_file.write_text("print('modified')")
        state2 = sync.capture_state(str(tmp_path))

        diff = sync.compute_diff(state1, state2)

        assert diff.has_changes
        assert any("test.py" in f for f in diff.files_modified)

    def test_compute_diff_with_deleted_file(self, tmp_path):
        """Test computing diff with deleted file"""
        test_file = tmp_path / "test.py"
        test_file.write_text("print('hello')")
        (tmp_path / "keep.py").write_text("keep")

        sync = BidirectionalSync()
        state1 = sync.capture_state(str(tmp_path))

        # Delete the file
        test_file.unlink()
        state2 = sync.capture_state(str(tmp_path))

        diff = sync.compute_diff(state1, state2)

        assert diff.has_changes
        assert any("test.py" in f for f in diff.files_deleted)


class TestDetectLanguage:
    """Tests for _detect_language helper"""

    def test_python(self):
        """Test Python detection"""
        assert _detect_language(".py") == "python"

    def test_javascript(self):
        """Test JavaScript detection"""
        assert _detect_language(".js") == "javascript"

    def test_typescript(self):
        """Test TypeScript detection"""
        assert _detect_language(".ts") == "typescript"
        assert _detect_language(".tsx") == "typescript"

    def test_rust(self):
        """Test Rust detection"""
        assert _detect_language(".rs") == "rust"

    def test_go(self):
        """Test Go detection"""
        assert _detect_language(".go") == "go"

    def test_unknown(self):
        """Test unknown extension"""
        result = _detect_language(".xyz")
        assert result is None


class TestDependencyParsing:
    """Tests for dependency parsing"""

    def test_parse_python_dependencies(self, tmp_path):
        """Test parsing Python dependencies"""
        requirements = tmp_path / "requirements.txt"
        requirements.write_text("flask==2.0.0\nrequests>=2.25\npytest")

        sync = BidirectionalSync()
        deps = sync._parse_dependencies(tmp_path)

        assert "python" in deps
        assert "flask" in deps["python"]
        assert "requests" in deps["python"]

    def test_parse_node_dependencies(self, tmp_path):
        """Test parsing Node.js dependencies"""
        package_json = tmp_path / "package.json"
        package_json.write_text(json.dumps({
            "dependencies": {
                "express": "^4.17.0",
                "lodash": "^4.17.21"
            }
        }))

        sync = BidirectionalSync()
        deps = sync._parse_dependencies(tmp_path)

        assert "npm" in deps
        assert "express" in deps["npm"]
        assert "lodash" in deps["npm"]

    def test_parse_no_dependencies(self, tmp_path):
        """Test parsing when no dependency files exist"""
        sync = BidirectionalSync()
        deps = sync._parse_dependencies(tmp_path)

        assert deps == {} or len(deps) == 0


class TestGitInfo:
    """Tests for git info extraction"""

    def test_get_git_info_no_repo(self, tmp_path):
        """Test getting git info from non-git directory"""
        sync = BidirectionalSync()
        commit, branch = sync._get_git_info(tmp_path)

        assert commit is None
        assert branch is None

    def test_get_git_info_with_repo(self, tmp_path):
        """Test getting git info from git repository"""
        import subprocess

        # Initialize git repo
        try:
            subprocess.run(["git", "init"], cwd=tmp_path, capture_output=True)
            subprocess.run(["git", "config", "user.email", "test@test.com"], cwd=tmp_path, capture_output=True)
            subprocess.run(["git", "config", "user.name", "Test"], cwd=tmp_path, capture_output=True)

            # Create and commit a file
            (tmp_path / "test.txt").write_text("test")
            subprocess.run(["git", "add", "."], cwd=tmp_path, capture_output=True)
            subprocess.run(["git", "commit", "-m", "Initial"], cwd=tmp_path, capture_output=True)

            sync = BidirectionalSync()
            commit, branch = sync._get_git_info(tmp_path)

            assert commit is not None
            assert branch is not None
        except FileNotFoundError:
            pytest.skip("Git not available")
