"""
Tests for Templates Module

Tests the templates module including:
- base.py: TemplateFile, TemplateConfig, ProjectTemplate, TemplateRegistry
- python_templates.py: FastAPITemplate, FlaskTemplate, DjangoTemplate, CLITemplate
- node_templates.py: ExpressTemplate, NextJSTemplate, ViteReactTemplate
"""

import json
import pytest
from pathlib import Path
from typing import List

from ucts.templates.base import (
    TemplateFile,
    TemplateConfig,
    ProjectTemplate,
    TemplateRegistry,
)
from ucts.templates.python_templates import (
    FastAPITemplate,
    FlaskTemplate,
    DjangoTemplate,
    CLITemplate,
)
from ucts.templates.node_templates import (
    ExpressTemplate,
    NextJSTemplate,
    ViteReactTemplate,
)


# ============================================================================
# TemplateFile Tests
# ============================================================================

class TestTemplateFile:
    """Tests for TemplateFile dataclass"""

    def test_creation(self):
        """Test basic file creation"""
        tf = TemplateFile(path="main.py", content="print('hello')")
        assert tf.path == "main.py"
        assert tf.content == "print('hello')"
        assert tf.executable is False

    def test_executable_flag(self):
        """Test executable flag"""
        tf = TemplateFile(path="script.sh", content="#!/bin/bash", executable=True)
        assert tf.executable is True

    def test_default_not_executable(self):
        """Test files are not executable by default"""
        tf = TemplateFile(path="app.py", content="")
        assert tf.executable is False

    def test_multiline_content(self):
        """Test multiline content"""
        content = """def main():
    print('hello')
    return 0
"""
        tf = TemplateFile(path="main.py", content=content)
        assert "def main():" in tf.content
        assert "return 0" in tf.content


# ============================================================================
# TemplateConfig Tests
# ============================================================================

class TestTemplateConfig:
    """Tests for TemplateConfig dataclass"""

    def test_minimal_config(self):
        """Test minimal configuration"""
        config = TemplateConfig(project_name="my-app")
        assert config.project_name == "my-app"
        assert config.author == "Author"
        assert config.version == "0.1.0"

    def test_full_config(self):
        """Test full configuration"""
        config = TemplateConfig(
            project_name="my-project",
            author="Jane Doe",
            description="A cool project",
            version="1.0.0",
            extras={"license": "MIT"}
        )
        assert config.project_name == "my-project"
        assert config.author == "Jane Doe"
        assert config.description == "A cool project"
        assert config.version == "1.0.0"
        assert config.extras["license"] == "MIT"

    def test_default_description(self):
        """Test default empty description"""
        config = TemplateConfig(project_name="test")
        assert config.description == ""

    def test_extras_default_empty(self):
        """Test extras defaults to empty dict"""
        config = TemplateConfig(project_name="test")
        assert config.extras == {}


# ============================================================================
# TemplateRegistry Tests
# ============================================================================

class TestTemplateRegistry:
    """Tests for TemplateRegistry class"""

    def test_empty_registry(self):
        """Test empty registry"""
        registry = TemplateRegistry()
        assert registry.list_all() == []

    def test_register_template(self):
        """Test registering a template"""
        registry = TemplateRegistry()
        template = FastAPITemplate()

        registry.register(template)

        assert registry.get("fastapi") is not None
        assert registry.get("fastapi").name == "fastapi"

    def test_get_nonexistent(self):
        """Test getting nonexistent template"""
        registry = TemplateRegistry()
        assert registry.get("nonexistent") is None

    def test_list_all(self):
        """Test listing all templates"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(FlaskTemplate())

        templates = registry.list_all()

        assert len(templates) == 2
        names = [t["name"] for t in templates]
        assert "fastapi" in names
        assert "flask" in names

    def test_list_all_structure(self):
        """Test list_all returns proper structure"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())

        templates = registry.list_all()

        assert len(templates) == 1
        t = templates[0]
        assert "name" in t
        assert "display_name" in t
        assert "description" in t
        assert "language" in t
        assert "tags" in t

    def test_search_by_name(self):
        """Test searching by name"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(FlaskTemplate())

        results = registry.search("fast")

        assert len(results) == 1
        assert results[0].name == "fastapi"

    def test_search_by_description(self):
        """Test searching by description"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(FlaskTemplate())

        results = registry.search("async")

        assert len(results) == 1
        assert results[0].name == "fastapi"

    def test_search_by_tag(self):
        """Test searching by tag"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(ExpressTemplate())

        results = registry.search("rest")

        assert len(results) == 2

    def test_search_case_insensitive(self):
        """Test search is case insensitive"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())

        results = registry.search("FASTAPI")

        assert len(results) == 1

    def test_by_language_python(self):
        """Test filtering by Python language"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(ExpressTemplate())

        results = registry.by_language("python")

        assert len(results) == 1
        assert results[0].name == "fastapi"

    def test_by_language_typescript(self):
        """Test filtering by TypeScript language"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(ExpressTemplate())
        registry.register(NextJSTemplate())

        results = registry.by_language("typescript")

        assert len(results) == 2

    def test_by_language_case_insensitive(self):
        """Test language filter is case insensitive"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())

        results = registry.by_language("PYTHON")

        assert len(results) == 1


# ============================================================================
# FastAPITemplate Tests
# ============================================================================

class TestFastAPITemplate:
    """Tests for FastAPITemplate"""

    def test_properties(self):
        """Test template properties"""
        template = FastAPITemplate()
        assert template.name == "fastapi"
        assert template.display_name == "FastAPI"
        assert template.language == "python"
        assert "api" in template.tags
        assert "async" in template.tags

    def test_dependencies(self):
        """Test template dependencies"""
        template = FastAPITemplate()
        deps = template.dependencies

        assert "python" in deps
        assert any("fastapi" in d for d in deps["python"])
        assert any("uvicorn" in d for d in deps["python"])

    def test_dev_dependencies(self):
        """Test template dev dependencies"""
        template = FastAPITemplate()
        deps = template.dev_dependencies

        assert "python" in deps
        assert any("pytest" in d for d in deps["python"])

    def test_generate_files(self):
        """Test generating files"""
        template = FastAPITemplate()
        config = TemplateConfig(project_name="my-api", version="1.0.0")

        files = template.generate(config)

        file_paths = [f.path for f in files]
        assert "main.py" in file_paths
        assert "app/__init__.py" in file_paths
        assert "app/config.py" in file_paths
        assert "requirements.txt" in file_paths

    def test_generate_main_content(self):
        """Test main.py content"""
        template = FastAPITemplate()
        config = TemplateConfig(
            project_name="my-api",
            description="My API",
            version="2.0.0"
        )

        files = template.generate(config)
        main = next(f for f in files if f.path == "main.py")

        assert "my-api" in main.content
        assert "My API" in main.content
        assert "2.0.0" in main.content
        assert "FastAPI" in main.content

    def test_generate_tests(self):
        """Test test file generation"""
        template = FastAPITemplate()
        config = TemplateConfig(project_name="my-api")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "tests/test_main.py" in file_paths
        assert "tests/conftest.py" in file_paths

    def test_write_to_disk(self, tmp_path):
        """Test writing files to disk"""
        template = FastAPITemplate()
        config = TemplateConfig(project_name="my-api")

        created = template.write_to(str(tmp_path), config)

        assert len(created) > 0
        assert (tmp_path / "main.py").exists()
        assert (tmp_path / "app" / "config.py").exists()


# ============================================================================
# FlaskTemplate Tests
# ============================================================================

class TestFlaskTemplate:
    """Tests for FlaskTemplate"""

    def test_properties(self):
        """Test template properties"""
        template = FlaskTemplate()
        assert template.name == "flask"
        assert template.language == "python"
        assert "flask" in template.tags

    def test_dependencies(self):
        """Test dependencies include Flask"""
        template = FlaskTemplate()
        deps = template.dependencies

        assert any("flask" in d for d in deps["python"])

    def test_generate_files(self):
        """Test generating files"""
        template = FlaskTemplate()
        config = TemplateConfig(project_name="my-flask-app")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "app.py" in file_paths
        assert "routes.py" in file_paths
        assert "requirements.txt" in file_paths

    def test_generate_app_content(self):
        """Test app.py content"""
        template = FlaskTemplate()
        config = TemplateConfig(project_name="my-flask-app")

        files = template.generate(config)
        app = next(f for f in files if f.path == "app.py")

        assert "Flask" in app.content
        assert "create_app" in app.content


# ============================================================================
# DjangoTemplate Tests
# ============================================================================

class TestDjangoTemplate:
    """Tests for DjangoTemplate"""

    def test_properties(self):
        """Test template properties"""
        template = DjangoTemplate()
        assert template.name == "django"
        assert template.language == "python"
        assert "django" in template.tags
        assert "orm" in template.tags

    def test_dependencies(self):
        """Test dependencies"""
        template = DjangoTemplate()
        deps = template.dependencies

        assert any("django" in d for d in deps["python"])
        assert any("djangorestframework" in d for d in deps["python"])

    def test_generate_files(self):
        """Test generating files"""
        template = DjangoTemplate()
        config = TemplateConfig(project_name="my-django-app")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "manage.py" in file_paths
        # Django uses underscores in project name
        assert "my_django_app/settings.py" in file_paths
        assert "my_django_app/urls.py" in file_paths

    def test_manage_is_executable(self):
        """Test manage.py is executable"""
        template = DjangoTemplate()
        config = TemplateConfig(project_name="myapp")

        files = template.generate(config)
        manage = next(f for f in files if f.path == "manage.py")

        assert manage.executable is True

    def test_settings_content(self):
        """Test settings.py content"""
        template = DjangoTemplate()
        config = TemplateConfig(project_name="myapp")

        files = template.generate(config)
        settings = next(f for f in files if f.path == "myapp/settings.py")

        assert "INSTALLED_APPS" in settings.content
        assert "rest_framework" in settings.content
        assert "DATABASES" in settings.content


# ============================================================================
# CLITemplate Tests
# ============================================================================

class TestCLITemplate:
    """Tests for CLITemplate"""

    def test_properties(self):
        """Test template properties"""
        template = CLITemplate()
        assert template.name == "cli"
        assert template.display_name == "Python CLI"
        assert template.language == "python"
        assert "cli" in template.tags
        assert "command-line" in template.tags

    def test_dependencies(self):
        """Test dependencies include Click"""
        template = CLITemplate()
        deps = template.dependencies

        assert any("click" in d for d in deps["python"])
        assert any("rich" in d for d in deps["python"])

    def test_generate_files(self):
        """Test generating files"""
        template = CLITemplate()
        config = TemplateConfig(project_name="my-cli")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        # Uses underscored project name
        assert "my_cli/cli.py" in file_paths
        assert "my_cli/__init__.py" in file_paths
        assert "my_cli/__main__.py" in file_paths
        assert "pyproject.toml" in file_paths

    def test_cli_content(self):
        """Test CLI file content"""
        template = CLITemplate()
        config = TemplateConfig(
            project_name="my-cli",
            description="My CLI tool",
            version="1.0.0"
        )

        files = template.generate(config)
        cli = next(f for f in files if f.path == "my_cli/cli.py")

        assert "@click.group()" in cli.content
        assert "@cli.command()" in cli.content  # Uses group.command() syntax
        assert "1.0.0" in cli.content

    def test_pyproject_entry_point(self):
        """Test pyproject.toml has entry point"""
        template = CLITemplate()
        config = TemplateConfig(project_name="my-cli")

        files = template.generate(config)
        pyproject = next(f for f in files if f.path == "pyproject.toml")

        assert "[project.scripts]" in pyproject.content
        assert "my-cli" in pyproject.content


# ============================================================================
# ExpressTemplate Tests
# ============================================================================

class TestExpressTemplate:
    """Tests for ExpressTemplate"""

    def test_properties(self):
        """Test template properties"""
        template = ExpressTemplate()
        assert template.name == "express"
        assert template.display_name == "Express.js"
        assert template.language == "typescript"
        assert "express" in template.tags
        assert "node" in template.tags

    def test_dependencies(self):
        """Test dependencies"""
        template = ExpressTemplate()
        deps = template.dependencies

        assert "express" in deps["node"]
        assert "cors" in deps["node"]
        assert "helmet" in deps["node"]

    def test_dev_dependencies(self):
        """Test dev dependencies"""
        template = ExpressTemplate()
        deps = template.dev_dependencies

        assert "typescript" in deps["node"]
        assert any("jest" in d for d in deps["node"])

    def test_generate_files(self):
        """Test generating files"""
        template = ExpressTemplate()
        config = TemplateConfig(project_name="my-express-api")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "package.json" in file_paths
        assert "tsconfig.json" in file_paths
        assert "src/index.ts" in file_paths
        assert "src/routes/index.ts" in file_paths

    def test_package_json_valid(self):
        """Test package.json is valid JSON"""
        template = ExpressTemplate()
        config = TemplateConfig(project_name="my-api", version="1.0.0")

        files = template.generate(config)
        pkg_file = next(f for f in files if f.path == "package.json")
        pkg = json.loads(pkg_file.content)

        assert pkg["name"] == "my-api"
        assert pkg["version"] == "1.0.0"
        assert "express" in pkg["dependencies"]

    def test_tsconfig_valid(self):
        """Test tsconfig.json is valid JSON"""
        template = ExpressTemplate()
        config = TemplateConfig(project_name="my-api")

        files = template.generate(config)
        tsconfig_file = next(f for f in files if f.path == "tsconfig.json")
        tsconfig = json.loads(tsconfig_file.content)

        assert tsconfig["compilerOptions"]["strict"] is True


# ============================================================================
# NextJSTemplate Tests
# ============================================================================

class TestNextJSTemplate:
    """Tests for NextJSTemplate"""

    def test_properties(self):
        """Test template properties"""
        template = NextJSTemplate()
        assert template.name == "nextjs"
        assert template.display_name == "Next.js"
        assert template.language == "typescript"
        assert "react" in template.tags
        assert "ssr" in template.tags

    def test_generate_files(self):
        """Test generating files"""
        template = NextJSTemplate()
        config = TemplateConfig(project_name="my-nextjs-app")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "package.json" in file_paths
        assert "tsconfig.json" in file_paths
        assert "next.config.js" in file_paths
        assert "app/layout.tsx" in file_paths
        assert "app/page.tsx" in file_paths

    def test_package_json_has_next(self):
        """Test package.json has Next.js"""
        template = NextJSTemplate()
        config = TemplateConfig(project_name="my-app")

        files = template.generate(config)
        pkg_file = next(f for f in files if f.path == "package.json")
        pkg = json.loads(pkg_file.content)

        assert "next" in pkg["dependencies"]
        assert "react" in pkg["dependencies"]

    def test_layout_has_metadata(self):
        """Test layout.tsx has metadata"""
        template = NextJSTemplate()
        config = TemplateConfig(
            project_name="my-app",
            description="My Next.js app"
        )

        files = template.generate(config)
        layout = next(f for f in files if f.path == "app/layout.tsx")

        assert "Metadata" in layout.content
        assert "my-app" in layout.content


# ============================================================================
# ViteReactTemplate Tests
# ============================================================================

class TestViteReactTemplate:
    """Tests for ViteReactTemplate"""

    def test_properties(self):
        """Test template properties"""
        template = ViteReactTemplate()
        assert template.name == "vite-react"
        assert template.display_name == "Vite + React"
        assert template.language == "typescript"
        assert "vite" in template.tags
        assert "react" in template.tags

    def test_generate_files(self):
        """Test generating files"""
        template = ViteReactTemplate()
        config = TemplateConfig(project_name="my-vite-app")

        files = template.generate(config)
        file_paths = [f.path for f in files]

        assert "package.json" in file_paths
        assert "vite.config.ts" in file_paths
        assert "tsconfig.json" in file_paths
        assert "index.html" in file_paths
        assert "src/main.tsx" in file_paths
        assert "src/App.tsx" in file_paths

    def test_package_json_has_vite(self):
        """Test package.json has Vite"""
        template = ViteReactTemplate()
        config = TemplateConfig(project_name="my-app")

        files = template.generate(config)
        pkg_file = next(f for f in files if f.path == "package.json")
        pkg = json.loads(pkg_file.content)

        assert "vite" in pkg["devDependencies"]
        assert "react" in pkg["dependencies"]

    def test_index_html_content(self):
        """Test index.html content"""
        template = ViteReactTemplate()
        config = TemplateConfig(project_name="my-app")

        files = template.generate(config)
        html = next(f for f in files if f.path == "index.html")

        assert "<title>my-app</title>" in html.content
        assert 'src="/src/main.tsx"' in html.content

    def test_app_tsx_content(self):
        """Test App.tsx content"""
        template = ViteReactTemplate()
        config = TemplateConfig(
            project_name="my-app",
            description="My Vite app"
        )

        files = template.generate(config)
        app = next(f for f in files if f.path == "src/App.tsx")

        assert "useState" in app.content
        assert "my-app" in app.content


# ============================================================================
# ProjectTemplate Abstract Base Class Tests
# ============================================================================

class TestProjectTemplateBase:
    """Tests for ProjectTemplate base class behavior"""

    def test_default_tags_empty(self):
        """Test default tags returns empty list"""
        template = FastAPITemplate()
        # FastAPI has tags, but testing the default behavior
        assert isinstance(template.tags, list)

    def test_default_dependencies_empty(self):
        """Test templates without dependencies return empty dict"""
        # FlaskTemplate has no dev_dependencies defined
        template = FlaskTemplate()
        # It only has dependencies, checking default behavior
        assert isinstance(template.dependencies, dict)

    def test_write_to_creates_directories(self, tmp_path):
        """Test write_to creates nested directories"""
        template = FastAPITemplate()
        config = TemplateConfig(project_name="my-app")

        template.write_to(str(tmp_path), config)

        assert (tmp_path / "app").is_dir()
        assert (tmp_path / "tests").is_dir()

    def test_write_to_returns_paths(self, tmp_path):
        """Test write_to returns list of created file paths"""
        template = FlaskTemplate()
        config = TemplateConfig(project_name="my-app")

        created = template.write_to(str(tmp_path), config)

        assert len(created) > 0
        for path in created:
            assert Path(path).exists()

    def test_write_to_creates_output_dir(self, tmp_path):
        """Test write_to creates output directory if not exists"""
        template = FlaskTemplate()
        config = TemplateConfig(project_name="my-app")
        output_path = tmp_path / "new" / "nested" / "dir"

        template.write_to(str(output_path), config)

        assert output_path.exists()


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration tests for templates"""

    def test_register_all_templates(self):
        """Test registering all templates"""
        registry = TemplateRegistry()

        # Register all templates
        registry.register(FastAPITemplate())
        registry.register(FlaskTemplate())
        registry.register(DjangoTemplate())
        registry.register(CLITemplate())
        registry.register(ExpressTemplate())
        registry.register(NextJSTemplate())
        registry.register(ViteReactTemplate())

        templates = registry.list_all()
        assert len(templates) == 7

    def test_generate_full_project(self, tmp_path):
        """Test generating a full project from template"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())

        template = registry.get("fastapi")
        config = TemplateConfig(
            project_name="production-api",
            author="Developer",
            description="A production-ready API",
            version="1.0.0"
        )

        created = template.write_to(str(tmp_path), config)

        # Verify structure
        assert (tmp_path / "main.py").exists()
        assert (tmp_path / "app" / "config.py").exists()
        assert (tmp_path / "tests" / "test_main.py").exists()
        assert (tmp_path / "requirements.txt").exists()
        assert (tmp_path / "README.md").exists()

        # Verify content
        main_content = (tmp_path / "main.py").read_text()
        assert "production-api" in main_content
        assert "A production-ready API" in main_content

    def test_search_and_generate(self, tmp_path):
        """Test searching for template and generating"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(ExpressTemplate())

        # Search for REST API templates
        results = registry.search("rest")
        assert len(results) >= 1

        # Generate from first result
        template = results[0]
        config = TemplateConfig(project_name="my-api")
        files = template.generate(config)

        assert len(files) > 0

    def test_language_filter_and_generate(self, tmp_path):
        """Test filtering by language and generating"""
        registry = TemplateRegistry()
        registry.register(FastAPITemplate())
        registry.register(FlaskTemplate())
        registry.register(ExpressTemplate())

        # Get Python templates
        python_templates = registry.by_language("python")
        assert len(python_templates) == 2

        # Generate from first Python template
        template = python_templates[0]
        config = TemplateConfig(project_name="python-app")
        files = template.generate(config)

        file_paths = [f.path for f in files]
        # Python projects have requirements.txt
        assert "requirements.txt" in file_paths
