"""
Tests for Test Generation Module

Tests the generators/tests.py module including:
- TestConfig dataclass
- PytestGenerator class
- JestGenerator class
- VitestGenerator class
- TestGenerator unified class
"""

import pytest
from pathlib import Path
from dataclasses import dataclass, field
from typing import Dict, List

from ucts.generators.tests import (
    TestConfig,
    PytestGenerator,
    JestGenerator,
    VitestGenerator,
    TestGenerator,
)


# ============================================================================
# Mock ProjectStructure
# ============================================================================

@dataclass
class MockProjectStructure:
    """Mock project structure for testing"""
    name: str = "test-project"
    languages: List[str] = field(default_factory=lambda: ["python"])
    files: Dict[str, str] = field(default_factory=dict)
    directories: List[str] = field(default_factory=list)
    dependencies: Dict[str, List[str]] = field(default_factory=dict)


# ============================================================================
# TestConfig Tests
# ============================================================================

class TestTestConfig:
    """Tests for TestConfig dataclass"""

    def test_default_values(self):
        """Test default configuration values"""
        config = TestConfig()
        assert config.framework == "auto"
        assert config.coverage_target == 80
        assert config.include_mocks is True
        assert config.include_fixtures is True
        assert config.include_integration is True

    def test_custom_values(self):
        """Test custom configuration values"""
        config = TestConfig(
            framework="pytest",
            coverage_target=90,
            include_mocks=False,
            include_fixtures=False,
            include_integration=False
        )
        assert config.framework == "pytest"
        assert config.coverage_target == 90
        assert config.include_mocks is False

    def test_coverage_target_value(self):
        """Test coverage target can be set to any value"""
        config = TestConfig(coverage_target=100)
        assert config.coverage_target == 100

    def test_framework_values(self):
        """Test framework can be set to various values"""
        for framework in ["pytest", "jest", "vitest", "auto"]:
            config = TestConfig(framework=framework)
            assert config.framework == framework


# ============================================================================
# PytestGenerator Tests
# ============================================================================

class TestPytestGenerator:
    """Tests for PytestGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = PytestGenerator()
        assert gen.config is not None
        assert gen.config.framework == "auto"

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = TestConfig(coverage_target=90)
        gen = PytestGenerator(config)
        assert gen.config.coverage_target == 90

    def test_generate_creates_conftest(self, tmp_path):
        """Test generate creates conftest.py"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"app.py": "def main(): pass"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/conftest.py" in files
        assert (tmp_path / "tests" / "conftest.py").exists()

    def test_generate_creates_pytest_ini(self, tmp_path):
        """Test generate creates pytest.ini"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"app.py": "def main(): pass"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "pytest.ini" in files
        assert (tmp_path / "pytest.ini").exists()

    def test_generate_creates_test_files(self, tmp_path):
        """Test generate creates test files for source files"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={
                "utils.py": "def helper(): pass\ndef another(): pass",
                "models.py": "class User: pass"
            }
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/test_utils.py" in files
        assert "tests/test_models.py" in files

    def test_generate_skips_test_files(self, tmp_path):
        """Test generate skips existing test files"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={
                "app.py": "def main(): pass",
                "test_app.py": "def test_main(): pass"  # Should be skipped
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Should not create test_test_app.py
        assert "tests/test_test_app.py" not in files

    def test_conftest_detects_fastapi(self, tmp_path):
        """Test conftest includes FastAPI fixtures"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"main.py": "from fastapi import FastAPI\napp = FastAPI()"}
        )

        files = gen.generate(structure, str(tmp_path))
        conftest = files["tests/conftest.py"]

        assert "TestClient" in conftest
        assert "fastapi" in conftest.lower()

    def test_conftest_detects_flask(self, tmp_path):
        """Test conftest includes Flask fixtures"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"main.py": "from flask import Flask\napp = Flask(__name__)"}
        )

        files = gen.generate(structure, str(tmp_path))
        conftest = files["tests/conftest.py"]

        assert "test_client" in conftest
        assert "flask" in conftest.lower()

    def test_conftest_detects_database(self, tmp_path):
        """Test conftest includes database fixtures"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"models.py": "from sqlalchemy import Column, Integer"}
        )

        files = gen.generate(structure, str(tmp_path))
        conftest = files["tests/conftest.py"]

        assert "db_session" in conftest
        assert "SessionLocal" in conftest

    def test_test_file_extracts_functions(self, tmp_path):
        """Test test file generation extracts functions"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"utils.py": """
def process_data(data):
    return data

def validate_input(value):
    return True
"""}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["tests/test_utils.py"]

        assert "process_data" in test_content
        assert "validate_input" in test_content
        assert "test_process_data" in test_content

    def test_test_file_extracts_classes(self, tmp_path):
        """Test test file generation extracts classes"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"models.py": """
class User:
    pass

class Product:
    pass
"""}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["tests/test_models.py"]

        assert "TestUser" in test_content
        assert "TestProduct" in test_content

    def test_test_file_skips_private_functions(self, tmp_path):
        """Test test file skips private functions"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"utils.py": """
def public_func(): pass
def _private_func(): pass
def __dunder_func(): pass
"""}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["tests/test_utils.py"]

        assert "public_func" in test_content
        assert "_private_func" not in test_content
        assert "__dunder_func" not in test_content

    def test_pytest_ini_coverage_target(self, tmp_path):
        """Test pytest.ini uses config coverage target"""
        config = TestConfig(coverage_target=95)
        gen = PytestGenerator(config)
        structure = MockProjectStructure(files={"app.py": "def f(): pass"})

        files = gen.generate(structure, str(tmp_path))
        ini_content = files["pytest.ini"]

        assert "--cov-fail-under=95" in ini_content

    def test_empty_file_returns_none(self, tmp_path):
        """Test empty file doesn't generate test"""
        gen = PytestGenerator()
        structure = MockProjectStructure(
            files={"empty.py": "# Just a comment"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/test_empty.py" not in files


# ============================================================================
# JestGenerator Tests
# ============================================================================

class TestJestGenerator:
    """Tests for JestGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = JestGenerator()
        assert gen.config is not None

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = TestConfig(include_mocks=False)
        gen = JestGenerator(config)
        assert gen.config.include_mocks is False

    def test_generate_creates_jest_config(self, tmp_path):
        """Test generate creates jest.config.js"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function helper() {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "jest.config.js" in files

    def test_generate_typescript_jest_config(self, tmp_path):
        """Test generate creates jest.config.ts for TypeScript"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "jest.config.ts" in files
        config = files["jest.config.ts"]
        assert "ts-jest" in config

    def test_generate_creates_setup(self, tmp_path):
        """Test generate creates jest.setup.js"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function helper() {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "jest.setup.js" in files

    def test_generate_creates_test_files(self, tmp_path):
        """Test generate creates test files"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={
                "utils.js": "export function helper() {}\nexport const value = 1;"
            }
        )

        files = gen.generate(structure, str(tmp_path))

        assert "__tests__/utils.test.js" in files

    def test_generate_typescript_test_files(self, tmp_path):
        """Test generate creates .test.ts files for TypeScript"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "__tests__/utils.test.ts" in files

    def test_generate_skips_test_files(self, tmp_path):
        """Test generate skips existing test files"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={
                "utils.js": "export function helper() {}",
                "utils.test.js": "// existing tests"
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Should not create test for test file
        assert "__tests__/utils.test.test.js" not in files

    def test_test_file_extracts_exports(self, tmp_path):
        """Test test file extracts exports"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": """
export function processData(data) { return data; }
export const CONFIG = {};
export class Handler {}
"""}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["__tests__/utils.test.js"]

        assert "processData" in test_content
        assert "CONFIG" in test_content
        assert "Handler" in test_content

    def test_test_file_handles_commonjs(self, tmp_path):
        """Test test file handles CommonJS exports"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": """
module.exports = { helper, process, validate };
"""}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["__tests__/utils.test.js"]

        assert "helper" in test_content

    def test_jest_config_coverage_thresholds(self, tmp_path):
        """Test jest config includes coverage thresholds"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function f() {}"}
        )

        files = gen.generate(structure, str(tmp_path))
        config = files["jest.config.js"]

        assert "coverageThreshold" in config
        assert "branches: 80" in config
        assert "lines: 80" in config

    def test_setup_includes_fetch_mock(self, tmp_path):
        """Test setup includes fetch mock"""
        gen = JestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"api.js": "export function fetch() {}"}
        )

        files = gen.generate(structure, str(tmp_path))
        setup = files["jest.setup.js"]

        assert "global.fetch = jest.fn" in setup


# ============================================================================
# VitestGenerator Tests
# ============================================================================

class TestVitestGenerator:
    """Tests for VitestGenerator class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = VitestGenerator()
        assert gen.config is not None

    def test_generate_creates_vitest_config(self, tmp_path):
        """Test generate creates vitest.config.ts"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "vitest.config.ts" in files
        config = files["vitest.config.ts"]
        assert "defineConfig" in config

    def test_generate_creates_setup(self, tmp_path):
        """Test generate creates tests/setup.ts"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/setup.ts" in files

    def test_generate_creates_test_files(self, tmp_path):
        """Test generate creates test files"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/utils.test.ts" in files

    def test_generate_handles_js_files(self, tmp_path):
        """Test generate handles JavaScript files"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function helper() {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/utils.test.ts" in files

    def test_generate_handles_jsx_tsx(self, tmp_path):
        """Test generate handles JSX/TSX files"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"Component.tsx": "export function Button() { return <button/>; }"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/Component.test.ts" in files

    def test_generate_skips_test_files(self, tmp_path):
        """Test generate skips existing test/spec files"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={
                "main.test.ts": "// existing test file - should be skipped",
                "other.spec.ts": "// existing spec file - should be skipped"
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Test/spec files themselves should not generate additional tests
        assert "tests/main.test.test.ts" not in files
        assert "tests/other.spec.test.ts" not in files

    def test_test_file_imports_vitest(self, tmp_path):
        """Test test file imports from vitest"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))
        test_content = files["tests/utils.test.ts"]

        assert "import { describe, it, expect, vi" in test_content

    def test_vitest_config_coverage(self, tmp_path):
        """Test vitest config includes coverage settings"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function f(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path))
        config = files["vitest.config.ts"]

        assert "coverage" in config
        assert "thresholds" in config
        assert "lines: 80" in config

    def test_setup_uses_vi(self, tmp_path):
        """Test setup uses vi from vitest"""
        gen = VitestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"api.ts": "export function fetch() {}"}
        )

        files = gen.generate(structure, str(tmp_path))
        setup = files["tests/setup.ts"]

        assert "import { vi } from 'vitest'" in setup
        assert "vi.fn" in setup


# ============================================================================
# TestGenerator Unified Tests
# ============================================================================

class TestTestGenerator:
    """Tests for TestGenerator unified class"""

    def test_init_default_config(self):
        """Test initialization with default config"""
        gen = TestGenerator()
        assert gen.config is not None
        assert gen.config.framework == "auto"

    def test_init_custom_config(self):
        """Test initialization with custom config"""
        config = TestConfig(framework="pytest")
        gen = TestGenerator(config)
        assert gen.config.framework == "pytest"

    def test_detect_framework_python(self):
        """Test framework detection for Python"""
        gen = TestGenerator()
        structure = MockProjectStructure(languages=["python"])

        framework = gen._detect_framework(structure)

        assert framework == "pytest"

    def test_detect_framework_javascript(self):
        """Test framework detection for JavaScript defaults to vitest"""
        gen = TestGenerator()
        structure = MockProjectStructure(languages=["javascript"])

        framework = gen._detect_framework(structure)

        assert framework == "vitest"

    def test_detect_framework_typescript(self):
        """Test framework detection for TypeScript defaults to vitest"""
        gen = TestGenerator()
        structure = MockProjectStructure(languages=["typescript"])

        framework = gen._detect_framework(structure)

        assert framework == "vitest"

    def test_detect_framework_existing_jest(self):
        """Test framework detection with existing jest.config"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"jest.config.js": "module.exports = {};"}
        )

        framework = gen._detect_framework(structure)

        assert framework == "jest"

    def test_detect_framework_existing_vitest(self):
        """Test framework detection with existing vitest.config"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"vitest.config.ts": "export default {};"}
        )

        framework = gen._detect_framework(structure)

        assert framework == "vitest"

    def test_generate_auto_python(self, tmp_path):
        """Test auto-generate for Python project"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            files={"app.py": "def main(): pass"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "tests/conftest.py" in files
        assert "pytest.ini" in files

    def test_generate_auto_javascript(self, tmp_path):
        """Test auto-generate for JavaScript project"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function helper() {}"}
        )

        files = gen.generate(structure, str(tmp_path))

        assert "vitest.config.ts" in files

    def test_generate_explicit_pytest(self, tmp_path):
        """Test explicit pytest framework"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            files={"app.py": "def main(): pass"}
        )

        files = gen.generate(structure, str(tmp_path), framework="pytest")

        assert "pytest.ini" in files

    def test_generate_explicit_jest(self, tmp_path):
        """Test explicit jest framework"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["javascript"],
            files={"utils.js": "export function helper() {}"}
        )

        files = gen.generate(structure, str(tmp_path), framework="jest")

        assert "jest.config.js" in files

    def test_generate_explicit_vitest(self, tmp_path):
        """Test explicit vitest framework"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={"utils.ts": "export function helper(): void {}"}
        )

        files = gen.generate(structure, str(tmp_path), framework="vitest")

        assert "vitest.config.ts" in files

    def test_generate_unknown_framework_raises(self, tmp_path):
        """Test unknown framework raises error"""
        gen = TestGenerator()
        structure = MockProjectStructure()

        with pytest.raises(ValueError, match="Unknown framework"):
            gen.generate(structure, str(tmp_path), framework="unknown")

    def test_list_frameworks(self):
        """Test list_frameworks returns all frameworks"""
        frameworks = TestGenerator.list_frameworks()

        assert len(frameworks) == 3
        names = [f["name"] for f in frameworks]
        assert "pytest" in names
        assert "jest" in names
        assert "vitest" in names

    def test_list_frameworks_structure(self):
        """Test list_frameworks returns proper structure"""
        frameworks = TestGenerator.list_frameworks()

        for fw in frameworks:
            assert "name" in fw
            assert "language" in fw
            assert "description" in fw


# ============================================================================
# Integration Tests
# ============================================================================

class TestIntegration:
    """Integration tests for test generation"""

    def test_full_python_project(self, tmp_path):
        """Test generating tests for full Python project"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["python"],
            files={
                "main.py": """
from fastapi import FastAPI
from sqlalchemy import Column, Integer

app = FastAPI()

def get_items():
    return []
""",
                "models.py": """
class User:
    def __init__(self, name):
        self.name = name

class Product:
    def __init__(self, price):
        self.price = price
""",
                "utils.py": """
def validate_email(email):
    return '@' in email

def format_date(date):
    return str(date)
"""
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Check all expected files
        assert "tests/conftest.py" in files
        assert "pytest.ini" in files
        assert "tests/test_main.py" in files
        assert "tests/test_models.py" in files
        assert "tests/test_utils.py" in files

        # Check conftest has FastAPI and DB fixtures
        conftest = files["tests/conftest.py"]
        assert "TestClient" in conftest
        assert "db_session" in conftest

        # Check files written to disk
        assert (tmp_path / "tests" / "conftest.py").exists()
        assert (tmp_path / "pytest.ini").exists()

    def test_full_typescript_project(self, tmp_path):
        """Test generating tests for full TypeScript project"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["typescript"],
            files={
                "api.ts": """
export function fetchData(url: string): Promise<any> {
    return fetch(url).then(r => r.json());
}

export function postData(url: string, data: any): Promise<any> {
    return fetch(url, { method: 'POST', body: JSON.stringify(data) });
}
""",
                "utils.ts": """
export function formatCurrency(value: number): string {
    return '$' + value.toFixed(2);
}

export const DEFAULT_OPTIONS = { timeout: 5000 };
"""
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Check vitest config
        assert "vitest.config.ts" in files
        assert "tests/setup.ts" in files

        # Check test files
        assert "tests/api.test.ts" in files
        assert "tests/utils.test.ts" in files

        # Check test content
        api_tests = files["tests/api.test.ts"]
        assert "fetchData" in api_tests
        assert "postData" in api_tests
        assert "describe" in api_tests

    def test_mixed_language_project(self, tmp_path):
        """Test generating tests prefers primary language"""
        gen = TestGenerator()
        structure = MockProjectStructure(
            languages=["python", "javascript"],  # Python first
            files={
                "app.py": "def main(): pass",
                "script.js": "export function helper() {}"
            }
        )

        files = gen.generate(structure, str(tmp_path))

        # Should use pytest (Python is first language)
        assert "pytest.ini" in files
        assert "tests/test_app.py" in files
