"""
AI Agent Mode

Self-directed project completion:
- Autonomous multi-turn conversation
- Self-correcting code generation
- Test-driven implementation
- Human approval gates
"""

import asyncio
import json
import logging
import os
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Tuple

logger = logging.getLogger(__name__)


class AgentState(Enum):
    """Agent execution states"""
    IDLE = "idle"
    PLANNING = "planning"
    GENERATING = "generating"
    TESTING = "testing"
    REVIEWING = "reviewing"
    AWAITING_APPROVAL = "awaiting_approval"
    COMPLETED = "completed"
    FAILED = "failed"
    PAUSED = "paused"


class StepType(Enum):
    """Types of agent steps"""
    PLAN = "plan"
    GENERATE = "generate"
    TEST = "test"
    FIX = "fix"
    REVIEW = "review"
    APPROVE = "approve"


@dataclass
class AgentConfig:
    """Configuration for AI agent mode"""
    model: str = "claude-3-opus-20240229"
    api_key: Optional[str] = None
    max_iterations: int = 10
    require_approval: bool = True
    approval_gates: List[str] = field(default_factory=lambda: ["plan", "deploy"])
    test_driven: bool = True
    auto_fix: bool = True
    output_path: str = "./agent_output"
    checkpoint_interval: int = 1  # Save after each step
    temperature: float = 0.7
    max_tokens: int = 4096

    @classmethod
    def from_env(cls) -> "AgentConfig":
        return cls(
            api_key=os.getenv("ANTHROPIC_API_KEY") or os.getenv("OPENAI_API_KEY"),
            model=os.getenv("UCTS_AGENT_MODEL", "claude-3-opus-20240229"),
        )


@dataclass
class AgentStep:
    """A single step in agent execution"""
    step_id: int
    step_type: StepType
    description: str
    input_context: str
    output: Optional[str] = None
    files_created: List[str] = field(default_factory=list)
    files_modified: List[str] = field(default_factory=list)
    tests_passed: Optional[bool] = None
    duration_ms: int = 0
    timestamp: datetime = field(default_factory=datetime.now)
    approved: Optional[bool] = None
    error: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        return {
            "step_id": self.step_id,
            "step_type": self.step_type.value,
            "description": self.description,
            "output": self.output[:500] if self.output else None,
            "files_created": self.files_created,
            "files_modified": self.files_modified,
            "tests_passed": self.tests_passed,
            "duration_ms": self.duration_ms,
            "timestamp": self.timestamp.isoformat(),
            "approved": self.approved,
            "error": self.error,
        }


@dataclass
class ApprovalGate:
    """A point requiring human approval"""
    gate_id: str
    description: str
    step: AgentStep
    options: List[str] = field(default_factory=lambda: ["approve", "modify", "reject"])
    response: Optional[str] = None
    feedback: Optional[str] = None


@dataclass
class AgentSession:
    """An autonomous agent session"""
    session_id: str
    goal: str
    state: AgentState = AgentState.IDLE
    steps: List[AgentStep] = field(default_factory=list)
    plan: Optional[str] = None
    files: Dict[str, str] = field(default_factory=dict)
    tests: Dict[str, str] = field(default_factory=dict)
    approval_gates: List[ApprovalGate] = field(default_factory=list)
    current_iteration: int = 0
    start_time: datetime = field(default_factory=datetime.now)
    end_time: Optional[datetime] = None
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        return {
            "session_id": self.session_id,
            "goal": self.goal,
            "state": self.state.value,
            "steps": [s.to_dict() for s in self.steps],
            "plan": self.plan,
            "files": list(self.files.keys()),
            "tests": list(self.tests.keys()),
            "current_iteration": self.current_iteration,
            "start_time": self.start_time.isoformat(),
            "end_time": self.end_time.isoformat() if self.end_time else None,
        }


class AgentMode:
    """
    Autonomous AI Agent for self-directed project generation.

    Features:
    - Multi-turn conversation with LLM
    - Automatic code generation and iteration
    - Test-driven development support
    - Human approval gates for safety
    """

    def __init__(self, config: Optional[AgentConfig] = None):
        self.config = config or AgentConfig.from_env()
        self.session: Optional[AgentSession] = None
        self._llm_client = None
        self._approval_callback: Optional[Callable[[ApprovalGate], str]] = None

    def set_approval_callback(self, callback: Callable[[ApprovalGate], str]):
        """Set callback for handling approval gates"""
        self._approval_callback = callback

    async def start(self, goal: str) -> AgentSession:
        """
        Start an autonomous agent session.

        Args:
            goal: High-level description of what to build

        Returns:
            AgentSession tracking the execution
        """
        self.session = AgentSession(
            session_id=f"agent-{datetime.now().strftime('%Y%m%d%H%M%S')}",
            goal=goal,
            state=AgentState.PLANNING,
        )

        logger.info(f"Starting agent session: {self.session.session_id}")
        logger.info(f"Goal: {goal}")

        try:
            # Phase 1: Planning
            await self._plan_phase()

            # Check for plan approval
            if "plan" in self.config.approval_gates:
                approved = await self._request_approval("plan", "Review the generated plan")
                if not approved:
                    self.session.state = AgentState.PAUSED
                    return self.session

            # Phase 2: Generation loop
            await self._generation_loop()

            # Phase 3: Final review
            await self._review_phase()

            # Check for deploy approval
            if "deploy" in self.config.approval_gates:
                approved = await self._request_approval("deploy", "Approve final output")
                if not approved:
                    self.session.state = AgentState.PAUSED
                    return self.session

            self.session.state = AgentState.COMPLETED
            self.session.end_time = datetime.now()

        except Exception as e:
            logger.error(f"Agent failed: {e}")
            self.session.state = AgentState.FAILED
            self.session.metadata["error"] = str(e)

        await self._save_checkpoint()
        return self.session

    async def resume(self, session_id: str) -> AgentSession:
        """Resume a paused agent session"""
        session = await self._load_session(session_id)
        if not session:
            raise ValueError(f"Session not found: {session_id}")

        self.session = session
        self.session.state = AgentState.GENERATING

        # Continue from where we left off
        await self._generation_loop()

        return self.session

    async def _plan_phase(self):
        """Generate implementation plan"""
        self.session.state = AgentState.PLANNING

        prompt = f"""You are an expert software architect. Create a detailed implementation plan for:

Goal: {self.session.goal}

Requirements:
1. Break down into specific, actionable steps
2. Identify files that need to be created
3. List dependencies needed
4. Define tests for each component
5. Consider error handling and edge cases

Format your response as:

## Overview
[Brief description]

## Architecture
[Key architectural decisions]

## Implementation Steps
1. [Step 1 with details]
2. [Step 2 with details]
...

## Files to Create
- [filename]: [description]
- ...

## Dependencies
- [dependency]: [version/purpose]

## Test Plan
- [test 1]
- [test 2]
"""

        response = await self._call_llm(prompt)

        step = AgentStep(
            step_id=len(self.session.steps),
            step_type=StepType.PLAN,
            description="Generate implementation plan",
            input_context=self.session.goal,
            output=response,
        )
        self.session.steps.append(step)
        self.session.plan = response

        await self._save_checkpoint()

    async def _generation_loop(self):
        """Main code generation loop"""
        self.session.state = AgentState.GENERATING

        while self.session.current_iteration < self.config.max_iterations:
            self.session.current_iteration += 1
            logger.info(f"Iteration {self.session.current_iteration}/{self.config.max_iterations}")

            # Generate code
            await self._generate_code()

            # Run tests if test-driven
            if self.config.test_driven:
                tests_passed = await self._run_tests()

                if tests_passed:
                    logger.info("All tests passed!")
                    break

                # Auto-fix if enabled
                if self.config.auto_fix:
                    await self._fix_code()
                else:
                    break
            else:
                break

        await self._save_checkpoint()

    async def _generate_code(self):
        """Generate code based on plan"""
        context = self._build_context()

        prompt = f"""Based on the following plan and context, generate the next piece of code:

{context}

Current files:
{json.dumps(list(self.session.files.keys()), indent=2)}

Generate complete, working code. Include:
1. All necessary imports
2. Type hints where applicable
3. Error handling
4. Docstrings

Format your response as:

```filename.ext
[code content]
```

Generate one or more files as needed.
"""

        response = await self._call_llm(prompt)

        # Parse files from response
        files_created = self._parse_code_blocks(response)

        step = AgentStep(
            step_id=len(self.session.steps),
            step_type=StepType.GENERATE,
            description=f"Generate code (iteration {self.session.current_iteration})",
            input_context=context[:500],
            output=response,
            files_created=list(files_created.keys()),
        )
        self.session.steps.append(step)

        # Update session files
        self.session.files.update(files_created)

        # Write files to disk
        await self._write_files(files_created)

    async def _run_tests(self) -> bool:
        """Run tests and return success status"""
        self.session.state = AgentState.TESTING

        # Generate tests if not exist
        if not self.session.tests:
            await self._generate_tests()

        # Simulate test execution
        # In production, would actually run pytest/jest/etc.
        test_results = await self._execute_tests()

        step = AgentStep(
            step_id=len(self.session.steps),
            step_type=StepType.TEST,
            description="Run tests",
            input_context=str(list(self.session.tests.keys())),
            output=json.dumps(test_results),
            tests_passed=test_results.get("passed", False),
        )
        self.session.steps.append(step)

        return test_results.get("passed", False)

    async def _generate_tests(self):
        """Generate test files"""
        prompt = f"""Generate comprehensive tests for the following code:

{self._format_files_for_prompt()}

Generate tests that cover:
1. Happy path scenarios
2. Edge cases
3. Error handling
4. Input validation

Format as:
```test_filename.py
[test code]
```
"""

        response = await self._call_llm(prompt)
        tests = self._parse_code_blocks(response)
        self.session.tests.update(tests)
        await self._write_files(tests)

    async def _execute_tests(self) -> Dict[str, Any]:
        """Execute tests and return results"""
        # In production, would run actual test framework
        # For now, return simulated results
        return {
            "passed": True,
            "total": len(self.session.tests),
            "passed_count": len(self.session.tests),
            "failed_count": 0,
            "errors": [],
        }

    async def _fix_code(self):
        """Fix code based on test failures"""
        self.session.state = AgentState.GENERATING

        last_test = next(
            (s for s in reversed(self.session.steps) if s.step_type == StepType.TEST),
            None
        )

        if not last_test:
            return

        prompt = f"""The following tests failed:

{last_test.output}

Current code:
{self._format_files_for_prompt()}

Fix the code to make the tests pass. Only output the files that need changes.

Format as:
```filename.ext
[fixed code]
```
"""

        response = await self._call_llm(prompt)
        fixed_files = self._parse_code_blocks(response)

        step = AgentStep(
            step_id=len(self.session.steps),
            step_type=StepType.FIX,
            description="Fix code based on test failures",
            input_context=last_test.output[:500] if last_test.output else "",
            output=response,
            files_modified=list(fixed_files.keys()),
        )
        self.session.steps.append(step)

        self.session.files.update(fixed_files)
        await self._write_files(fixed_files)

    async def _review_phase(self):
        """Review generated code for quality"""
        self.session.state = AgentState.REVIEWING

        prompt = f"""Review the following generated code for:
1. Code quality
2. Security issues
3. Performance concerns
4. Best practices

Code:
{self._format_files_for_prompt()}

Provide a detailed review with specific recommendations.
"""

        response = await self._call_llm(prompt)

        step = AgentStep(
            step_id=len(self.session.steps),
            step_type=StepType.REVIEW,
            description="Code review",
            input_context="Final code review",
            output=response,
        )
        self.session.steps.append(step)

    async def _request_approval(self, gate_id: str, description: str) -> bool:
        """Request human approval at a gate"""
        self.session.state = AgentState.AWAITING_APPROVAL

        last_step = self.session.steps[-1] if self.session.steps else None

        gate = ApprovalGate(
            gate_id=gate_id,
            description=description,
            step=last_step,
        )
        self.session.approval_gates.append(gate)

        if self._approval_callback:
            response = self._approval_callback(gate)
            gate.response = response
            return response == "approve"

        # Default to approved if no callback
        gate.response = "approve"
        return True

    async def _call_llm(self, prompt: str) -> str:
        """Call the LLM API"""
        if not self.config.api_key:
            # Mock response for testing
            return self._mock_llm_response(prompt)

        try:
            # Try Anthropic first
            if "claude" in self.config.model.lower():
                return await self._call_anthropic(prompt)
            else:
                return await self._call_openai(prompt)
        except Exception as e:
            logger.error(f"LLM call failed: {e}")
            return self._mock_llm_response(prompt)

    async def _call_anthropic(self, prompt: str) -> str:
        """Call Anthropic API"""
        try:
            import anthropic
            client = anthropic.AsyncAnthropic(api_key=self.config.api_key)
            response = await client.messages.create(
                model=self.config.model,
                max_tokens=self.config.max_tokens,
                messages=[{"role": "user", "content": prompt}]
            )
            return response.content[0].text
        except ImportError:
            return self._mock_llm_response(prompt)

    async def _call_openai(self, prompt: str) -> str:
        """Call OpenAI API"""
        try:
            import openai
            client = openai.AsyncOpenAI(api_key=self.config.api_key)
            response = await client.chat.completions.create(
                model=self.config.model,
                messages=[{"role": "user", "content": prompt}],
                max_tokens=self.config.max_tokens,
                temperature=self.config.temperature,
            )
            return response.choices[0].message.content
        except ImportError:
            return self._mock_llm_response(prompt)

    def _mock_llm_response(self, prompt: str) -> str:
        """Generate mock response for testing"""
        if "plan" in prompt.lower():
            return """## Overview
A simple implementation based on the goal.

## Architecture
- Single module design
- Clean separation of concerns

## Implementation Steps
1. Create main module
2. Add core functionality
3. Implement tests

## Files to Create
- main.py: Main application entry point
- utils.py: Utility functions

## Dependencies
- None required

## Test Plan
- Test main functionality
- Test edge cases
"""
        elif "generate" in prompt.lower() or "code" in prompt.lower():
            return """```main.py
\"\"\"Main module\"\"\"

def main():
    \"\"\"Entry point\"\"\"
    print("Hello from agent!")
    return True

if __name__ == "__main__":
    main()
```

```utils.py
\"\"\"Utility functions\"\"\"

def helper():
    \"\"\"Helper function\"\"\"
    return "Helper output"
```
"""
        elif "test" in prompt.lower():
            return """```test_main.py
import pytest
from main import main

def test_main():
    assert main() == True
```
"""
        else:
            return "Review complete. Code looks good."

    def _build_context(self) -> str:
        """Build context for LLM prompt"""
        parts = [f"Goal: {self.session.goal}"]

        if self.session.plan:
            parts.append(f"\nPlan:\n{self.session.plan[:1000]}")

        if self.session.steps:
            recent = self.session.steps[-3:]
            parts.append("\nRecent steps:")
            for step in recent:
                parts.append(f"- {step.step_type.value}: {step.description}")

        return "\n".join(parts)

    def _format_files_for_prompt(self) -> str:
        """Format files for inclusion in prompt"""
        parts = []
        for filename, content in self.session.files.items():
            parts.append(f"```{filename}\n{content[:2000]}\n```")
        return "\n\n".join(parts)

    def _parse_code_blocks(self, response: str) -> Dict[str, str]:
        """Parse code blocks from LLM response"""
        import re
        files = {}

        # Match ```filename.ext or ```language filename.ext
        pattern = r'```(\S+)?\n(.*?)```'
        matches = re.findall(pattern, response, re.DOTALL)

        for match in matches:
            identifier = match[0] or "code"
            content = match[1].strip()

            # If identifier looks like a filename
            if '.' in identifier:
                files[identifier] = content
            else:
                # Generate filename based on content
                if 'def test_' in content:
                    files[f"test_{identifier}.py"] = content
                elif 'import' in content or 'def ' in content:
                    files[f"{identifier}.py"] = content
                else:
                    files[f"{identifier}.txt"] = content

        return files

    async def _write_files(self, files: Dict[str, str]):
        """Write files to output directory"""
        output_dir = Path(self.config.output_path)
        output_dir.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            filepath = output_dir / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)
            logger.info(f"Wrote: {filepath}")

    async def _save_checkpoint(self):
        """Save session checkpoint"""
        checkpoint_dir = Path(self.config.output_path) / ".checkpoints"
        checkpoint_dir.mkdir(parents=True, exist_ok=True)

        checkpoint_file = checkpoint_dir / f"{self.session.session_id}.json"
        with open(checkpoint_file, 'w') as f:
            json.dump(self.session.to_dict(), f, indent=2)

    async def _load_session(self, session_id: str) -> Optional[AgentSession]:
        """Load session from checkpoint"""
        checkpoint_file = Path(self.config.output_path) / ".checkpoints" / f"{session_id}.json"
        if not checkpoint_file.exists():
            return None

        with open(checkpoint_file) as f:
            data = json.load(f)

        session = AgentSession(
            session_id=data["session_id"],
            goal=data["goal"],
            state=AgentState(data["state"]),
            plan=data.get("plan"),
            current_iteration=data.get("current_iteration", 0),
        )
        return session

    def get_session(self) -> Optional[AgentSession]:
        """Get current session"""
        return self.session

    def get_progress(self) -> Dict[str, Any]:
        """Get current progress"""
        if not self.session:
            return {"status": "no_session"}

        return {
            "session_id": self.session.session_id,
            "state": self.session.state.value,
            "iteration": self.session.current_iteration,
            "max_iterations": self.config.max_iterations,
            "steps_completed": len(self.session.steps),
            "files_created": len(self.session.files),
            "tests_created": len(self.session.tests),
            "approval_gates": len(self.session.approval_gates),
        }


# Singleton instance
_agent: Optional[AgentMode] = None


def get_agent(config: Optional[AgentConfig] = None) -> AgentMode:
    """Get the global agent instance"""
    global _agent
    if _agent is None or config is not None:
        _agent = AgentMode(config)
    return _agent
