"""
Intelligent Code Enhancement

AI-powered improvements to generated code:
- Detect missing error handling patterns
- Suggest security improvements
- Add documentation where missing
- Identify incomplete implementations
- Pattern matching rules engine
- Integration with static analysis tools
"""

import re
import logging
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Tuple

logger = logging.getLogger(__name__)


class EnhancementCategory(Enum):
    """Categories of code enhancements"""
    ERROR_HANDLING = "error_handling"
    SECURITY = "security"
    DOCUMENTATION = "documentation"
    TYPE_SAFETY = "type_safety"
    PERFORMANCE = "performance"
    CODE_QUALITY = "code_quality"
    COMPLETENESS = "completeness"
    BEST_PRACTICES = "best_practices"


class Severity(Enum):
    """Severity levels for suggestions"""
    INFO = "info"
    WARNING = "warning"
    ERROR = "error"
    CRITICAL = "critical"


@dataclass
class EnhancementConfig:
    """Configuration for code enhancement"""
    enabled_categories: List[EnhancementCategory] = field(
        default_factory=lambda: list(EnhancementCategory)
    )
    min_severity: Severity = Severity.INFO
    include_suggestions: bool = True
    include_code_fixes: bool = True
    max_suggestions_per_file: int = 20
    language_specific: bool = True


@dataclass
class CodeLocation:
    """Location in source code"""
    file: str
    line: int
    column: int = 0
    end_line: Optional[int] = None
    end_column: Optional[int] = None

    def __str__(self) -> str:
        if self.end_line and self.end_line != self.line:
            return f"{self.file}:{self.line}-{self.end_line}"
        return f"{self.file}:{self.line}"


@dataclass
class Enhancement:
    """A suggested code enhancement"""
    category: EnhancementCategory
    severity: Severity
    title: str
    description: str
    location: CodeLocation
    suggestion: Optional[str] = None
    code_fix: Optional[str] = None
    original_code: Optional[str] = None
    references: List[str] = field(default_factory=list)
    tags: List[str] = field(default_factory=list)

    def to_dict(self) -> Dict[str, Any]:
        return {
            "category": self.category.value,
            "severity": self.severity.value,
            "title": self.title,
            "description": self.description,
            "location": str(self.location),
            "suggestion": self.suggestion,
            "code_fix": self.code_fix,
            "tags": self.tags,
        }


@dataclass
class EnhancementReport:
    """Report of all enhancements for a project"""
    files_analyzed: int
    total_suggestions: int
    by_category: Dict[str, int] = field(default_factory=dict)
    by_severity: Dict[str, int] = field(default_factory=dict)
    enhancements: List[Enhancement] = field(default_factory=list)

    def to_markdown(self) -> str:
        """Generate markdown report"""
        lines = [
            "# Code Enhancement Report",
            "",
            f"**Files analyzed:** {self.files_analyzed}",
            f"**Total suggestions:** {self.total_suggestions}",
            "",
            "## Summary by Category",
        ]

        for cat, count in sorted(self.by_category.items(), key=lambda x: -x[1]):
            lines.append(f"- {cat}: {count}")

        lines.extend(["", "## Summary by Severity"])
        for sev, count in sorted(self.by_severity.items()):
            lines.append(f"- {sev}: {count}")

        lines.extend(["", "## Detailed Suggestions", ""])

        # Group by file
        by_file: Dict[str, List[Enhancement]] = {}
        for e in self.enhancements:
            if e.location.file not in by_file:
                by_file[e.location.file] = []
            by_file[e.location.file].append(e)

        for file, enhancements in by_file.items():
            lines.append(f"### `{file}`")
            lines.append("")

            for e in enhancements:
                severity_icon = {
                    Severity.INFO: "ℹ️",
                    Severity.WARNING: "⚠️",
                    Severity.ERROR: "❌",
                    Severity.CRITICAL: "🚨",
                }.get(e.severity, "")

                lines.append(f"#### {severity_icon} {e.title}")
                lines.append(f"*Line {e.location.line} | {e.category.value}*")
                lines.append("")
                lines.append(e.description)

                if e.suggestion:
                    lines.append("")
                    lines.append(f"**Suggestion:** {e.suggestion}")

                if e.code_fix:
                    lines.append("")
                    lines.append("**Suggested fix:**")
                    lines.append("```")
                    lines.append(e.code_fix)
                    lines.append("```")

                lines.append("")

        return "\n".join(lines)


class PatternRule:
    """A pattern-based enhancement rule"""

    def __init__(
        self,
        name: str,
        category: EnhancementCategory,
        severity: Severity,
        pattern: str,
        description: str,
        suggestion: str,
        languages: Optional[List[str]] = None,
        code_fix_template: Optional[str] = None,
    ):
        self.name = name
        self.category = category
        self.severity = severity
        self.pattern = re.compile(pattern, re.MULTILINE)
        self.description = description
        self.suggestion = suggestion
        self.languages = languages
        self.code_fix_template = code_fix_template

    def check(self, code: str, language: str) -> List[Tuple[int, str, str]]:
        """
        Check code for pattern matches.

        Returns list of (line_number, matched_text, suggested_fix)
        """
        if self.languages and language.lower() not in self.languages:
            return []

        matches = []
        for match in self.pattern.finditer(code):
            # Calculate line number
            line_num = code[:match.start()].count('\n') + 1
            matched_text = match.group(0)

            # Generate fix if template available
            fix = None
            if self.code_fix_template:
                try:
                    fix = self.code_fix_template.format(*match.groups())
                except Exception:
                    fix = self.code_fix_template

            matches.append((line_num, matched_text, fix))

        return matches


class CodeEnhancer:
    """
    Intelligent code enhancement engine.

    Analyzes code for potential improvements and generates suggestions.
    """

    # Built-in pattern rules
    DEFAULT_RULES = [
        # Error Handling
        PatternRule(
            name="bare_except",
            category=EnhancementCategory.ERROR_HANDLING,
            severity=Severity.WARNING,
            pattern=r"except\s*:",
            description="Bare except clause catches all exceptions including KeyboardInterrupt and SystemExit",
            suggestion="Catch specific exceptions instead of using bare except",
            languages=["python"],
            code_fix_template="except Exception as e:",
        ),
        PatternRule(
            name="empty_catch",
            category=EnhancementCategory.ERROR_HANDLING,
            severity=Severity.WARNING,
            pattern=r"catch\s*\([^)]*\)\s*\{\s*\}",
            description="Empty catch block silently swallows errors",
            suggestion="Log the error or handle it appropriately",
            languages=["javascript", "typescript", "java"],
        ),
        PatternRule(
            name="todo_fixme",
            category=EnhancementCategory.COMPLETENESS,
            severity=Severity.INFO,
            pattern=r"#\s*(TODO|FIXME|XXX|HACK)[:.\s](.+)",
            description="Code contains TODO/FIXME comment indicating incomplete implementation",
            suggestion="Complete the implementation or create a tracking issue",
            languages=["python"],
        ),
        PatternRule(
            name="todo_js",
            category=EnhancementCategory.COMPLETENESS,
            severity=Severity.INFO,
            pattern=r"//\s*(TODO|FIXME|XXX|HACK)[:.\s](.+)",
            description="Code contains TODO/FIXME comment indicating incomplete implementation",
            suggestion="Complete the implementation or create a tracking issue",
            languages=["javascript", "typescript"],
        ),

        # Security
        PatternRule(
            name="hardcoded_password",
            category=EnhancementCategory.SECURITY,
            severity=Severity.CRITICAL,
            pattern=r"(password|passwd|pwd)\s*=\s*['\"][^'\"]+['\"]",
            description="Hardcoded password detected",
            suggestion="Use environment variables or secure vault for credentials",
            languages=None,
        ),
        PatternRule(
            name="sql_string_format",
            category=EnhancementCategory.SECURITY,
            severity=Severity.CRITICAL,
            pattern=r"execute\s*\(\s*f?['\"].*\{.*\}.*['\"]",
            description="Possible SQL injection - using string formatting in SQL query",
            suggestion="Use parameterized queries instead of string formatting",
            languages=["python"],
        ),
        PatternRule(
            name="eval_usage",
            category=EnhancementCategory.SECURITY,
            severity=Severity.ERROR,
            pattern=r"\beval\s*\(",
            description="Use of eval() can lead to code injection",
            suggestion="Avoid eval() - use safer alternatives like JSON.parse() or ast.literal_eval()",
            languages=["python", "javascript"],
        ),

        # Documentation
        PatternRule(
            name="missing_docstring_py",
            category=EnhancementCategory.DOCUMENTATION,
            severity=Severity.INFO,
            pattern=r"def\s+(\w+)\s*\([^)]*\)\s*:\s*\n\s*(?!\"\"\"|\'\'\')(?=[^\s#])",
            description="Function missing docstring",
            suggestion="Add a docstring explaining the function's purpose, parameters, and return value",
            languages=["python"],
        ),
        PatternRule(
            name="missing_jsdoc",
            category=EnhancementCategory.DOCUMENTATION,
            severity=Severity.INFO,
            pattern=r"\n\s*(export\s+)?(async\s+)?function\s+\w+\s*\([^)]*\)\s*\{",
            description="Function may be missing JSDoc documentation",
            suggestion="Add JSDoc comment describing the function",
            languages=["javascript", "typescript"],
        ),

        # Type Safety
        PatternRule(
            name="any_type",
            category=EnhancementCategory.TYPE_SAFETY,
            severity=Severity.WARNING,
            pattern=r":\s*any\b",
            description="Use of 'any' type defeats TypeScript type checking",
            suggestion="Use a more specific type or 'unknown' for truly unknown types",
            languages=["typescript"],
        ),
        PatternRule(
            name="type_ignore",
            category=EnhancementCategory.TYPE_SAFETY,
            severity=Severity.WARNING,
            pattern=r"#\s*type:\s*ignore",
            description="Type checking disabled with # type: ignore",
            suggestion="Fix the type issue instead of ignoring it",
            languages=["python"],
        ),

        # Performance
        PatternRule(
            name="n_plus_one",
            category=EnhancementCategory.PERFORMANCE,
            severity=Severity.WARNING,
            pattern=r"for\s+\w+\s+in\s+\w+:\s*\n\s*\w+\.query\(",
            description="Potential N+1 query pattern detected",
            suggestion="Use eager loading or batch queries to avoid N+1 problem",
            languages=["python"],
        ),
        PatternRule(
            name="sync_in_async",
            category=EnhancementCategory.PERFORMANCE,
            severity=Severity.WARNING,
            pattern=r"async\s+def\s+\w+[^:]+:\s*\n(?:.*\n)*?\s+time\.sleep\(",
            description="Using synchronous sleep in async function",
            suggestion="Use await asyncio.sleep() instead of time.sleep()",
            languages=["python"],
        ),

        # Code Quality
        PatternRule(
            name="magic_number",
            category=EnhancementCategory.CODE_QUALITY,
            severity=Severity.INFO,
            pattern=r"(?<![.\w])[0-9]{3,}(?![.\w])",
            description="Magic number detected - consider using a named constant",
            suggestion="Extract magic number to a named constant for clarity",
            languages=None,
        ),
        PatternRule(
            name="long_function",
            category=EnhancementCategory.CODE_QUALITY,
            severity=Severity.WARNING,
            pattern=r"def\s+\w+[^:]+:(?:\n(?:\s{4}.*|\s*))*",
            description="Function may be too long",
            suggestion="Consider breaking into smaller functions",
            languages=["python"],
        ),

        # Best Practices
        PatternRule(
            name="mutable_default",
            category=EnhancementCategory.BEST_PRACTICES,
            severity=Severity.ERROR,
            pattern=r"def\s+\w+\s*\([^)]*=\s*(\[\]|\{\}|set\(\))",
            description="Mutable default argument - can cause unexpected behavior",
            suggestion="Use None as default and create mutable object inside function",
            languages=["python"],
            code_fix_template="=None",
        ),
        PatternRule(
            name="console_log",
            category=EnhancementCategory.BEST_PRACTICES,
            severity=Severity.INFO,
            pattern=r"console\.(log|debug|info)\s*\(",
            description="console.log statement should be removed for production",
            suggestion="Remove console.log or use proper logging library",
            languages=["javascript", "typescript"],
        ),
        PatternRule(
            name="print_debug",
            category=EnhancementCategory.BEST_PRACTICES,
            severity=Severity.INFO,
            pattern=r"^\s*print\s*\(",
            description="print() statement - consider using logging instead",
            suggestion="Use logging module for better control over output",
            languages=["python"],
        ),
    ]

    def __init__(self, config: Optional[EnhancementConfig] = None):
        self.config = config or EnhancementConfig()
        self.rules: List[PatternRule] = list(self.DEFAULT_RULES)
        self._custom_analyzers: List[Callable] = []

    def add_rule(self, rule: PatternRule):
        """Add a custom pattern rule"""
        self.rules.append(rule)

    def add_analyzer(self, analyzer: Callable[[str, str], List[Enhancement]]):
        """Add a custom code analyzer function"""
        self._custom_analyzers.append(analyzer)

    def analyze_code(
        self,
        code: str,
        filename: str,
        language: Optional[str] = None
    ) -> List[Enhancement]:
        """
        Analyze code and return enhancement suggestions.

        Args:
            code: Source code to analyze
            filename: Name of the file
            language: Programming language (auto-detected if not provided)

        Returns:
            List of Enhancement suggestions
        """
        if language is None:
            language = self._detect_language(filename)

        enhancements = []

        # Apply pattern rules
        for rule in self.rules:
            if rule.category not in self.config.enabled_categories:
                continue

            if rule.severity.value < self.config.min_severity.value:
                continue

            matches = rule.check(code, language)

            for line_num, matched_text, fix in matches:
                enhancement = Enhancement(
                    category=rule.category,
                    severity=rule.severity,
                    title=rule.name.replace("_", " ").title(),
                    description=rule.description,
                    location=CodeLocation(file=filename, line=line_num),
                    suggestion=rule.suggestion if self.config.include_suggestions else None,
                    code_fix=fix if self.config.include_code_fixes else None,
                    original_code=matched_text[:100],
                    tags=[rule.name, language],
                )
                enhancements.append(enhancement)

        # Run custom analyzers
        for analyzer in self._custom_analyzers:
            try:
                results = analyzer(code, language)
                enhancements.extend(results)
            except Exception as e:
                logger.error(f"Custom analyzer failed: {e}")

        # Language-specific deep analysis
        if self.config.language_specific:
            enhancements.extend(self._language_specific_analysis(code, filename, language))

        # Limit per file
        if len(enhancements) > self.config.max_suggestions_per_file:
            # Prioritize by severity
            enhancements.sort(key=lambda e: e.severity.value, reverse=True)
            enhancements = enhancements[:self.config.max_suggestions_per_file]

        return enhancements

    def analyze_project(
        self,
        structure: Any,
        output_path: Optional[str] = None
    ) -> EnhancementReport:
        """
        Analyze entire project structure.

        Args:
            structure: ProjectStructure from analysis
            output_path: Optional path to save report

        Returns:
            EnhancementReport with all suggestions
        """
        all_enhancements = []

        for filepath, content in structure.files.items():
            enhancements = self.analyze_code(content, filepath)
            all_enhancements.extend(enhancements)

        # Build report
        report = EnhancementReport(
            files_analyzed=len(structure.files),
            total_suggestions=len(all_enhancements),
            enhancements=all_enhancements,
        )

        # Count by category
        for e in all_enhancements:
            cat = e.category.value
            report.by_category[cat] = report.by_category.get(cat, 0) + 1

        # Count by severity
        for e in all_enhancements:
            sev = e.severity.value
            report.by_severity[sev] = report.by_severity.get(sev, 0) + 1

        # Save report if path provided
        if output_path:
            Path(output_path).write_text(report.to_markdown())
            logger.info(f"Enhancement report saved to {output_path}")

        return report

    def _detect_language(self, filename: str) -> str:
        """Detect language from filename"""
        suffix = Path(filename).suffix.lower()
        mapping = {
            '.py': 'python',
            '.js': 'javascript',
            '.ts': 'typescript',
            '.tsx': 'typescript',
            '.jsx': 'javascript',
            '.go': 'go',
            '.rs': 'rust',
            '.java': 'java',
            '.rb': 'ruby',
            '.php': 'php',
            '.c': 'c',
            '.cpp': 'cpp',
            '.cs': 'csharp',
            '.swift': 'swift',
            '.kt': 'kotlin',
        }
        return mapping.get(suffix, 'unknown')

    def _language_specific_analysis(
        self,
        code: str,
        filename: str,
        language: str
    ) -> List[Enhancement]:
        """Perform language-specific deep analysis"""
        enhancements = []

        if language == 'python':
            enhancements.extend(self._analyze_python(code, filename))
        elif language in ['javascript', 'typescript']:
            enhancements.extend(self._analyze_javascript(code, filename))

        return enhancements

    def _analyze_python(self, code: str, filename: str) -> List[Enhancement]:
        """Python-specific analysis"""
        enhancements = []

        # Check for missing __init__ in class
        class_pattern = r"class\s+(\w+)[^:]*:\s*\n(?!\s+def\s+__init__)"
        for match in re.finditer(class_pattern, code):
            line_num = code[:match.start()].count('\n') + 1
            enhancements.append(Enhancement(
                category=EnhancementCategory.COMPLETENESS,
                severity=Severity.INFO,
                title="Missing __init__ Method",
                description=f"Class {match.group(1)} may need an __init__ method",
                location=CodeLocation(file=filename, line=line_num),
                suggestion="Add __init__ method if the class needs initialization",
            ))

        # Check for unhandled async context managers
        async_with_pattern = r"async\s+with\s+(?!aiohttp|asyncio)"
        for match in re.finditer(async_with_pattern, code):
            line_num = code[:match.start()].count('\n') + 1
            enhancements.append(Enhancement(
                category=EnhancementCategory.ERROR_HANDLING,
                severity=Severity.INFO,
                title="Async Context Manager",
                description="Ensure async context manager properly handles exceptions",
                location=CodeLocation(file=filename, line=line_num),
                suggestion="Wrap in try/except for proper error handling",
            ))

        return enhancements

    def _analyze_javascript(self, code: str, filename: str) -> List[Enhancement]:
        """JavaScript/TypeScript-specific analysis"""
        enhancements = []

        # Check for unhandled promise rejections
        promise_pattern = r"\.then\s*\([^)]+\)(?!\s*\.catch)"
        for match in re.finditer(promise_pattern, code):
            line_num = code[:match.start()].count('\n') + 1
            enhancements.append(Enhancement(
                category=EnhancementCategory.ERROR_HANDLING,
                severity=Severity.WARNING,
                title="Unhandled Promise Rejection",
                description="Promise chain without .catch() handler",
                location=CodeLocation(file=filename, line=line_num),
                suggestion="Add .catch() to handle promise rejections",
                code_fix=".catch(error => console.error(error))",
            ))

        # Check for missing return types in TypeScript
        if filename.endswith('.ts') or filename.endswith('.tsx'):
            func_pattern = r"(export\s+)?(async\s+)?function\s+\w+\s*\([^)]*\)(?!\s*:)"
            for match in re.finditer(func_pattern, code):
                line_num = code[:match.start()].count('\n') + 1
                enhancements.append(Enhancement(
                    category=EnhancementCategory.TYPE_SAFETY,
                    severity=Severity.INFO,
                    title="Missing Return Type",
                    description="Function missing explicit return type annotation",
                    location=CodeLocation(file=filename, line=line_num),
                    suggestion="Add return type annotation for better type safety",
                ))

        return enhancements

    def get_enhancement_summary(self, enhancements: List[Enhancement]) -> str:
        """Generate a brief summary of enhancements"""
        by_severity = {}
        for e in enhancements:
            sev = e.severity.value
            by_severity[sev] = by_severity.get(sev, 0) + 1

        parts = []
        for sev in ['critical', 'error', 'warning', 'info']:
            if sev in by_severity:
                parts.append(f"{by_severity[sev]} {sev}")

        return ", ".join(parts) if parts else "No issues found"

    @staticmethod
    def list_categories() -> List[Dict[str, str]]:
        """List available enhancement categories"""
        return [
            {"name": c.value, "description": c.name.replace("_", " ").title()}
            for c in EnhancementCategory
        ]

    @staticmethod
    def list_rules() -> List[Dict[str, Any]]:
        """List available pattern rules"""
        return [
            {
                "name": r.name,
                "category": r.category.value,
                "severity": r.severity.value,
                "description": r.description,
                "languages": r.languages,
            }
            for r in CodeEnhancer.DEFAULT_RULES
        ]
