"""
UCTS CLI - Universal Context Transfer System Command Line Interface
"""
import click
from rich.console import Console
from rich.table import Table
from rich.panel import Panel
from pathlib import Path

console = Console()


@click.group()
@click.version_option(version="2.0.0", prog_name="ucts")
def cli():
    """UCTS - Universal Context Transfer System

    Turn any AI conversation into a deployable project in one command.
    """
    pass


@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.option('--format', '-f',
              type=click.Choice(['auto', 'claude', 'vscode', 'web', 'transcript']),
              default='auto',
              help='Source format (auto-detect by default)')
def ingest(source, format):
    """Ingest a conversation session.

    SOURCE is the path to the conversation file (JSON, MD, or TXT).
    """
    from ucts.ingestion import get_ingester

    console.print(f"\n[bold blue]Ingesting:[/bold blue] {source}")

    ingester = get_ingester(format, source)
    session = ingester.ingest(source)

    console.print(f"\n[green]Successfully ingested {len(session.messages)} messages[/green]")
    console.print(f"[blue]Found {len(session.code_blocks)} code blocks[/blue]")

    if session.todos:
        console.print(f"[yellow]Extracted {len(session.todos)} TODOs[/yellow]")

    if session.decisions:
        console.print(f"[magenta]Found {len(session.decisions)} decisions[/magenta]")

    # Show summary
    table = Table(title="Session Summary")
    table.add_column("Property", style="cyan")
    table.add_column("Value", style="green")

    table.add_row("Source", session.source)
    table.add_row("Messages", str(len(session.messages)))
    table.add_row("Code Blocks", str(len(session.code_blocks)))
    table.add_row("Languages", ", ".join(set(cb.language for cb in session.code_blocks)) or "N/A")

    console.print(table)


@cli.command()
@click.argument('sources', type=click.Path(exists=True), nargs=-1, required=True)
@click.argument('output', type=click.Path())
@click.option('--target', '-t',
              type=click.Choice(['vscode', 'gitlab', 'github']),
              default='vscode',
              help='Target project type')
@click.option('--name', '-n', default=None, help='Project name (auto-inferred if not provided)')
@click.option('--merge', '-m', is_flag=True, help='Merge into existing project instead of creating new')
@click.option('--merge-strategy', '-s',
              type=click.Choice(['skip', 'overwrite', 'append', 'smart']),
              default='smart',
              help='Merge strategy for existing files (default: smart)')
@click.option('--gitlab-url', envvar='GITLAB_URL', help='GitLab server URL')
@click.option('--gitlab-token', envvar='GITLAB_TOKEN', help='GitLab API token')
@click.option('--github-token', envvar='GITHUB_TOKEN', help='GitHub API token')
def forge(sources, output, target, name, merge, merge_strategy, gitlab_url, gitlab_token, github_token):
    """Convert conversation session(s) to a project (forge).

    SOURCES is one or more conversation file paths.
    OUTPUT is the destination path or project name.

    Examples:
        ucts forge chat.json ./my-project
        ucts forge session1.json session2.json ./combined-project
        ucts forge new-features.json ./existing-project --merge

    Use --merge to add code to an existing project instead of creating new.
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators import VSCodeGenerator, GitLabGenerator, GitHubGenerator
    from ucts.generators.merger import ProjectMerger, MergeStrategy
    from ucts.core.models import Session

    multi_session = len(sources) > 1
    mode = "Merging into" if merge else ("Combining sessions into" if multi_session else "Converting session to")
    console.print(Panel.fit(
        f"[bold]UCTS Forge[/bold] - {mode} project",
        border_style="blue"
    ))

    # Step 1: Ingest (support multiple sources)
    console.print(f"\n[bold]Step 1:[/bold] Ingesting {'sessions' if multi_session else 'session'}...")

    sessions = []
    for source in sources:
        ingester = get_ingester('auto', source)
        session = ingester.ingest(source)
        sessions.append(session)
        console.print(f"  [green]{Path(source).name}: {len(session.messages)} messages[/green]")

    # Merge sessions if multiple
    if multi_session:
        session = Session.merge_all(sessions)
        console.print(f"  [cyan]Combined: {len(session.messages)} total messages[/cyan]")
    else:
        session = sessions[0]

    # Step 2: Analyze
    console.print("\n[bold]Step 2:[/bold] Analyzing conversation...")
    engine = AnalysisEngine()
    structure = engine.analyze(session)

    if name:
        structure.name = name

    console.print(f"  [green]Project: {structure.name}[/green]")
    console.print(f"  [green]Languages: {', '.join(structure.languages) or 'None detected'}[/green]")
    console.print(f"  [green]Files: {len(structure.files)}[/green]")

    # Step 3: Generate or Merge
    if merge:
        # Merge mode - add to existing project
        console.print(f"\n[bold]Step 3:[/bold] Merging into existing project ({merge_strategy} strategy)...")

        output_path = Path(output)
        if not output_path.exists():
            console.print(f"[red]Error: Target path does not exist: {output}[/red]")
            console.print("  Use without --merge to create a new project")
            return

        strategy_map = {
            'skip': MergeStrategy.SKIP,
            'overwrite': MergeStrategy.OVERWRITE,
            'append': MergeStrategy.APPEND,
            'smart': MergeStrategy.SMART,
        }
        merger = ProjectMerger(strategy=strategy_map[merge_strategy])
        result = merger.merge(structure, output)

        # Display merge results
        console.print(f"\n[bold green]Merge complete![/bold green]")

        if result.files_created:
            console.print(f"\n[green]Files created ({len(result.files_created)}):[/green]")
            for f in result.files_created[:10]:
                console.print(f"  + {f}")
            if len(result.files_created) > 10:
                console.print(f"  ... and {len(result.files_created) - 10} more")

        if result.files_updated:
            console.print(f"\n[yellow]Files updated ({len(result.files_updated)}):[/yellow]")
            for f in result.files_updated[:10]:
                console.print(f"  ~ {f}")
            if len(result.files_updated) > 10:
                console.print(f"  ... and {len(result.files_updated) - 10} more")

        if result.files_skipped:
            console.print(f"\n[dim]Files skipped ({len(result.files_skipped)}):[/dim]")
            for f in result.files_skipped[:5]:
                console.print(f"  - {f}")
            if len(result.files_skipped) > 5:
                console.print(f"  ... and {len(result.files_skipped) - 5} more")

        if result.conflicts:
            console.print(f"\n[red]Conflicts ({len(result.conflicts)}):[/red]")
            for f in result.conflicts:
                console.print(f"  ! {f}")
            console.print("  [dim]Review these files manually[/dim]")

        if result.dependencies_added:
            console.print(f"\n[cyan]Dependencies added:[/cyan]")
            for eco, deps in result.dependencies_added.items():
                console.print(f"  {eco}: {', '.join(deps)}")

    else:
        # Create mode - generate new project
        console.print(f"\n[bold]Step 3:[/bold] Generating {target} project...")

        if target == 'vscode':
            gen = VSCodeGenerator()
            result = gen.generate(structure, output)
            console.print(f"\n[bold green]VS Code workspace created:[/bold green] {result}")
            console.print(f"  Open with: [cyan]code {result}[/cyan]")

        elif target == 'gitlab':
            if not gitlab_url or not gitlab_token:
                console.print("[red]Error: GitLab URL and token required[/red]")
                console.print("  Set GITLAB_URL and GITLAB_TOKEN environment variables")
                return

            gen = GitLabGenerator(gitlab_url, gitlab_token)
            url = gen.generate(structure)
            console.print(f"\n[bold green]GitLab repository created:[/bold green] {url}")

        elif target == 'github':
            if not github_token:
                console.print("[red]Error: GitHub token required[/red]")
                console.print("  Set GITHUB_TOKEN environment variable")
                return

            gen = GitHubGenerator(github_token)
            url = gen.generate(structure)
            console.print(f"\n[bold green]GitHub repository created:[/bold green] {url}")


@cli.command()
@click.option('--project', '-p', default='default', help='Project name')
def status(project):
    """Show UCTS status and statistics."""
    from ucts.core import UniversalContextManager

    manager = UniversalContextManager(project)
    status_info = manager.get_status()

    table = Table(title=f"UCTS Status - {project}")
    table.add_column("Metric", style="cyan")
    table.add_column("Value", style="green")

    table.add_row("Project", status_info['project'])
    table.add_row("Current Messages", str(status_info['message_count']))
    table.add_row("Estimated Tokens", str(status_info['estimated_tokens']))
    table.add_row("Saved Chunks", str(status_info['chunk_count']))
    table.add_row("Storage Path", status_info['storage_path'])

    console.print(table)


@cli.command()
@click.option('--project', '-p', default='default', help='Project name')
@click.option('--output', '-o', type=click.Path(), help='Output file path')
def transfer(project, output):
    """Create a context transfer package."""
    from ucts.core import UniversalContextManager

    manager = UniversalContextManager(project)
    package = manager.create_transfer_package()

    console.print(f"\n[bold green]Transfer package created![/bold green]")
    console.print(f"  ID: {package['transfer_id']}")
    console.print(f"  Messages: {package['total_messages']}")
    console.print(f"  Tokens: ~{package['estimated_tokens']}")

    if output:
        output_path = Path(output)
        if output_path.suffix == '.md':
            output_path.write_text(package['markdown'])
        else:
            import json
            output_path.write_text(json.dumps(package, indent=2))
        console.print(f"  Saved to: {output}")


@cli.command()
@click.argument('source', type=click.Path(exists=True))
def analyze(source):
    """Analyze a conversation without generating a project."""
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine

    console.print(f"\n[bold blue]Analyzing:[/bold blue] {source}")

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    # Display analysis
    console.print(Panel.fit(
        f"[bold]{structure.name}[/bold]\n{structure.description}",
        title="Project Analysis",
        border_style="green"
    ))

    # Languages
    if structure.languages:
        table = Table(title="Languages Detected")
        table.add_column("Language", style="cyan")
        for lang in structure.languages:
            table.add_row(lang)
        console.print(table)

    # Dependencies
    if structure.dependencies:
        table = Table(title="Dependencies")
        table.add_column("Ecosystem", style="cyan")
        table.add_column("Packages", style="green")
        for eco, deps in structure.dependencies.items():
            table.add_row(eco, ", ".join(deps[:5]) + ("..." if len(deps) > 5 else ""))
        console.print(table)

    # Files
    if structure.files:
        table = Table(title="Files to Generate")
        table.add_column("Path", style="cyan")
        table.add_column("Size", style="green")
        for path, content in list(structure.files.items())[:10]:
            table.add_row(path, f"{len(content)} chars")
        if len(structure.files) > 10:
            table.add_row("...", f"+{len(structure.files) - 10} more")
        console.print(table)

    # TODOs
    if structure.todos:
        console.print("\n[bold yellow]TODOs Found:[/bold yellow]")
        for todo in structure.todos[:5]:
            console.print(f"  - {todo}")


@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--port', '-p', default=8000, help='Port to expose (default: 8000)')
@click.option('--python-version', default='3.11', help='Python version (default: 3.11)')
@click.option('--node-version', default='20', help='Node.js version (default: 20)')
@click.option('--alpine', is_flag=True, help='Use Alpine-based images')
@click.option('--no-multi-stage', is_flag=True, help='Disable multi-stage builds')
def docker(source, output, port, python_version, node_version, alpine, no_multi_stage):
    """Generate Docker configuration for a project.

    SOURCE is the conversation file path.
    OUTPUT is the destination directory for Docker files.

    Generates Dockerfile, docker-compose.yml, and .dockerignore.
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators.docker import DockerGenerator, DockerConfig

    console.print(Panel.fit(
        "[bold]UCTS Docker Generator[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print("\n[bold]Step 1:[/bold] Analyzing project...")
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    console.print(f"  [green]Languages: {', '.join(structure.languages)}[/green]")
    console.print(f"  [green]Dependencies: {sum(len(d) for d in structure.dependencies.values())} packages[/green]")

    # Generate Docker files
    console.print("\n[bold]Step 2:[/bold] Generating Docker configuration...")

    config = DockerConfig(
        python_version=python_version,
        node_version=node_version,
        expose_port=port,
        use_alpine=alpine,
        multi_stage=not no_multi_stage
    )

    generator = DockerGenerator(config)
    files = generator.generate(structure, output)

    console.print(f"\n[bold green]Docker files created in {output}:[/bold green]")
    for filename in files:
        console.print(f"  [cyan]{filename}[/cyan]")

    console.print(f"\n[dim]Build: docker build -t {structure.name} .[/dim]")
    console.print(f"[dim]Run:   docker compose up[/dim]")


@cli.command()
@click.argument('template_name')
@click.argument('output', type=click.Path())
@click.option('--name', '-n', help='Project name (defaults to output folder name)')
@click.option('--description', '-d', default='', help='Project description')
@click.option('--author', '-a', default='Author', help='Author name')
def template(template_name, output, name, description, author):
    """Generate a project from a template.

    TEMPLATE_NAME is the template to use (e.g., fastapi, nextjs).
    OUTPUT is the destination directory.

    Use 'ucts templates' to list available templates.
    """
    from ucts.templates import get_template, TemplateConfig

    console.print(Panel.fit(
        f"[bold]UCTS Template Generator[/bold]",
        border_style="blue"
    ))

    tpl = get_template(template_name)
    if not tpl:
        console.print(f"[red]Error: Template '{template_name}' not found[/red]")
        console.print("Run [cyan]ucts templates[/cyan] to see available templates")
        return

    project_name = name or Path(output).name

    console.print(f"\n[bold]Template:[/bold] {tpl.display_name}")
    console.print(f"[bold]Language:[/bold] {tpl.language}")
    console.print(f"[bold]Project:[/bold] {project_name}")

    config = TemplateConfig(
        project_name=project_name,
        author=author,
        description=description,
    )

    console.print(f"\n[bold]Generating files...[/bold]")
    created = tpl.write_to(output, config)

    console.print(f"\n[bold green]Created {len(created)} files in {output}[/bold green]")

    # Show next steps
    console.print(f"\n[bold]Next steps:[/bold]")
    console.print(f"  cd {output}")
    if tpl.language == 'python':
        console.print("  pip install -r requirements.txt")
    else:
        console.print("  npm install")


@cli.command("templates")
def list_templates():
    """List available project templates."""
    from ucts.templates import list_templates

    console.print(Panel.fit(
        "[bold]Available Templates[/bold]",
        border_style="blue"
    ))

    templates = list_templates()

    table = Table()
    table.add_column("Name", style="cyan")
    table.add_column("Description", style="white")
    table.add_column("Language", style="green")
    table.add_column("Tags", style="dim")

    for tpl in templates:
        table.add_row(
            tpl["name"],
            tpl["description"],
            tpl["language"],
            ", ".join(tpl["tags"][:3])
        )

    console.print(table)
    console.print("\n[dim]Use: ucts template <name> <output-path>[/dim]")


@cli.command()
def version():
    """Show UCTS version information."""
    from ucts import __version__

    console.print(Panel.fit(
        f"[bold]UCTS[/bold] - Universal Context Transfer System\n"
        f"Version: {__version__}\n\n"
        f"Turn any AI conversation into a deployable project.",
        border_style="blue"
    ))


# Ecosystem Integration Commands

@cli.group()
def ecosystem():
    """Coherence ecosystem integration commands."""
    pass


@ecosystem.command("status")
def ecosystem_status():
    """Show ecosystem integration status."""
    from ucts.integrations import get_ecosystem

    console.print(Panel.fit(
        "[bold]Coherence Ecosystem Status[/bold]",
        border_style="blue"
    ))

    eco = get_ecosystem()
    eco.initialize()

    status = eco.get_ecosystem_status()

    # UCTS status
    ucts = status.get("ucts", {})
    console.print(f"\n[bold cyan]UCTS[/bold cyan]")
    console.print(f"  Status: [green]{ucts.get('status', 'unknown')}[/green]")
    console.print(f"  Version: {ucts.get('version', 'unknown')}")
    console.print(f"  Mode: {ucts.get('mode', 'file')}")

    # Integrations
    console.print(f"\n[bold cyan]Integrations[/bold cyan]")

    table = Table()
    table.add_column("Project", style="cyan")
    table.add_column("Available", style="green")
    table.add_column("Mode", style="dim")
    table.add_column("Last Sync", style="dim")

    for name, info in status.get("integrations", {}).items():
        available = "[green]Yes[/green]" if info.get("available") else "[red]No[/red]"
        table.add_row(
            name.upper(),
            available,
            info.get("mode", "file"),
            info.get("last_sync", "Never")[:19] if info.get("last_sync") else "Never"
        )

    console.print(table)


@ecosystem.command("health")
def ecosystem_health():
    """Check health of all ecosystem integrations."""
    from ucts.integrations import get_ecosystem

    console.print(Panel.fit(
        "[bold]Ecosystem Health Check[/bold]",
        border_style="blue"
    ))

    eco = get_ecosystem()
    health = eco.health_check()

    status_icon = "[green]HEALTHY[/green]" if health.get("healthy") else "[red]UNHEALTHY[/red]"
    console.print(f"\nOverall: {status_icon}")
    console.print(f"Mode: {health.get('mode', 'file')}")

    console.print(f"\n[bold]Integration Health:[/bold]")
    for name, info in health.get("integrations", {}).items():
        status = info.get("status", "unknown")
        icon = "[green]+[/green]" if status != "error" else "[red]x[/red]"
        console.print(f"  {icon} {name.upper()}: {status}")
        if info.get("error"):
            console.print(f"      [red]Error: {info.get('error')}[/red]")


@ecosystem.command("register")
def ecosystem_register():
    """Register UCTS with ecosystem services."""
    from ucts.integrations import get_ecosystem

    console.print(Panel.fit(
        "[bold]Registering UCTS with Ecosystem[/bold]",
        border_style="blue"
    ))

    eco = get_ecosystem()
    eco.initialize()

    # Register with DanAI
    if eco.danai:
        console.print("\n[bold]Registering with DanAI agent marketplace...[/bold]")
        result = eco.danai.register_ucts_agents()
        console.print(f"  Registered {result.get('agent_count', 0)} agents")
        for agent in result.get('agents', [])[:5]:
            console.print(f"    [cyan]{agent}[/cyan]")
        if len(result.get('agents', [])) > 5:
            console.print(f"    ... and {len(result.get('agents', [])) - 5} more")

    # Report health to GCU
    if eco.gcu:
        console.print("\n[bold]Reporting health to GCU...[/bold]")
        result = eco.gcu.report_health("healthy", {"registered": True})
        console.print(f"  [green]Health reported: {result.get('status')}[/green]")

    console.print("\n[bold green]Registration complete![/bold green]")


@ecosystem.command("scan")
@click.argument('source', type=click.Path(exists=True))
def ecosystem_scan(source):
    """Security scan a conversation or project file."""
    from ucts.ingestion import get_ingester
    from ucts.integrations import SentinelIntegration

    console.print(Panel.fit(
        "[bold]Sentinel Security Scan[/bold]",
        border_style="yellow"
    ))

    # Ingest the source
    console.print(f"\n[bold]Scanning:[/bold] {source}")

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    # Collect code for scanning
    code_blocks = {}
    for i, block in enumerate(session.code_blocks):
        filename = f"block_{i}.{block.language or 'txt'}"
        code_blocks[filename] = block.code

    # Run security scan
    sentinel = SentinelIntegration()
    result = sentinel.scan_project_security(code_blocks)

    # Display results
    if result.passed:
        console.print(f"\n[bold green]PASSED[/bold green] - No critical/high issues found")
    else:
        console.print(f"\n[bold red]FAILED[/bold red] - {len(result.violations)} violation(s) found")

    console.print(f"Scanned: {result.scanned_files} code blocks in {result.scan_duration_ms}ms")

    if result.violations:
        console.print(f"\n[bold red]Violations ({len(result.violations)}):[/bold red]")
        for v in result.violations[:10]:
            console.print(f"  [{v['severity']}] {v['name']} (line {v['line']})")
            console.print(f"      {v['description']}")
            console.print(f"      [dim]{v['recommendation']}[/dim]")

    if result.warnings:
        console.print(f"\n[bold yellow]Warnings ({len(result.warnings)}):[/bold yellow]")
        for w in result.warnings[:5]:
            console.print(f"  [{w['severity']}] {w['name']} (line {w['line']})")
            console.print(f"      {w['description']}")


@ecosystem.command("store")
@click.argument('source', type=click.Path(exists=True))
@click.option('--tags', '-t', multiple=True, help='Tags for the session')
def ecosystem_store(source, tags):
    """Store a forge session in ecosystem memory."""
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.integrations import get_ecosystem
    from datetime import datetime

    console.print(Panel.fit(
        "[bold]Storing Session in Ecosystem Memory[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print(f"\n[bold]Processing:[/bold] {source}")

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    # Store in MMB
    eco = get_ecosystem()
    eco.initialize()

    if eco.mmb:
        session_id = f"session-{datetime.now().strftime('%Y%m%d%H%M%S')}"

        # Detect patterns
        patterns = []
        for block in session.code_blocks:
            if "class " in block.code:
                patterns.append("class-definition")
            if "def " in block.code or "function " in block.code:
                patterns.append("function-definition")
            if "import " in block.code or "require(" in block.code:
                patterns.append("imports")

        result = eco.mmb.store_session(
            session_id=session_id,
            summary=structure.description or f"Project {structure.name}",
            languages=structure.languages,
            dependencies=structure.dependencies,
            code_blocks=len(session.code_blocks),
            patterns_detected=list(set(patterns)),
            tags=list(tags)
        )

        console.print(f"\n[bold green]Session stored![/bold green]")
        console.print(f"  Session ID: [cyan]{result.get('session_id')}[/cyan]")
        console.print(f"  Languages: {', '.join(structure.languages)}")
        console.print(f"  Code blocks: {len(session.code_blocks)}")
        console.print(f"  Patterns: {', '.join(set(patterns)) or 'None detected'}")

        # Extract and store code patterns
        console.print(f"\n[bold]Extracting code patterns...[/bold]")
        patterns_stored = eco.mmb.extract_patterns_from_session(session)
        console.print(f"  Stored {len(patterns_stored)} patterns")
    else:
        console.print("[yellow]MMB integration not available[/yellow]")


@ecosystem.command("search")
@click.argument('query')
@click.option('--language', '-l', help='Filter by language')
@click.option('--limit', '-n', default=10, help='Maximum results')
def ecosystem_search(query, language, limit):
    """Search ecosystem memory for sessions and patterns."""
    from ucts.integrations import get_ecosystem

    console.print(Panel.fit(
        f"[bold]Searching Ecosystem:[/bold] {query}",
        border_style="blue"
    ))

    eco = get_ecosystem()
    eco.initialize()

    if not eco.mmb:
        console.print("[yellow]MMB integration not available[/yellow]")
        return

    # Search sessions
    console.print(f"\n[bold cyan]Sessions:[/bold cyan]")
    sessions = eco.mmb.search_sessions(
        query=query,
        languages=[language] if language else None,
        limit=limit
    )

    if sessions:
        for s in sessions:
            console.print(f"  [cyan]{s.get('session_id')}[/cyan]")
            console.print(f"    {s.get('summary', 'No summary')[:60]}...")
            console.print(f"    Languages: {', '.join(s.get('languages', []))}")
    else:
        console.print("  [dim]No matching sessions[/dim]")

    # Search patterns
    console.print(f"\n[bold cyan]Code Patterns:[/bold cyan]")
    patterns = eco.mmb.search_patterns(
        query=query,
        language=language,
        limit=limit
    )

    if patterns:
        for p in patterns:
            console.print(f"  [cyan]{p.get('name')}[/cyan] ({p.get('language')})")
            console.print(f"    Type: {p.get('pattern_type')}")
            console.print(f"    {p.get('description', 'No description')[:50]}...")
    else:
        console.print("  [dim]No matching patterns[/dim]")


@ecosystem.command("metrics")
def ecosystem_metrics():
    """Show UCTS metrics from ecosystem dashboard."""
    from ucts.integrations import get_ecosystem

    console.print(Panel.fit(
        "[bold]UCTS Ecosystem Metrics[/bold]",
        border_style="blue"
    ))

    eco = get_ecosystem()
    eco.initialize()

    if not eco.wb2:
        console.print("[yellow]WB2 integration not available[/yellow]")
        return

    metrics = eco.wb2.get_metrics()
    charts = eco.wb2.get_chart_data()

    # Summary metrics
    table = Table(title="Forge Statistics")
    table.add_column("Metric", style="cyan")
    table.add_column("Value", style="green")

    table.add_row("Total Forges", str(metrics.get("total_forges", 0)))
    table.add_row("Successful", str(metrics.get("successful_forges", 0)))
    table.add_row("Failed", str(metrics.get("failed_forges", 0)))
    table.add_row("Success Rate", f"{metrics.get('success_rate', 0):.1f}%")
    table.add_row("Files Created", str(metrics.get("total_files_created", 0)))
    table.add_row("Avg Duration", f"{metrics.get('avg_duration_seconds', 0):.2f}s")

    console.print(table)

    # Language breakdown
    langs = metrics.get("languages_used", {})
    if langs:
        console.print(f"\n[bold]Languages Used:[/bold]")
        for lang, count in sorted(langs.items(), key=lambda x: -x[1])[:5]:
            bar = "[green]" + "=" * min(count, 20) + "[/green]"
            console.print(f"  {lang:12} {bar} {count}")

    # Target breakdown
    targets = metrics.get("targets_used", {})
    if targets:
        console.print(f"\n[bold]Targets Used:[/bold]")
        for target, count in sorted(targets.items(), key=lambda x: -x[1]):
            console.print(f"  {target}: {count}")


# Cloud Deployment Commands

@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--platform', '-p',
              type=click.Choice(['auto', 'aws', 'gcp', 'azure', 'vercel', 'railway']),
              default='auto',
              help='Cloud platform (auto-detect by default)')
@click.option('--region', '-r', default=None, help='Deployment region')
@click.option('--memory', '-m', default=512, help='Memory allocation in MB')
@click.option('--timeout', default=30, help='Function timeout in seconds')
def deploy(source, output, platform, region, memory, timeout):
    """Generate cloud deployment configuration.

    SOURCE is the conversation file path.
    OUTPUT is the destination directory for deployment files.

    Supported platforms:
      - aws: CloudFormation/SAM templates
      - gcp: Terraform + Cloud Run
      - azure: ARM templates
      - vercel: vercel.json + functions
      - railway: railway.json
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators.cloud import CloudGenerator, CloudConfig

    console.print(Panel.fit(
        "[bold]UCTS Cloud Deployment Generator[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print("\n[bold]Step 1:[/bold] Analyzing project...")
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    console.print(f"  [green]Project: {structure.name}[/green]")
    console.print(f"  [green]Languages: {', '.join(structure.languages)}[/green]")

    # Detect platform if auto
    detected_platform = platform
    if platform == 'auto':
        # Simple heuristics
        has_python = 'python' in [l.lower() for l in structure.languages]
        has_node = any(l.lower() in ['javascript', 'typescript'] for l in structure.languages)

        if has_python:
            detected_platform = 'aws'  # Lambda is popular for Python
        elif has_node:
            detected_platform = 'vercel'  # Vercel is popular for Node
        else:
            detected_platform = 'aws'

        console.print(f"  [yellow]Auto-detected platform: {detected_platform}[/yellow]")

    # Generate deployment files
    console.print(f"\n[bold]Step 2:[/bold] Generating {detected_platform.upper()} deployment configuration...")

    config = CloudConfig(
        region=region or "us-east-1",
        memory_mb=memory,
        timeout_seconds=timeout,
    )

    generator = CloudGenerator(config)
    files = generator.generate(structure, output, detected_platform)

    console.print(f"\n[bold green]Deployment files created in {output}:[/bold green]")
    for filename in sorted(files.keys()):
        console.print(f"  [cyan]{filename}[/cyan]")

    # Platform-specific next steps
    console.print(f"\n[bold]Next steps for {detected_platform.upper()}:[/bold]")
    if detected_platform == 'aws':
        console.print("  sam build && sam deploy --guided")
    elif detected_platform == 'gcp':
        console.print("  terraform init && terraform apply")
    elif detected_platform == 'azure':
        console.print("  az deployment group create -g myResourceGroup -f azuredeploy.json")
    elif detected_platform == 'vercel':
        console.print("  vercel deploy")
    elif detected_platform == 'railway':
        console.print("  railway up")


@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--namespace', '-n', default='default', help='Kubernetes namespace')
@click.option('--replicas', '-r', default=2, help='Number of replicas')
@click.option('--port', '-p', default=8000, help='Container port')
@click.option('--registry', default='', help='Container registry (e.g., gcr.io/myproject)')
@click.option('--ingress/--no-ingress', default=True, help='Generate Ingress resource')
@click.option('--hpa/--no-hpa', default=True, help='Generate HorizontalPodAutoscaler')
def k8s(source, output, namespace, replicas, port, registry, ingress, hpa):
    """Generate Kubernetes manifests from a conversation.

    SOURCE is the conversation file path.
    OUTPUT is the destination directory for K8s manifests.

    Generates: Deployment, Service, ConfigMap, Secret, Ingress, HPA, Kustomization.
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators.kubernetes import KubernetesGenerator, KubernetesConfig

    console.print(Panel.fit(
        "[bold]UCTS Kubernetes Generator[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print("\n[bold]Step 1:[/bold] Analyzing project...")
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    console.print(f"  [green]Project: {structure.name}[/green]")
    console.print(f"  [green]Languages: {', '.join(structure.languages)}[/green]")

    # Generate K8s manifests
    console.print(f"\n[bold]Step 2:[/bold] Generating Kubernetes manifests...")

    config = KubernetesConfig(
        namespace=namespace,
        replicas=replicas,
        container_port=port,
        registry=registry,
        generate_ingress=ingress,
        generate_hpa=hpa,
    )

    generator = KubernetesGenerator(config)
    files = generator.generate(structure, output)

    console.print(f"\n[bold green]Kubernetes manifests created in {output}:[/bold green]")
    for filename in sorted(files.keys()):
        console.print(f"  [cyan]{filename}[/cyan]")

    console.print(f"\n[bold]Apply with:[/bold]")
    console.print(f"  kubectl apply -k {output}")
    console.print(f"\nOr apply individual files:")
    console.print(f"  kubectl apply -f {output}/")


@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--name', '-n', default=None, help='Chart name (defaults to project name)')
@click.option('--version', '-v', default='0.1.0', help='Chart version')
@click.option('--app-version', default='1.0.0', help='Application version')
@click.option('--namespace', default='default', help='Default namespace')
@click.option('--replicas', '-r', default=2, help='Default replicas')
def helm(source, output, name, version, app_version, namespace, replicas):
    """Generate a Helm chart from a conversation.

    SOURCE is the conversation file path.
    OUTPUT is the destination directory for the Helm chart.

    Generates a complete Helm chart with templates, values.yaml, and Chart.yaml.
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators.kubernetes import HelmGenerator, HelmConfig

    console.print(Panel.fit(
        "[bold]UCTS Helm Chart Generator[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print("\n[bold]Step 1:[/bold] Analyzing project...")
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    chart_name = name or structure.name

    console.print(f"  [green]Chart: {chart_name}[/green]")
    console.print(f"  [green]Languages: {', '.join(structure.languages)}[/green]")

    # Generate Helm chart
    console.print(f"\n[bold]Step 2:[/bold] Generating Helm chart...")

    config = HelmConfig(
        chart_version=version,
        app_version=app_version,
        namespace=namespace,
        replicas=replicas,
    )

    generator = HelmGenerator(config)
    files = generator.generate(structure, output, chart_name)

    console.print(f"\n[bold green]Helm chart created in {output}/{chart_name}/:[/bold green]")
    for filename in sorted(files.keys()):
        console.print(f"  [cyan]{filename}[/cyan]")

    console.print(f"\n[bold]Install with:[/bold]")
    console.print(f"  helm install {chart_name} {output}/{chart_name}")
    console.print(f"\nPreview templates:")
    console.print(f"  helm template {chart_name} {output}/{chart_name}")


@cli.command("test-gen")
@click.argument('source', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--framework', '-f',
              type=click.Choice(['auto', 'pytest', 'jest', 'vitest']),
              default='auto',
              help='Test framework (auto-detect by default)')
@click.option('--coverage', '-c', default=80, help='Coverage target percentage')
@click.option('--no-mocks', is_flag=True, help='Disable mock generation')
@click.option('--no-fixtures', is_flag=True, help='Disable fixture generation')
def test_gen(source, output, framework, coverage, no_mocks, no_fixtures):
    """Generate test files from a conversation.

    SOURCE is the conversation file path.
    OUTPUT is the destination directory for test files.

    Supported frameworks:
      - pytest: Python testing with pytest
      - jest: JavaScript/TypeScript with Jest
      - vitest: Modern Vite-native testing
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine
    from ucts.generators.tests import TestGenerator, TestConfig

    console.print(Panel.fit(
        "[bold]UCTS Test Generator[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print("\n[bold]Step 1:[/bold] Analyzing project...")
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    console.print(f"  [green]Project: {structure.name}[/green]")
    console.print(f"  [green]Languages: {', '.join(structure.languages)}[/green]")
    console.print(f"  [green]Source files: {len(structure.files)}[/green]")

    # Detect framework if auto
    detected_framework = framework
    if framework == 'auto':
        if 'python' in [l.lower() for l in structure.languages]:
            detected_framework = 'pytest'
        elif any(l.lower() in ['typescript', 'javascript'] for l in structure.languages):
            # Check for vitest config indicators
            has_vitest = any('vitest' in str(v).lower() for v in structure.dependencies.values())
            detected_framework = 'vitest' if has_vitest else 'jest'
        else:
            detected_framework = 'pytest'

        console.print(f"  [yellow]Auto-detected framework: {detected_framework}[/yellow]")

    # Generate tests
    console.print(f"\n[bold]Step 2:[/bold] Generating {detected_framework} test files...")

    config = TestConfig(
        framework=detected_framework,
        coverage_target=coverage,
        include_mocks=not no_mocks,
        include_fixtures=not no_fixtures,
    )

    generator = TestGenerator(config)
    files = generator.generate(structure, output, detected_framework)

    console.print(f"\n[bold green]Test files created in {output}:[/bold green]")
    for filename in sorted(files.keys()):
        console.print(f"  [cyan]{filename}[/cyan]")

    # Framework-specific next steps
    console.print(f"\n[bold]Run tests with:[/bold]")
    if detected_framework == 'pytest':
        console.print(f"  pytest {output}/tests -v --cov")
    elif detected_framework == 'jest':
        console.print("  npm test")
    elif detected_framework == 'vitest':
        console.print("  npm run test")

    # Show available frameworks
    console.print(f"\n[dim]Available frameworks: pytest, jest, vitest[/dim]")


# Real-Time Capture Commands

@cli.command()
@click.option('--source', '-s',
              type=click.Choice(['claude_api', 'openai_api', 'websocket', 'sse', 'file_watch']),
              default='file_watch',
              help='Capture source')
@click.option('--api-key', envvar='CAPTURE_API_KEY', help='API key for the source')
@click.option('--url', default=None, help='URL or file path to watch')
@click.option('--checkpoint-interval', '-c', default=30, help='Checkpoint interval in seconds')
@click.option('--output', '-o', default='.ucts/capture', help='Output directory for checkpoints')
def watch(source, api_key, url, checkpoint_interval, output):
    """Start real-time capture from AI assistant.

    Watch live AI conversations and capture them incrementally.

    Examples:
        ucts watch --source file_watch --url ./conversation.md
        ucts watch --source claude_api --api-key $ANTHROPIC_API_KEY
    """
    import asyncio
    from ucts.capture import RealtimeCapture, CaptureConfig, CaptureSource

    console.print(Panel.fit(
        "[bold]UCTS Real-Time Capture[/bold]",
        border_style="blue"
    ))

    source_enum = CaptureSource(source)

    config = CaptureConfig(
        source=source_enum,
        api_key=api_key,
        api_url=url,
        checkpoint_interval=checkpoint_interval,
        checkpoint_dir=output,
    )

    capture = RealtimeCapture(config)

    # Register callback to display messages
    def on_message(msg):
        role_color = "cyan" if msg.role == "user" else "green"
        console.print(f"[{role_color}]{msg.role}[/{role_color}]: {msg.content[:100]}...")

    capture.on_message(on_message)

    console.print(f"\n[bold]Source:[/bold] {source}")
    console.print(f"[bold]Checkpoint interval:[/bold] {checkpoint_interval}s")
    console.print(f"[bold]Output:[/bold] {output}")
    console.print("\n[yellow]Starting capture... Press Ctrl+C to stop[/yellow]\n")

    async def run_capture():
        try:
            session = await capture.start()
            console.print(f"[green]Session started: {session.session_id}[/green]")

            # Keep running until interrupted
            while session.state.value in ['streaming', 'paused']:
                await asyncio.sleep(1)

        except KeyboardInterrupt:
            console.print("\n[yellow]Stopping capture...[/yellow]")
        finally:
            session = await capture.stop()
            console.print(f"\n[bold green]Capture complete![/bold green]")
            console.print(f"  Messages: {len(session.messages)}")
            console.print(f"  Tokens: {session.total_tokens}")
            console.print(f"  Checkpoints: {len(session.checkpoints)}")

            # Analyze
            if session.messages:
                console.print("\n[bold]Analyzing captured content...[/bold]")
                analysis = await capture.analyze_incremental()
                console.print(f"  Languages: {', '.join(analysis.get('languages', []))}")
                console.print(f"  Code blocks: {analysis.get('code_block_count', 0)}")

    asyncio.run(run_capture())


# Bidirectional Sync Commands

@cli.group()
def sync():
    """Bidirectional sync commands for AI context management."""
    pass


@sync.command("capture")
@click.argument('project_path', type=click.Path(exists=True))
def sync_capture(project_path):
    """Capture current project state for sync tracking."""
    from ucts.sync import get_sync_manager

    console.print(Panel.fit(
        "[bold]Capturing Project State[/bold]",
        border_style="blue"
    ))

    manager = get_sync_manager()
    state = manager.capture_state(project_path)

    console.print(f"\n[bold green]State captured![/bold green]")
    console.print(f"  Files: {len(state.files)}")
    console.print(f"  Git: {state.git_branch or 'N/A'} @ {state.git_commit or 'N/A'}")
    console.print(f"  Dependencies: {sum(len(d) for d in state.dependencies.values())}")
    console.print(f"  Timestamp: {state.timestamp.isoformat()}")


@sync.command("diff")
@click.argument('project_path', type=click.Path(exists=True))
def sync_diff(project_path):
    """Show changes since last captured state."""
    from ucts.sync import get_sync_manager

    console.print(Panel.fit(
        "[bold]Project Diff[/bold]",
        border_style="blue"
    ))

    manager = get_sync_manager()

    # Load previous states
    states = manager.load_states()
    if not states:
        console.print("[yellow]No previous state found. Capturing initial state...[/yellow]")
        manager.capture_state(project_path)
        console.print("[green]Initial state captured. Run again after making changes.[/green]")
        return

    # Capture current state
    current = manager.capture_state(project_path)

    # Compute diff from last state
    diff = manager.compute_diff(states[-1], current)

    if not diff.has_changes:
        console.print("[green]No changes detected[/green]")
        return

    console.print(f"\n[bold]{diff.summary}[/bold]")

    if diff.files_added:
        console.print(f"\n[green]Added ({len(diff.files_added)}):[/green]")
        for f in diff.files_added[:10]:
            console.print(f"  + {f}")

    if diff.files_modified:
        console.print(f"\n[yellow]Modified ({len(diff.files_modified)}):[/yellow]")
        for f in diff.files_modified[:10]:
            console.print(f"  ~ {f}")

    if diff.files_deleted:
        console.print(f"\n[red]Deleted ({len(diff.files_deleted)}):[/red]")
        for f in diff.files_deleted[:10]:
            console.print(f"  - {f}")


@sync.command("context")
@click.argument('project_path', type=click.Path(exists=True))
@click.option('--output', '-o', type=click.Path(), help='Output file path')
@click.option('--max-tokens', '-t', default=4000, help='Maximum tokens for compressed output')
def sync_context(project_path, output, max_tokens):
    """Generate context summary for new AI sessions."""
    from ucts.sync import get_sync_manager

    console.print(Panel.fit(
        "[bold]Generating Context Summary[/bold]",
        border_style="blue"
    ))

    manager = get_sync_manager()
    summary = manager.generate_context_summary(project_path)

    console.print(f"\n[bold]Project:[/bold] {summary.project_name}")
    console.print(f"[bold]Languages:[/bold] {', '.join(summary.languages)}")
    console.print(f"[bold]Key Files:[/bold] {len(summary.key_files)}")
    console.print(f"[bold]Token Estimate:[/bold] ~{summary.token_count}")

    if output:
        Path(output).write_text(summary.to_markdown())
        console.print(f"\n[green]Saved to {output}[/green]")
    else:
        console.print("\n[bold]Summary Preview:[/bold]")
        console.print(summary.to_compressed(max_tokens)[:1000] + "...")


@sync.command("handoff")
@click.argument('source', type=click.Path(exists=True))
@click.argument('project_path', type=click.Path(exists=True))
@click.argument('output', type=click.Path())
@click.option('--summary', '-s', default='', help='Conversation summary')
@click.option('--format', '-f', type=click.Choice(['markdown', 'json', 'prompt']), default='markdown')
def sync_handoff(source, project_path, output, summary, format):
    """Create handoff document for session continuation."""
    from ucts.ingestion import get_ingester
    from ucts.sync import get_sync_manager

    console.print(Panel.fit(
        "[bold]Creating Handoff Document[/bold]",
        border_style="blue"
    ))

    # Ingest session
    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    # Create handoff
    manager = get_sync_manager()
    handoff = manager.create_handoff(
        session=session,
        project_path=project_path,
        conversation_summary=summary or f"Session with {len(session.messages)} messages"
    )

    # Export
    output_file = manager.export_for_resume(handoff, output, format)

    console.print(f"\n[bold green]Handoff document created![/bold green]")
    console.print(f"  Session: {handoff.session_id}")
    console.print(f"  Format: {format}")
    console.print(f"  Output: {output_file}")
    console.print(f"  Decisions: {len(handoff.key_decisions)}")
    console.print(f"  Pending tasks: {len(handoff.pending_tasks)}")


# Code Enhancement Commands

@cli.command()
@click.argument('source', type=click.Path(exists=True))
@click.option('--output', '-o', type=click.Path(), help='Output report path')
@click.option('--category', '-c', multiple=True,
              type=click.Choice(['error_handling', 'security', 'documentation',
                               'type_safety', 'performance', 'code_quality',
                               'completeness', 'best_practices']),
              help='Categories to check (default: all)')
@click.option('--min-severity', '-s',
              type=click.Choice(['info', 'warning', 'error', 'critical']),
              default='info',
              help='Minimum severity to report')
def enhance(source, output, category, min_severity):
    """Analyze code for potential enhancements.

    Detects missing error handling, security issues, documentation gaps,
    and other improvement opportunities.
    """
    from ucts.ingestion import get_ingester
    from ucts.analysis import AnalysisEngine, CodeEnhancer, EnhancementConfig, EnhancementCategory, Severity

    console.print(Panel.fit(
        "[bold]UCTS Code Enhancement Analysis[/bold]",
        border_style="blue"
    ))

    # Ingest and analyze
    console.print(f"\n[bold]Analyzing:[/bold] {source}")

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    engine = AnalysisEngine()
    structure = engine.analyze(session)

    console.print(f"  Files: {len(structure.files)}")
    console.print(f"  Languages: {', '.join(structure.languages)}")

    # Configure enhancer
    categories = [EnhancementCategory(c) for c in category] if category else list(EnhancementCategory)

    config = EnhancementConfig(
        enabled_categories=categories,
        min_severity=Severity(min_severity),
    )

    # Run enhancement analysis
    console.print(f"\n[bold]Running enhancement analysis...[/bold]")

    enhancer = CodeEnhancer(config)
    report = enhancer.analyze_project(structure, output)

    # Display results
    console.print(f"\n[bold]Enhancement Report[/bold]")
    console.print(f"  Files analyzed: {report.files_analyzed}")
    console.print(f"  Total suggestions: {report.total_suggestions}")

    if report.by_severity:
        console.print(f"\n[bold]By Severity:[/bold]")
        for sev, count in sorted(report.by_severity.items()):
            icon = {"critical": "🚨", "error": "❌", "warning": "⚠️", "info": "ℹ️"}.get(sev, "")
            console.print(f"  {icon} {sev}: {count}")

    if report.by_category:
        console.print(f"\n[bold]By Category:[/bold]")
        for cat, count in sorted(report.by_category.items(), key=lambda x: -x[1]):
            console.print(f"  {cat}: {count}")

    # Show top issues
    if report.enhancements:
        console.print(f"\n[bold]Top Issues:[/bold]")
        for e in report.enhancements[:10]:
            severity_color = {
                "critical": "red",
                "error": "red",
                "warning": "yellow",
                "info": "dim"
            }.get(e.severity.value, "white")

            console.print(f"\n  [{severity_color}][{e.severity.value.upper()}][/{severity_color}] {e.title}")
            console.print(f"    {e.location}")
            console.print(f"    {e.description[:80]}...")
            if e.suggestion:
                console.print(f"    [cyan]→ {e.suggestion[:60]}...[/cyan]")

    if output:
        console.print(f"\n[green]Full report saved to {output}[/green]")


# SDK Mode Commands

@cli.group()
def sdk():
    """Ecosystem SDK mode commands."""
    pass


@sdk.command("status")
def sdk_status():
    """Show SDK mode status."""
    from ucts.integrations import get_ecosystem_sdk

    console.print(Panel.fit(
        "[bold]Ecosystem SDK Status[/bold]",
        border_style="blue"
    ))

    sdk_instance = get_ecosystem_sdk()
    status = sdk_instance.get_status()

    table = Table()
    table.add_column("Property", style="cyan")
    table.add_column("Value", style="green")

    for key, value in status.items():
        table.add_row(key.replace("_", " ").title(), str(value))

    console.print(table)


@sdk.command("init")
@click.option('--mode', '-m',
              type=click.Choice(['file', 'network', 'hybrid']),
              default='file',
              help='SDK mode')
@click.option('--url', envvar='COHERENCE_URL', help='Coherence server URL')
@click.option('--api-key', envvar='COHERENCE_API_KEY', help='Coherence API key')
def sdk_init(mode, url, api_key):
    """Initialize SDK connection."""
    import asyncio
    from ucts.integrations import SDKConfig, SDKMode, initialize_sdk

    console.print(Panel.fit(
        "[bold]Initializing Ecosystem SDK[/bold]",
        border_style="blue"
    ))

    config = SDKConfig(
        mode=SDKMode(mode),
        coherence_url=url or "",
        api_key=api_key or "",
    )

    async def init():
        sdk_instance = await initialize_sdk(config)
        return sdk_instance.get_status()

    status = asyncio.run(init())

    console.print(f"\n[bold green]SDK initialized![/bold green]")
    console.print(f"  Mode: {status['mode']}")
    console.print(f"  Connected: {status['connected']}")
    console.print(f"  Agent ID: {status['agent_id']}")


@sdk.command("broadcast")
@click.argument('event_type')
@click.option('--data', '-d', default='{}', help='Event data as JSON')
def sdk_broadcast(event_type, data):
    """Broadcast an event to the ecosystem."""
    import asyncio
    import json
    from ucts.integrations import get_ecosystem_sdk, Event, EventType

    console.print(Panel.fit(
        f"[bold]Broadcasting Event: {event_type}[/bold]",
        border_style="blue"
    ))

    try:
        event_data = json.loads(data)
    except json.JSONDecodeError:
        console.print("[red]Invalid JSON data[/red]")
        return

    async def broadcast():
        sdk_instance = get_ecosystem_sdk()
        await sdk_instance.initialize()

        # Try to match event type
        try:
            etype = EventType(event_type)
        except ValueError:
            etype = EventType.METRICS_RECORDED

        event = Event(
            event_type=etype,
            source="ucts-cli",
            data=event_data,
        )

        success = await sdk_instance.broadcast_event(event)
        return success

    success = asyncio.run(broadcast())

    if success:
        console.print(f"[green]Event broadcasted successfully[/green]")
    else:
        console.print(f"[yellow]Event buffered (no network connection)[/yellow]")


@sdk.command("health")
def sdk_health():
    """Report health to ecosystem."""
    import asyncio
    from ucts.integrations import get_ecosystem_sdk

    console.print(Panel.fit(
        "[bold]Reporting Health[/bold]",
        border_style="blue"
    ))

    async def report():
        sdk_instance = get_ecosystem_sdk()
        await sdk_instance.initialize()
        return await sdk_instance.report_health()

    health = asyncio.run(report())

    console.print(f"\n[bold green]Health reported![/bold green]")
    for key, value in health.items():
        console.print(f"  {key}: {value}")


# AI Agent Mode Commands

@cli.command()
@click.argument('goal')
@click.option('--output', '-o', type=click.Path(), default='./agent-output', help='Output directory')
@click.option('--model', '-m', default='claude', help='LLM model to use')
@click.option('--max-iterations', '-i', default=10, help='Maximum iterations')
@click.option('--auto-approve', is_flag=True, help='Auto-approve all gates (not recommended)')
@click.option('--test-driven', is_flag=True, default=True, help='Use test-driven development')
def agent(goal, output, model, max_iterations, auto_approve, test_driven):
    """Start autonomous AI agent to build a project.

    GOAL is the project description/goal to achieve.

    Examples:
        ucts agent "Build a todo app with user auth"
        ucts agent "Create a REST API for inventory management" -o ./inventory
    """
    import asyncio
    from ucts.agent import AgentMode, AgentConfig

    console.print(Panel.fit(
        "[bold]UCTS AI Agent Mode[/bold]\nAutonomous project generation",
        border_style="blue"
    ))

    config = AgentConfig(
        model=model,
        max_iterations=max_iterations,
        output_dir=output,
        test_driven=test_driven,
        auto_approve=auto_approve,
    )

    agent_mode = AgentMode(config)

    console.print(f"\n[bold]Goal:[/bold] {goal}")
    console.print(f"[bold]Output:[/bold] {output}")
    console.print(f"[bold]Max iterations:[/bold] {max_iterations}")
    console.print(f"[bold]Test-driven:[/bold] {test_driven}")

    if not auto_approve:
        console.print("\n[yellow]Human approval gates are enabled.[/yellow]")
        console.print("[dim]You will be prompted at key decision points.[/dim]\n")

    async def run_agent():
        try:
            session = await agent_mode.start(goal)

            console.print(f"\n[bold green]Agent session complete![/bold green]")
            console.print(f"  Session ID: {session.session_id}")
            console.print(f"  Steps completed: {len(session.steps)}")
            console.print(f"  Files created: {len(session.files_created)}")
            console.print(f"  Status: {session.status}")

            if session.files_created:
                console.print(f"\n[bold]Files created:[/bold]")
                for f in session.files_created[:10]:
                    console.print(f"  [cyan]{f}[/cyan]")
                if len(session.files_created) > 10:
                    console.print(f"  ... and {len(session.files_created) - 10} more")

        except KeyboardInterrupt:
            console.print("\n[yellow]Agent interrupted by user[/yellow]")
        except Exception as e:
            console.print(f"\n[red]Agent error: {e}[/red]")

    asyncio.run(run_agent())


# Plugin System Commands

@cli.group()
def plugins():
    """Plugin system commands."""
    pass


@plugins.command("list")
def plugins_list():
    """List all loaded plugins."""
    from ucts.plugins import get_plugin_manager

    console.print(Panel.fit(
        "[bold]UCTS Plugins[/bold]",
        border_style="blue"
    ))

    manager = get_plugin_manager()
    manager.load_all_plugins()

    loaded = manager.list_plugins()

    if not loaded:
        console.print("[dim]No plugins loaded[/dim]")
        console.print(f"\nPlugin directories:")
        for d in manager.plugin_dirs:
            console.print(f"  {d}")
        return

    table = Table(title="Loaded Plugins")
    table.add_column("Name", style="cyan")
    table.add_column("Version", style="green")
    table.add_column("Type", style="dim")
    table.add_column("Enabled", style="green")

    for plugin in loaded:
        table.add_row(
            plugin.name,
            plugin.version,
            plugin.plugin_type.value,
            "[green]Yes[/green]" if plugin.enabled else "[red]No[/red]"
        )

    console.print(table)


@plugins.command("discover")
def plugins_discover():
    """Discover available plugins."""
    from ucts.plugins import get_plugin_manager

    console.print(Panel.fit(
        "[bold]Discovering Plugins[/bold]",
        border_style="blue"
    ))

    manager = get_plugin_manager()

    console.print("\n[bold]Plugin directories:[/bold]")
    for d in manager.plugin_dirs:
        console.print(f"  {d}")

    discovered = manager.discover_plugins()

    if not discovered:
        console.print("\n[dim]No plugins found[/dim]")
        return

    console.print(f"\n[bold]Found {len(discovered)} plugins:[/bold]")
    for plugin in discovered:
        console.print(f"  [cyan]{plugin.name}[/cyan] v{plugin.version} ({plugin.plugin_type.value})")


@plugins.command("create")
@click.argument('name')
@click.option('--type', '-t', 'plugin_type',
              type=click.Choice(['ingester', 'generator', 'analyzer', 'hook']),
              default='hook',
              help='Plugin type')
@click.option('--output', '-o', type=click.Path(), default='.', help='Output directory')
def plugins_create(name, plugin_type, output):
    """Create a new plugin template."""
    from ucts.plugins import get_plugin_manager, PluginType

    console.print(Panel.fit(
        f"[bold]Creating Plugin: {name}[/bold]",
        border_style="blue"
    ))

    manager = get_plugin_manager()
    ptype = PluginType(plugin_type)

    files = manager.create_plugin_template(name, ptype, output)

    console.print(f"\n[bold green]Plugin template created![/bold green]")
    for filename, path in files.items():
        console.print(f"  [cyan]{filename}[/cyan]: {path}")

    console.print(f"\n[bold]Next steps:[/bold]")
    console.print(f"  1. Edit {output}/{name}/__init__.py to customize")
    console.print(f"  2. Copy to ~/.ucts/plugins/")
    console.print(f"  3. Run 'ucts plugins list' to verify")


# Analytics Commands

@cli.group()
def analytics():
    """Conversation analytics commands."""
    pass


@analytics.command("analyze")
@click.argument('source', type=click.Path(exists=True))
@click.option('--model', '-m', default='default', help='Pricing model for cost estimation')
@click.option('--output', '-o', type=click.Path(), help='Output report path')
def analytics_analyze(source, model, output):
    """Analyze a conversation for usage and quality metrics."""
    from ucts.ingestion import get_ingester
    from ucts.analytics import get_analytics, AnalyticsConfig

    console.print(Panel.fit(
        "[bold]Conversation Analytics[/bold]",
        border_style="blue"
    ))

    # Ingest
    console.print(f"\n[bold]Analyzing:[/bold] {source}")

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    # Analyze
    config = AnalyticsConfig(model=model)
    analytics_engine = get_analytics(config)
    report = analytics_engine.analyze_session(session)

    # Display results
    console.print(f"\n[bold]Usage Metrics[/bold]")
    console.print(f"  Total tokens: {report.usage.total_tokens:,}")
    console.print(f"  Input tokens: {report.usage.total_input_tokens:,}")
    console.print(f"  Output tokens: {report.usage.total_output_tokens:,}")
    console.print(f"  Messages: {report.usage.message_count}")

    console.print(f"\n[bold]Cost Estimate ({model})[/bold]")
    console.print(f"  Input cost: ${report.costs.input_cost:.4f}")
    console.print(f"  Output cost: ${report.costs.output_cost:.4f}")
    console.print(f"  [bold]Total: ${report.costs.total_cost:.4f}[/bold]")

    console.print(f"\n[bold]Quality Metrics[/bold]")
    console.print(f"  Clarity: {report.quality.clarity_score:.0%}")
    console.print(f"  Efficiency: {report.quality.efficiency_score:.0%}")
    console.print(f"  Success rate: {report.quality.success_rate:.0%}")
    console.print(f"  [bold]Overall: {report.quality.overall_score():.0%}[/bold]")

    if report.patterns.language_distribution:
        console.print(f"\n[bold]Languages:[/bold]")
        for lang, count in report.patterns.language_distribution.items():
            console.print(f"  {lang}: {count}")

    if report.patterns.common_issues:
        console.print(f"\n[bold yellow]Issues detected:[/bold yellow]")
        for issue in report.patterns.common_issues[:5]:
            console.print(f"  - {issue}")

    if output:
        Path(output).write_text(report.to_markdown())
        console.print(f"\n[green]Report saved to {output}[/green]")


@analytics.command("history")
@click.option('--days', '-d', default=30, help='Number of days to look back')
def analytics_history(days):
    """View historical analytics."""
    from ucts.analytics import get_analytics

    console.print(Panel.fit(
        f"[bold]Analytics History ({days} days)[/bold]",
        border_style="blue"
    ))

    analytics_engine = get_analytics()
    report = analytics_engine.get_historical_report(days=days)

    if "message" in report:
        console.print(f"[dim]{report['message']}[/dim]")
        return

    summary = report.get("summary", {})
    console.print(f"\n[bold]Summary[/bold]")
    console.print(f"  Sessions: {summary.get('total_sessions', 0)}")
    console.print(f"  Total tokens: {summary.get('total_tokens', 0):,}")
    console.print(f"  Total cost: ${summary.get('total_cost', 0):.2f}")
    console.print(f"  Avg quality: {summary.get('avg_quality', 0):.0%}")

    daily = report.get("daily", {})
    if daily:
        console.print(f"\n[bold]Recent Days:[/bold]")
        for day, stats in sorted(daily.items(), reverse=True)[:7]:
            console.print(f"  {day}: {stats.get('sessions', 0)} sessions, ${stats.get('cost', 0):.2f}")


@analytics.command("cost")
@click.argument('source', type=click.Path(exists=True))
@click.option('--project-days', '-d', default=30, help='Days to project costs')
def analytics_cost(source, project_days):
    """Estimate and project costs for a conversation."""
    from ucts.ingestion import get_ingester
    from ucts.analytics import get_analytics

    console.print(Panel.fit(
        "[bold]Cost Analysis & Projection[/bold]",
        border_style="blue"
    ))

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    analytics_engine = get_analytics()
    report = analytics_engine.analyze_session(session)

    projection = analytics_engine.get_cost_projection(report.usage, project_days)

    console.print(f"\n[bold]Current Session[/bold]")
    console.print(f"  Tokens: {projection['current_daily']['tokens']:,}")
    console.print(f"  Cost: ${projection['current_daily']['cost']:.4f}")

    console.print(f"\n[bold]{project_days}-Day Projection[/bold]")
    console.print(f"  Tokens: {projection['projection']['total_tokens']:,}")
    console.print(f"  Cost: ${projection['projection']['total_cost']:.2f}")

    console.print(f"\n[bold]Estimates[/bold]")
    console.print(f"  Monthly: ${projection['monthly_estimate']['cost']:.2f}")
    console.print(f"  Yearly: ${projection['yearly_estimate']['cost']:.2f}")


# Team Commands

@cli.group()
def team():
    """Team and enterprise feature commands."""
    pass


@team.command("status")
def team_status():
    """Show team status."""
    from ucts.team import get_team_manager

    console.print(Panel.fit(
        "[bold]Team Status[/bold]",
        border_style="blue"
    ))

    manager = get_team_manager()
    status = manager.get_team_status()

    table = Table()
    table.add_column("Property", style="cyan")
    table.add_column("Value", style="green")

    for key, value in status.items():
        table.add_row(key.replace("_", " ").title(), str(value))

    console.print(table)


@team.command("add-member")
@click.argument('user_id')
@click.option('--name', '-n', required=True, help='Member name')
@click.option('--email', '-e', required=True, help='Member email')
@click.option('--role', '-r',
              type=click.Choice(['viewer', 'member', 'contributor', 'admin']),
              default='member',
              help='Member role')
def team_add_member(user_id, name, email, role):
    """Add a team member."""
    from ucts.team import get_team_manager, Role

    console.print(Panel.fit(
        "[bold]Adding Team Member[/bold]",
        border_style="blue"
    ))

    manager = get_team_manager()
    member = manager.add_member(
        user_id=user_id,
        name=name,
        email=email,
        role=Role(role),
    )

    console.print(f"\n[bold green]Member added![/bold green]")
    console.print(f"  ID: {member.user_id}")
    console.print(f"  Name: {member.name}")
    console.print(f"  Role: {member.role.value}")


@team.command("members")
def team_members():
    """List team members."""
    from ucts.team import get_team_manager

    console.print(Panel.fit(
        "[bold]Team Members[/bold]",
        border_style="blue"
    ))

    manager = get_team_manager()
    members = manager.list_members()

    if not members:
        console.print("[dim]No members[/dim]")
        return

    table = Table()
    table.add_column("ID", style="cyan")
    table.add_column("Name", style="green")
    table.add_column("Email", style="dim")
    table.add_column("Role", style="yellow")

    for m in members:
        table.add_row(m.user_id, m.name, m.email, m.role.value)

    console.print(table)


@team.command("share")
@click.argument('source', type=click.Path(exists=True))
@click.option('--title', '-t', help='Session title')
@click.option('--tags', multiple=True, help='Tags for the session')
@click.option('--user', '-u', default='default', help='User ID')
def team_share(source, title, tags, user):
    """Share a conversation with the team."""
    from ucts.ingestion import get_ingester
    from ucts.team import get_team_manager

    console.print(Panel.fit(
        "[bold]Sharing Session with Team[/bold]",
        border_style="blue"
    ))

    ingester = get_ingester('auto', source)
    session = ingester.ingest(source)

    manager = get_team_manager()
    shared = manager.add_to_library(
        session=session,
        user_id=user,
        title=title,
        tags=list(tags) if tags else None,
    )

    if shared:
        console.print(f"\n[bold green]Session shared![/bold green]")
        console.print(f"  Session ID: {shared.session_id}")
        console.print(f"  Title: {shared.title}")
        console.print(f"  Languages: {', '.join(shared.languages)}")
        console.print(f"  Code blocks: {shared.code_block_count}")
    else:
        console.print("[red]Failed to share session (permission denied)[/red]")


@team.command("library")
@click.option('--user', '-u', help='Filter by user')
@click.option('--tags', '-t', multiple=True, help='Filter by tags')
@click.option('--limit', '-n', default=20, help='Maximum results')
def team_library(user, tags, limit):
    """Browse the shared session library."""
    from ucts.team import get_team_manager

    console.print(Panel.fit(
        "[bold]Shared Session Library[/bold]",
        border_style="blue"
    ))

    manager = get_team_manager()
    sessions = manager.library.list_sessions(
        user_id=user,
        tags=list(tags) if tags else None,
        limit=limit,
    )

    if not sessions:
        console.print("[dim]No sessions in library[/dim]")
        return

    table = Table()
    table.add_column("ID", style="cyan")
    table.add_column("Title", style="green")
    table.add_column("Languages", style="dim")
    table.add_column("Owner", style="yellow")

    for s in sessions:
        table.add_row(
            s.session_id[:8],
            s.title[:30],
            ", ".join(s.languages[:3]),
            s.owner_id,
        )

    console.print(table)


@team.command("audit")
@click.option('--user', '-u', help='Filter by user')
@click.option('--action', '-a', help='Filter by action')
@click.option('--limit', '-n', default=50, help='Maximum results')
@click.option('--export', '-e', type=click.Path(), help='Export to file')
@click.option('--format', '-f',
              type=click.Choice(['json', 'csv', 'markdown']),
              default='json',
              help='Export format')
def team_audit(user, action, limit, export, format):
    """View audit log."""
    from ucts.team import get_team_manager

    console.print(Panel.fit(
        "[bold]Audit Log[/bold]",
        border_style="blue"
    ))

    manager = get_team_manager()

    if not manager.audit:
        console.print("[yellow]Audit logging is disabled[/yellow]")
        return

    entries = manager.audit.get_entries(
        user_id=user,
        action=action,
        limit=limit,
    )

    if export:
        output_file = manager.audit.export_report(export, format, user_id=user, action=action, limit=limit)
        console.print(f"[green]Exported to {output_file}[/green]")
        return

    if not entries:
        console.print("[dim]No audit entries[/dim]")
        return

    table = Table()
    table.add_column("Timestamp", style="dim")
    table.add_column("Action", style="cyan")
    table.add_column("User", style="green")
    table.add_column("Resource", style="yellow")

    for e in entries[:20]:
        table.add_row(
            e.timestamp.strftime("%Y-%m-%d %H:%M"),
            e.action,
            e.user_id,
            f"{e.resource_type}/{e.resource_id[:12]}",
        )

    console.print(table)


# Auth subgroup under team
@team.group("auth")
def team_auth():
    """Authentication management commands."""
    pass


@team_auth.command("status")
def auth_status():
    """Show authentication status and configuration."""
    from ucts.team import AuthManager, AuthConfig
    import os

    console.print(Panel.fit(
        "[bold]Authentication Status[/bold]",
        border_style="blue"
    ))

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)

    table = Table()
    table.add_column("Setting", style="cyan")
    table.add_column("Value", style="green")

    table.add_row("Provider", manager.config.provider.value)
    table.add_row("Enabled", str(manager.config.enabled))
    table.add_row("Session Timeout", f"{manager.config.session_timeout}s")
    table.add_row("MFA Required", str(manager.config.require_mfa))
    table.add_row("Active Sessions", str(len(manager._sessions)))

    console.print(table)


@team_auth.command("create-key")
@click.argument('user_id')
@click.option('--name', '-n', required=True, help='Key name/description')
@click.option('--scopes', '-s', multiple=True, default=['read', 'write'], help='Key scopes')
def auth_create_key(user_id, name, scopes):
    """Create an API key for a user."""
    from ucts.team import AuthManager
    import os

    console.print(Panel.fit(
        "[bold]Creating API Key[/bold]",
        border_style="blue"
    ))

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)
    key_id, api_key = manager.create_api_key(user_id, name, list(scopes))

    console.print(f"\n[bold green]API Key Created![/bold green]")
    console.print(f"  Key ID: {key_id}")
    console.print(f"  Scopes: {', '.join(scopes)}")
    console.print()
    console.print(Panel(
        f"[bold]{api_key}[/bold]",
        title="[yellow]Save this key - it won't be shown again[/yellow]",
        border_style="red"
    ))


@team_auth.command("list-keys")
@click.argument('user_id')
def auth_list_keys(user_id):
    """List API keys for a user."""
    from ucts.team import AuthManager
    import os

    console.print(Panel.fit(
        "[bold]API Keys[/bold]",
        border_style="blue"
    ))

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)
    keys = manager.list_api_keys(user_id)

    if not keys:
        console.print("[dim]No API keys found[/dim]")
        return

    table = Table()
    table.add_column("ID", style="cyan")
    table.add_column("Name", style="green")
    table.add_column("Scopes", style="dim")
    table.add_column("Created", style="yellow")
    table.add_column("Last Used", style="dim")

    for key in keys:
        table.add_row(
            key["key_id"],
            key["name"],
            ", ".join(key["scopes"]),
            key["created_at"][:10],
            key.get("last_used", "Never")[:10] if key.get("last_used") else "Never",
        )

    console.print(table)


@team_auth.command("revoke-key")
@click.argument('key_id')
def auth_revoke_key(key_id):
    """Revoke an API key."""
    from ucts.team import AuthManager
    import os

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)

    if manager.revoke_api_key(key_id):
        console.print(f"[bold green]API key {key_id} revoked[/bold green]")
    else:
        console.print(f"[red]API key {key_id} not found[/red]")


@team_auth.command("configure")
@click.option('--provider', '-p',
              type=click.Choice(['local', 'oauth2', 'oidc', 'saml', 'azure_ad', 'okta', 'google', 'github']),
              help='Authentication provider')
@click.option('--client-id', help='OAuth client ID')
@click.option('--client-secret', help='OAuth client secret')
@click.option('--tenant-id', help='Azure AD tenant ID or Okta domain')
@click.option('--enable', is_flag=True, help='Enable SSO')
@click.option('--disable', is_flag=True, help='Disable SSO')
def auth_configure(provider, client_id, client_secret, tenant_id, enable, disable):
    """Configure SSO authentication.

    Examples:
        ucts team auth configure --provider azure_ad --tenant-id xxx --client-id yyy
        ucts team auth configure --provider okta --tenant-id my-org.okta.com --client-id yyy
        ucts team auth configure --provider google --client-id yyy --client-secret zzz
    """
    from ucts.team import (
        AuthConfig, AuthProvider,
        get_azure_ad_config, get_okta_config, get_google_config, get_github_config
    )
    import os
    import json

    console.print(Panel.fit(
        "[bold]Configuring Authentication[/bold]",
        border_style="blue"
    ))

    home = os.path.expanduser("~")
    config_dir = os.path.join(home, ".ucts", "team")
    os.makedirs(config_dir, exist_ok=True)
    config_file = os.path.join(config_dir, "auth_config.json")

    # Load existing config
    config_data = {}
    if os.path.exists(config_file):
        with open(config_file) as f:
            config_data = json.load(f)

    if disable:
        config_data["enabled"] = False
        with open(config_file, 'w') as f:
            json.dump(config_data, f, indent=2)
        console.print("[yellow]SSO authentication disabled[/yellow]")
        return

    if provider:
        # Get provider-specific config
        if provider == "azure_ad" and tenant_id and client_id:
            config = get_azure_ad_config(tenant_id, client_id, client_secret or "")
        elif provider == "okta" and tenant_id and client_id:
            config = get_okta_config(tenant_id, client_id, client_secret or "")
        elif provider == "google" and client_id:
            config = get_google_config(client_id, client_secret or "")
        elif provider == "github" and client_id:
            config = get_github_config(client_id, client_secret or "")
        else:
            config = AuthConfig(provider=AuthProvider(provider))

        config_data = config.to_dict()

    if enable:
        config_data["enabled"] = True

    with open(config_file, 'w') as f:
        json.dump(config_data, f, indent=2)

    console.print(f"[green]Configuration saved to {config_file}[/green]")

    # Show current config
    table = Table(title="Current Configuration")
    table.add_column("Setting", style="cyan")
    table.add_column("Value", style="green")

    for key, value in config_data.items():
        if key == "client_secret":
            value = "***" if value else ""
        table.add_row(key, str(value)[:50])

    console.print(table)


@team_auth.command("sessions")
@click.option('--user', '-u', help='Filter by user ID')
def auth_sessions(user):
    """List active authentication sessions."""
    from ucts.team import AuthManager
    import os

    console.print(Panel.fit(
        "[bold]Active Sessions[/bold]",
        border_style="blue"
    ))

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)

    if user:
        sessions = manager.get_active_sessions(user)
    else:
        sessions = list(manager._sessions.values())

    if not sessions:
        console.print("[dim]No active sessions[/dim]")
        return

    table = Table()
    table.add_column("Session ID", style="cyan")
    table.add_column("User", style="green")
    table.add_column("Provider", style="dim")
    table.add_column("Created", style="yellow")
    table.add_column("Last Activity", style="dim")

    for s in sessions:
        table.add_row(
            s.session_id[:8],
            s.user_id[:20],
            s.token.provider.value,
            s.created_at.strftime("%Y-%m-%d %H:%M"),
            s.last_activity.strftime("%Y-%m-%d %H:%M"),
        )

    console.print(table)


@team_auth.command("logout")
@click.argument('session_id')
def auth_logout(session_id):
    """End an authentication session."""
    from ucts.team import AuthManager
    import os

    home = os.path.expanduser("~")
    storage_dir = os.path.join(home, ".ucts", "team")

    manager = AuthManager(storage_dir)

    # Find full session ID from prefix
    full_id = None
    for sid in manager._sessions:
        if sid.startswith(session_id):
            full_id = sid
            break

    if full_id and manager.logout(full_id):
        console.print(f"[bold green]Session {session_id} ended[/bold green]")
    else:
        console.print(f"[red]Session {session_id} not found[/red]")


if __name__ == '__main__':
    cli()
