"""
Universal Context Transfer System - Core Context Manager

Handles context buffering, token estimation, and transfer package generation.
"""
import json
import os
import platform
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Any, Optional

import yaml

from ucts.core.models import Message, Session


def get_default_storage_path() -> Path:
    """Get platform-appropriate default storage path.

    Returns:
        Path to UCTS storage directory:
        - Windows: %APPDATA%\\UCTS or %LOCALAPPDATA%\\UCTS
        - macOS: ~/Library/Application Support/UCTS
        - Linux/Unix: ~/.ucts
    """
    system = platform.system()

    if system == "Windows":
        # Prefer APPDATA, fall back to LOCALAPPDATA, then home
        appdata = os.environ.get("APPDATA")
        if appdata:
            return Path(appdata) / "UCTS"
        localappdata = os.environ.get("LOCALAPPDATA")
        if localappdata:
            return Path(localappdata) / "UCTS"
        return Path.home() / ".ucts"

    elif system == "Darwin":  # macOS
        # Use Application Support for macOS
        return Path.home() / "Library" / "Application Support" / "UCTS"

    else:  # Linux and other Unix-like systems
        # Use XDG_DATA_HOME if set, otherwise ~/.ucts
        xdg_data = os.environ.get("XDG_DATA_HOME")
        if xdg_data:
            return Path(xdg_data) / "ucts"
        return Path.home() / ".ucts"


class UniversalContextManager:
    """Universal context manager for AI conversation sessions"""

    def __init__(self, project_name: str = "default", base_path: str = None):
        self.project_name = project_name

        # Determine storage path
        if base_path:
            self.base_path = Path(base_path)
        else:
            # Use platform-appropriate default storage path
            self.base_path = get_default_storage_path()

        self.base_path.mkdir(exist_ok=True, parents=True)

        # Project directory
        safe_name = self._sanitize_filename(project_name)
        self.project_dir = self.base_path / "projects" / safe_name
        self.project_dir.mkdir(exist_ok=True, parents=True)

        # Current session
        self.session_id = datetime.now().strftime("%Y%m%d_%H%M%S")
        self.current_messages: List[Message] = []

        # Load config
        self.config = self._load_config()

    def _sanitize_filename(self, name: str) -> str:
        """Make filename safe for filesystem"""
        invalid_chars = '<>:"/\\|?*'
        for char in invalid_chars:
            name = name.replace(char, '_')
        return name[:100]

    def _load_config(self) -> Dict:
        """Load or create configuration"""
        config_file = self.base_path / "config.yaml"

        default_config = {
            'version': '2.0.0',
            'auto_monitor': True,
            'token_warning': 2500,
            'token_critical': 3000,
            'max_chunks': 5,
            'retention_days': 30
        }

        if config_file.exists():
            try:
                with open(config_file, 'r') as f:
                    loaded = yaml.safe_load(f)
                    return {**default_config, **(loaded or {})}
            except Exception:
                return default_config

        # Create default config
        with open(config_file, 'w') as f:
            yaml.dump(default_config, f)

        return default_config

    def add_message(self, role: str, content: str, metadata: Dict = None) -> Message:
        """Add a message to current session"""
        message = Message(
            role=role,
            content=content,
            timestamp=datetime.now().isoformat(),
            metadata=metadata or {}
        )

        self.current_messages.append(message)

        # Auto-check limits
        if len(self.current_messages) % 5 == 0:
            tokens = self.estimate_tokens()
            if tokens >= self.config['token_critical']:
                self._auto_transfer()

        return message

    def estimate_tokens(self) -> int:
        """Estimate token count (rough approximation: 1 token ~= 4 chars)"""
        if not self.current_messages:
            return 0
        text = "".join(m.content for m in self.current_messages)
        return len(text) // 4

    def save_chunk(self) -> Optional[Path]:
        """Save current messages as a chunk file"""
        if not self.current_messages:
            return None

        chunk_id = f"{self.session_id}_{len(self.current_messages)}"
        chunk_file = self.project_dir / f"chunk_{chunk_id}.json"

        chunk_data = {
            'chunk_id': chunk_id,
            'timestamp': datetime.now().isoformat(),
            'messages': [
                {
                    'role': m.role,
                    'content': m.content,
                    'timestamp': m.timestamp,
                    'metadata': m.metadata
                }
                for m in self.current_messages
            ],
            'message_count': len(self.current_messages),
            'estimated_tokens': self.estimate_tokens()
        }

        with open(chunk_file, 'w', encoding='utf-8') as f:
            json.dump(chunk_data, f, indent=2)

        self.current_messages = []
        return chunk_file

    def get_recent_chunks(self, limit: int = 5) -> List[Dict]:
        """Get recent chunk files"""
        chunk_files = sorted(
            self.project_dir.glob('chunk_*.json'),
            key=lambda x: x.stat().st_mtime,
            reverse=True
        )

        chunks = []
        for chunk_file in chunk_files[:limit]:
            try:
                with open(chunk_file, 'r') as f:
                    chunks.append(json.load(f))
            except Exception:
                continue

        return chunks

    def analyze_conversation(self) -> Dict[str, Any]:
        """Analyze conversation content"""
        chunks = self.get_recent_chunks()

        if not chunks:
            return {
                'total_messages': 0,
                'total_tokens': 0,
                'roles': {},
                'topics': [],
                'has_code': False,
                'has_errors': False,
                'has_decisions': False,
                'status': 'no_chunks'
            }

        all_messages = []
        for chunk in chunks:
            all_messages.extend(chunk.get('messages', []))

        analysis = {
            'total_messages': len(all_messages),
            'total_tokens': sum(c.get('estimated_tokens', 0) for c in chunks),
            'roles': {},
            'topics': [],
            'has_code': False,
            'has_errors': False,
            'has_decisions': False,
            'status': 'ok'
        }

        # Count roles
        for msg in all_messages:
            role = msg.get('role', 'unknown')
            analysis['roles'][role] = analysis['roles'].get(role, 0) + 1

        # Detect topics and features
        topics = set()
        for msg in all_messages:
            content = msg.get('content', '').lower()

            # Language detection
            if 'python' in content:
                topics.add('python')
            if 'javascript' in content or 'typescript' in content:
                topics.add('javascript')
            if 'rust' in content:
                topics.add('rust')

            # Feature detection
            if '```' in content:
                analysis['has_code'] = True
            if any(w in content for w in ['error', 'bug', 'fix', 'issue']):
                analysis['has_errors'] = True
                topics.add('debugging')
            if any(w in content for w in ['decided', 'decision', 'choose', 'chose']):
                analysis['has_decisions'] = True

        analysis['topics'] = list(topics)
        return analysis

    def create_transfer_package(self) -> Dict[str, Any]:
        """Create a complete transfer package"""
        # Save any pending messages
        if self.current_messages:
            self.save_chunk()

        chunks = self.get_recent_chunks()
        analysis = self.analyze_conversation()

        transfer_id = datetime.now().strftime("%Y%m%d_%H%M%S")

        package = {
            'transfer_id': transfer_id,
            'created_at': datetime.now().isoformat(),
            'project': self.project_name,
            'version': '2.0.0',
            'analysis': analysis,
            'chunks': len(chunks),
            'total_messages': analysis.get('total_messages', 0),
            'estimated_tokens': analysis.get('total_tokens', 0),
            'next_actions': self._suggest_next_actions(analysis)
        }

        # Create markdown summary
        package['markdown'] = self._create_markdown(package, chunks)

        # Save package
        transfer_file = self.project_dir / f"transfer_{transfer_id}.json"
        with open(transfer_file, 'w', encoding='utf-8') as f:
            json.dump(package, f, indent=2)

        # Also save markdown version
        md_file = self.project_dir / f"transfer_{transfer_id}.md"
        with open(md_file, 'w', encoding='utf-8') as f:
            f.write(package['markdown'])

        return package

    def _suggest_next_actions(self, analysis: Dict) -> List[str]:
        """Suggest next actions based on analysis"""
        actions = []

        if analysis.get('has_errors'):
            actions.append("Review and fix identified errors")
        if analysis.get('has_decisions'):
            actions.append("Implement documented decisions")
        if analysis.get('has_code'):
            actions.append("Review and test code changes")

        if not actions:
            actions = ["Continue development", "Run tests", "Update documentation"]

        return actions[:3]

    def _create_markdown(self, package: Dict, chunks: List[Dict]) -> str:
        """Create markdown transfer document"""
        all_messages = []
        for chunk in chunks:
            all_messages.extend(chunk.get('messages', []))

        recent = all_messages[-10:] if all_messages else []

        md = f"""# Context Transfer: {package['project']}

**Transfer ID**: {package['transfer_id']}
**Created**: {package['created_at']}

## Quick Stats
- **Messages**: {package['total_messages']}
- **Tokens**: ~{package['estimated_tokens']}
- **Topics**: {', '.join(package['analysis'].get('topics', []))}
- **Has Code**: {'Yes' if package['analysis'].get('has_code') else 'No'}

## Next Actions
"""
        for action in package.get('next_actions', []):
            md += f"- {action}\n"

        if recent:
            md += "\n## Recent Messages\n\n"
            for msg in recent:
                role = msg.get('role', 'unknown').upper()
                content = msg.get('content', '')[:300]
                if len(msg.get('content', '')) > 300:
                    content += "..."
                md += f"**{role}**: {content}\n\n"

        return md

    def _auto_transfer(self):
        """Auto-transfer when limits are reached"""
        self.save_chunk()
        return self.create_transfer_package()

    def get_status(self) -> Dict[str, Any]:
        """Get current manager status"""
        return {
            'project': self.project_name,
            'message_count': len(self.current_messages),
            'estimated_tokens': self.estimate_tokens(),
            'chunk_count': len(list(self.project_dir.glob('chunk_*.json'))),
            'storage_path': str(self.project_dir),
            'config': self.config
        }


# Singleton instance
_global_ucts: Optional[UniversalContextManager] = None


def get_ucts(project_name: str = "default") -> UniversalContextManager:
    """Get or create global UCTS instance"""
    global _global_ucts
    if _global_ucts is None:
        _global_ucts = UniversalContextManager(project_name)
    return _global_ucts
