"""
Forge Pipeline Hooks

Provides pre-forge and post-forge hooks that integrate with the Coherence ecosystem:
- Pre-Forge: Trust checks, policy validation, pattern search
- Post-Forge: Security scan, session storage, metrics recording
"""

import logging
import time
from typing import Dict, Any, Optional, List, Callable
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum

logger = logging.getLogger(__name__)


class HookPhase(Enum):
    """Pipeline phases for hooks"""
    PRE_INGEST = "pre_ingest"
    POST_INGEST = "post_ingest"
    PRE_ANALYZE = "pre_analyze"
    POST_ANALYZE = "post_analyze"
    PRE_GENERATE = "pre_generate"
    POST_GENERATE = "post_generate"
    PRE_MERGE = "pre_merge"
    POST_MERGE = "post_merge"


class HookResult(Enum):
    """Result of hook execution"""
    CONTINUE = "continue"      # Proceed with pipeline
    SKIP = "skip"              # Skip this step
    ABORT = "abort"            # Abort entire pipeline
    WARN = "warn"              # Continue but with warning


@dataclass
class HookContext:
    """Context passed to hooks"""
    phase: HookPhase
    source_paths: List[str] = field(default_factory=list)
    output_path: Optional[str] = None
    target_type: str = "vscode"
    session: Any = None
    structure: Any = None
    merge_result: Any = None
    metadata: Dict[str, Any] = field(default_factory=dict)
    start_time: float = field(default_factory=time.time)
    warnings: List[str] = field(default_factory=list)
    errors: List[str] = field(default_factory=list)


@dataclass
class HookResponse:
    """Response from a hook"""
    result: HookResult
    message: Optional[str] = None
    data: Dict[str, Any] = field(default_factory=dict)
    duration_ms: int = 0


class ForgeHooks:
    """
    Manages forge pipeline hooks.

    Provides integration points for ecosystem services:
    - GCU: Trust checks before generation
    - Sentinel: Policy validation and security scanning
    - MMB: Pattern search and session storage
    - WB2: Metrics recording
    - DanAI: Task completion reporting
    """

    def __init__(self, ecosystem_enabled: bool = True):
        self.ecosystem_enabled = ecosystem_enabled
        self._hooks: Dict[HookPhase, List[Callable]] = {phase: [] for phase in HookPhase}
        self._ecosystem = None

        if ecosystem_enabled:
            self._register_ecosystem_hooks()

    def _get_ecosystem(self):
        """Lazy load ecosystem manager"""
        if self._ecosystem is None and self.ecosystem_enabled:
            try:
                from ucts.integrations import get_ecosystem
                self._ecosystem = get_ecosystem()
                self._ecosystem.initialize()
            except Exception as e:
                logger.warning(f"Failed to initialize ecosystem: {e}")
                self.ecosystem_enabled = False
        return self._ecosystem

    def _register_ecosystem_hooks(self):
        """Register default ecosystem hooks"""
        # Pre-analyze: Search for similar patterns
        self.register(HookPhase.PRE_ANALYZE, self._hook_search_patterns)

        # Post-analyze: Validate policy
        self.register(HookPhase.POST_ANALYZE, self._hook_validate_policy)

        # Pre-generate: Trust check
        self.register(HookPhase.PRE_GENERATE, self._hook_trust_check)

        # Post-generate: Security scan, store session, record metrics
        self.register(HookPhase.POST_GENERATE, self._hook_security_scan)
        self.register(HookPhase.POST_GENERATE, self._hook_store_session)
        self.register(HookPhase.POST_GENERATE, self._hook_record_metrics)

        # Post-merge: Record merge metrics
        self.register(HookPhase.POST_MERGE, self._hook_record_merge_metrics)

    def register(self, phase: HookPhase, hook: Callable[[HookContext], HookResponse]):
        """Register a hook for a phase"""
        self._hooks[phase].append(hook)

    def unregister(self, phase: HookPhase, hook: Callable):
        """Unregister a hook"""
        if hook in self._hooks[phase]:
            self._hooks[phase].remove(hook)

    def execute(self, phase: HookPhase, context: HookContext) -> HookResponse:
        """
        Execute all hooks for a phase.

        Args:
            phase: Pipeline phase
            context: Hook context

        Returns:
            Aggregated hook response
        """
        if not self.ecosystem_enabled:
            return HookResponse(result=HookResult.CONTINUE)

        context.phase = phase
        start = time.time()

        aggregated_data = {}
        final_result = HookResult.CONTINUE

        for hook in self._hooks[phase]:
            try:
                response = hook(context)

                # Merge data
                aggregated_data.update(response.data)

                # Check result
                if response.result == HookResult.ABORT:
                    return HookResponse(
                        result=HookResult.ABORT,
                        message=response.message,
                        data=aggregated_data,
                        duration_ms=int((time.time() - start) * 1000)
                    )
                elif response.result == HookResult.WARN:
                    final_result = HookResult.WARN
                    if response.message:
                        context.warnings.append(response.message)

            except Exception as e:
                logger.error(f"Hook {hook.__name__} failed: {e}")
                context.errors.append(f"Hook {hook.__name__}: {str(e)}")

        return HookResponse(
            result=final_result,
            data=aggregated_data,
            duration_ms=int((time.time() - start) * 1000)
        )

    # Ecosystem Hooks

    def _hook_search_patterns(self, context: HookContext) -> HookResponse:
        """Search for similar patterns before analyzing"""
        eco = self._get_ecosystem()
        if not eco or not eco.mmb:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            # Get summary from source paths
            source_summary = " ".join(context.source_paths)

            # Search for similar sessions
            similar = eco.mmb.search_sessions(query=source_summary, limit=3)

            if similar:
                logger.info(f"Found {len(similar)} similar sessions in ecosystem")
                return HookResponse(
                    result=HookResult.CONTINUE,
                    data={"similar_sessions": similar}
                )
        except Exception as e:
            logger.debug(f"Pattern search skipped: {e}")

        return HookResponse(result=HookResult.CONTINUE)

    def _hook_validate_policy(self, context: HookContext) -> HookResponse:
        """Validate forge against Sentinel policies"""
        eco = self._get_ecosystem()
        if not eco or not eco.sentinel:
            return HookResponse(result=HookResult.CONTINUE)

        structure = context.structure
        if not structure:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            # Detect patterns in code
            has_auth = any(
                "auth" in f.lower() or "login" in f.lower() or "password" in f.lower()
                for f in structure.files.keys()
            )
            has_db = any(
                "database" in c.lower() or "sql" in c.lower() or "query" in c.lower()
                for c in structure.files.values()
            )
            has_api_keys = any(
                "api_key" in c.lower() or "apikey" in c.lower()
                for c in structure.files.values()
            )

            result = eco.sentinel.validate_forge_policy(
                languages=structure.languages,
                has_auth_patterns=has_auth,
                has_db_operations=has_db,
                has_api_keys=has_api_keys,
                estimated_complexity="high" if len(structure.files) > 20 else "medium"
            )

            if not result.get("approved"):
                return HookResponse(
                    result=HookResult.WARN,
                    message=f"Policy validation: {result.get('decision')}",
                    data={"policy_validation": result}
                )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"policy_validation": result}
            )
        except Exception as e:
            logger.debug(f"Policy validation skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)

    def _hook_trust_check(self, context: HookContext) -> HookResponse:
        """Check trust tier before generation"""
        eco = self._get_ecosystem()
        if not eco or not eco.gcu:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            # Determine operation type
            operation = "forge"
            if context.metadata.get("has_deploy_scripts"):
                operation = "forge_deploy"
            elif context.metadata.get("has_auth"):
                operation = "forge_auth"

            result = eco.gcu.check_trust("ucts-forge", operation)

            if not result.get("approved"):
                return HookResponse(
                    result=HookResult.WARN,
                    message=f"Trust check: {result.get('reason')}",
                    data={"trust_check": result}
                )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"trust_check": result}
            )
        except Exception as e:
            logger.debug(f"Trust check skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)

    def _hook_security_scan(self, context: HookContext) -> HookResponse:
        """Run security scan on generated code"""
        eco = self._get_ecosystem()
        if not eco or not eco.sentinel:
            return HookResponse(result=HookResult.CONTINUE)

        structure = context.structure
        if not structure or not structure.files:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            result = eco.sentinel.scan_project_security(structure.files)

            if not result.passed:
                return HookResponse(
                    result=HookResult.WARN,
                    message=f"Security scan found {len(result.violations)} violation(s)",
                    data={
                        "security_scan": {
                            "passed": result.passed,
                            "violations": len(result.violations),
                            "warnings": len(result.warnings)
                        }
                    }
                )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"security_scan": {"passed": True}}
            )
        except Exception as e:
            logger.debug(f"Security scan skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)

    def _hook_store_session(self, context: HookContext) -> HookResponse:
        """Store session in MMB"""
        eco = self._get_ecosystem()
        if not eco or not eco.mmb:
            return HookResponse(result=HookResult.CONTINUE)

        structure = context.structure
        session = context.session
        if not structure:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            session_id = f"forge-{datetime.now().strftime('%Y%m%d%H%M%S')}"

            eco.mmb.store_session(
                session_id=session_id,
                summary=structure.description or structure.name,
                languages=structure.languages,
                dependencies=structure.dependencies,
                code_blocks=len(session.code_blocks) if session else 0,
                patterns_detected=[],
                tags=["auto-stored", context.target_type]
            )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"stored_session_id": session_id}
            )
        except Exception as e:
            logger.debug(f"Session storage skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)

    def _hook_record_metrics(self, context: HookContext) -> HookResponse:
        """Record forge metrics to WB2"""
        eco = self._get_ecosystem()
        if not eco or not eco.wb2:
            return HookResponse(result=HookResult.CONTINUE)

        structure = context.structure
        if not structure:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            duration = time.time() - context.start_time

            eco.wb2.record_forge_activity(
                session_id=context.metadata.get("session_id", "unknown"),
                success=True,
                files_created=len(structure.files),
                languages=structure.languages,
                target_type=context.target_type,
                duration_seconds=duration,
                project_name=structure.name
            )

            # Also report to GCU
            if eco.gcu:
                eco.gcu.report_forge_metrics(
                    success=True,
                    duration_seconds=duration,
                    files_created=len(structure.files),
                    languages=structure.languages,
                    target_type=context.target_type
                )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"metrics_recorded": True}
            )
        except Exception as e:
            logger.debug(f"Metrics recording skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)

    def _hook_record_merge_metrics(self, context: HookContext) -> HookResponse:
        """Record merge metrics"""
        eco = self._get_ecosystem()
        if not eco or not eco.wb2:
            return HookResponse(result=HookResult.CONTINUE)

        merge_result = context.merge_result
        if not merge_result:
            return HookResponse(result=HookResult.CONTINUE)

        try:
            duration = time.time() - context.start_time

            eco.wb2.record_forge_activity(
                session_id=context.metadata.get("session_id", "unknown"),
                success=True,
                files_created=len(merge_result.files_created),
                languages=context.metadata.get("languages", []),
                target_type="merge",
                duration_seconds=duration,
                project_name=context.output_path
            )

            return HookResponse(
                result=HookResult.CONTINUE,
                data={"merge_metrics_recorded": True}
            )
        except Exception as e:
            logger.debug(f"Merge metrics recording skipped: {e}")
            return HookResponse(result=HookResult.CONTINUE)


# Singleton instance
_hooks: Optional[ForgeHooks] = None


def get_forge_hooks(ecosystem_enabled: bool = True) -> ForgeHooks:
    """Get the forge hooks singleton"""
    global _hooks
    if _hooks is None:
        _hooks = ForgeHooks(ecosystem_enabled)
    return _hooks


def reset_hooks():
    """Reset the hooks singleton"""
    global _hooks
    _hooks = None
