"""
UCTS Core Data Models
"""
from dataclasses import dataclass, field
from typing import Dict, List, Any, Optional
from datetime import datetime


@dataclass
class Message:
    """A single message in a conversation session"""
    role: str           # user, assistant, system
    content: str
    timestamp: str = field(default_factory=lambda: datetime.now().isoformat())
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class CodeBlock:
    """An extracted code block from a conversation"""
    language: str
    content: str
    filename: Optional[str] = None
    purpose: str = "implementation"  # implementation, example, config, test


@dataclass
class Session:
    """A complete conversation session from any source"""
    source: str         # claude_desktop, vscode, web, cli, api
    messages: List[Message] = field(default_factory=list)
    code_blocks: List[CodeBlock] = field(default_factory=list)
    files_created: List[Dict[str, str]] = field(default_factory=list)
    decisions: List[str] = field(default_factory=list)
    todos: List[str] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)

    @property
    def message_count(self) -> int:
        return len(self.messages)

    @property
    def has_code(self) -> bool:
        return len(self.code_blocks) > 0

    def to_dict(self) -> Dict[str, Any]:
        """Convert session to dictionary"""
        return {
            "source": self.source,
            "messages": [
                {
                    "role": m.role,
                    "content": m.content,
                    "timestamp": m.timestamp,
                    "metadata": m.metadata
                }
                for m in self.messages
            ],
            "code_blocks": [
                {
                    "language": cb.language,
                    "content": cb.content,
                    "filename": cb.filename,
                    "purpose": cb.purpose
                }
                for cb in self.code_blocks
            ],
            "files_created": self.files_created,
            "decisions": self.decisions,
            "todos": self.todos,
            "metadata": self.metadata
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "Session":
        """Create session from dictionary"""
        messages = [
            Message(
                role=m["role"],
                content=m["content"],
                timestamp=m.get("timestamp", ""),
                metadata=m.get("metadata", {})
            )
            for m in data.get("messages", [])
        ]

        code_blocks = [
            CodeBlock(
                language=cb["language"],
                content=cb["content"],
                filename=cb.get("filename"),
                purpose=cb.get("purpose", "implementation")
            )
            for cb in data.get("code_blocks", [])
        ]

        return cls(
            source=data.get("source", "unknown"),
            messages=messages,
            code_blocks=code_blocks,
            files_created=data.get("files_created", []),
            decisions=data.get("decisions", []),
            todos=data.get("todos", []),
            metadata=data.get("metadata", {})
        )

    def merge(self, other: "Session") -> "Session":
        """Merge another session into this one"""
        return Session(
            source=f"{self.source}+{other.source}",
            messages=self.messages + other.messages,
            code_blocks=self.code_blocks + other.code_blocks,
            files_created=self.files_created + other.files_created,
            decisions=list(set(self.decisions + other.decisions)),
            todos=list(set(self.todos + other.todos)),
            metadata={**self.metadata, **other.metadata}
        )

    @classmethod
    def merge_all(cls, sessions: List["Session"]) -> "Session":
        """Merge multiple sessions into one"""
        if not sessions:
            return cls(source="empty")
        if len(sessions) == 1:
            return sessions[0]

        result = sessions[0]
        for session in sessions[1:]:
            result = result.merge(session)
        return result


@dataclass
class ProjectStructure:
    """The analyzed project structure ready for generation"""
    name: str
    description: str
    languages: List[str] = field(default_factory=list)
    dependencies: Dict[str, List[str]] = field(default_factory=dict)  # {python: [requests, ...]}
    files: Dict[str, str] = field(default_factory=dict)  # {path: content}
    directories: List[str] = field(default_factory=list)
    readme_content: str = ""
    todos: List[str] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary"""
        return {
            "name": self.name,
            "description": self.description,
            "languages": self.languages,
            "dependencies": self.dependencies,
            "files": self.files,
            "directories": self.directories,
            "readme_content": self.readme_content,
            "todos": self.todos,
            "metadata": self.metadata
        }
