"""
Cloud Deployment Generators

Generates deployment configurations for major cloud platforms:
- AWS (CloudFormation/SAM)
- GCP (Terraform + Cloud Run)
- Azure (ARM templates)
- Vercel
- Railway
"""

import json
from pathlib import Path
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, field
from datetime import datetime


@dataclass
class CloudConfig:
    """Configuration for cloud deployment generation"""
    project_name: str = "app"
    region: str = "us-east-1"
    environment: str = "production"
    # Compute
    memory_mb: int = 512
    cpu: float = 0.5
    min_instances: int = 1
    max_instances: int = 10
    # Networking
    port: int = 8000
    health_check_path: str = "/health"
    # Database
    include_database: bool = False
    database_type: str = "postgres"  # postgres, mysql, redis
    # Secrets
    env_vars: Dict[str, str] = field(default_factory=dict)


class AWSGenerator:
    """Generate AWS CloudFormation/SAM templates"""

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate AWS deployment files"""
        files = {}

        # Detect primary language
        is_python = "python" in [l.lower() for l in structure.languages]
        is_node = any(l.lower() in ["javascript", "typescript"] for l in structure.languages)

        # SAM template
        files["template.yaml"] = self._generate_sam_template(is_python, is_node)

        # samconfig.toml
        files["samconfig.toml"] = self._generate_samconfig()

        # Buildspec for CodeBuild
        files["buildspec.yml"] = self._generate_buildspec(is_python, is_node)

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            (output / filename).write_text(content)

        return files

    def _generate_sam_template(self, is_python: bool, is_node: bool) -> str:
        runtime = "python3.11" if is_python else "nodejs20.x"
        handler = "app.handler" if is_python else "index.handler"

        template = f"""AWSTemplateFormatVersion: '2010-09-09'
Transform: AWS::Serverless-2016-10-31
Description: {self.config.project_name} - Generated by UCTS

Globals:
  Function:
    Timeout: 30
    MemorySize: {self.config.memory_mb}
    Runtime: {runtime}
    Environment:
      Variables:
        ENVIRONMENT: {self.config.environment}

Resources:
  # API Gateway
  ApiGateway:
    Type: AWS::Serverless::Api
    Properties:
      StageName: {self.config.environment}
      Cors:
        AllowOrigin: "'*'"
        AllowMethods: "'GET,POST,PUT,DELETE,OPTIONS'"
        AllowHeaders: "'Content-Type,Authorization'"

  # Lambda Function
  AppFunction:
    Type: AWS::Serverless::Function
    Properties:
      FunctionName: {self.config.project_name}-{self.config.environment}
      Handler: {handler}
      CodeUri: ./src
      Description: Main application function
      Events:
        ApiEvent:
          Type: Api
          Properties:
            RestApiId: !Ref ApiGateway
            Path: /{{proxy+}}
            Method: ANY
      Policies:
        - AWSLambdaBasicExecutionRole
"""

        if self.config.include_database:
            template += f"""
  # RDS Database
  Database:
    Type: AWS::RDS::DBInstance
    Properties:
      DBInstanceIdentifier: {self.config.project_name}-db
      DBInstanceClass: db.t3.micro
      Engine: {self.config.database_type}
      MasterUsername: admin
      MasterUserPassword: !Ref DBPassword
      AllocatedStorage: 20
      PubliclyAccessible: false

Parameters:
  DBPassword:
    Type: String
    NoEcho: true
    Description: Database password
"""

        template += f"""
Outputs:
  ApiUrl:
    Description: API Gateway URL
    Value: !Sub "https://${{ApiGateway}}.execute-api.${{AWS::Region}}.amazonaws.com/{self.config.environment}/"
  FunctionArn:
    Description: Lambda Function ARN
    Value: !GetAtt AppFunction.Arn
"""
        return template

    def _generate_samconfig(self) -> str:
        return f"""version = 0.1

[default.deploy.parameters]
stack_name = "{self.config.project_name}-stack"
resolve_s3 = true
s3_prefix = "{self.config.project_name}"
region = "{self.config.region}"
confirm_changeset = true
capabilities = "CAPABILITY_IAM"

[default.build.parameters]
cached = true
parallel = true

[default.sync.parameters]
watch = true
"""

    def _generate_buildspec(self, is_python: bool, is_node: bool) -> str:
        if is_python:
            install_cmd = "pip install -r requirements.txt -t ./src"
        else:
            install_cmd = "npm ci"

        return f"""version: 0.2

phases:
  install:
    runtime-versions:
      {"python: 3.11" if is_python else "nodejs: 20"}
    commands:
      - {install_cmd}

  build:
    commands:
      - sam build
      - sam package --output-template-file packaged.yaml --s3-bucket $S3_BUCKET

  post_build:
    commands:
      - sam deploy --template-file packaged.yaml --stack-name {self.config.project_name} --capabilities CAPABILITY_IAM --no-fail-on-empty-changeset

artifacts:
  files:
    - packaged.yaml
    - template.yaml
"""


class GCPGenerator:
    """Generate GCP Terraform + Cloud Run configuration"""

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate GCP deployment files"""
        files = {}

        is_python = "python" in [l.lower() for l in structure.languages]

        # Main Terraform
        files["main.tf"] = self._generate_main_tf()

        # Variables
        files["variables.tf"] = self._generate_variables_tf()

        # Outputs
        files["outputs.tf"] = self._generate_outputs_tf()

        # Cloud Build config
        files["cloudbuild.yaml"] = self._generate_cloudbuild(is_python)

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            (output / filename).write_text(content)

        return files

    def _generate_main_tf(self) -> str:
        tf = f"""terraform {{
  required_version = ">= 1.0"
  required_providers {{
    google = {{
      source  = "hashicorp/google"
      version = "~> 5.0"
    }}
  }}
}}

provider "google" {{
  project = var.project_id
  region  = var.region
}}

# Enable required APIs
resource "google_project_service" "run" {{
  service = "run.googleapis.com"
}}

resource "google_project_service" "artifactregistry" {{
  service = "artifactregistry.googleapis.com"
}}

# Artifact Registry for Docker images
resource "google_artifact_registry_repository" "repo" {{
  location      = var.region
  repository_id = "{self.config.project_name}-repo"
  format        = "DOCKER"
}}

# Cloud Run Service
resource "google_cloud_run_v2_service" "app" {{
  name     = "{self.config.project_name}"
  location = var.region

  template {{
    containers {{
      image = "${{var.region}}-docker.pkg.dev/${{var.project_id}}/{self.config.project_name}-repo/{self.config.project_name}:latest"

      ports {{
        container_port = {self.config.port}
      }}

      resources {{
        limits = {{
          cpu    = "{self.config.cpu}"
          memory = "{self.config.memory_mb}Mi"
        }}
      }}

      startup_probe {{
        http_get {{
          path = "{self.config.health_check_path}"
        }}
      }}

      liveness_probe {{
        http_get {{
          path = "{self.config.health_check_path}"
        }}
      }}
    }}

    scaling {{
      min_instance_count = {self.config.min_instances}
      max_instance_count = {self.config.max_instances}
    }}
  }}

  traffic {{
    type    = "TRAFFIC_TARGET_ALLOCATION_TYPE_LATEST"
    percent = 100
  }}

  depends_on = [google_project_service.run]
}}

# Allow unauthenticated access
resource "google_cloud_run_v2_service_iam_member" "public" {{
  location = google_cloud_run_v2_service.app.location
  name     = google_cloud_run_v2_service.app.name
  role     = "roles/run.invoker"
  member   = "allUsers"
}}
"""

        if self.config.include_database:
            tf += f"""
# Cloud SQL Instance
resource "google_sql_database_instance" "db" {{
  name             = "{self.config.project_name}-db"
  database_version = "POSTGRES_15"
  region           = var.region

  settings {{
    tier = "db-f1-micro"

    ip_configuration {{
      ipv4_enabled = true
    }}
  }}

  deletion_protection = false
}}

resource "google_sql_database" "database" {{
  name     = "{self.config.project_name}"
  instance = google_sql_database_instance.db.name
}}
"""
        return tf

    def _generate_variables_tf(self) -> str:
        return f"""variable "project_id" {{
  description = "GCP Project ID"
  type        = string
}}

variable "region" {{
  description = "GCP Region"
  type        = string
  default     = "{self.config.region.replace('us-east-1', 'us-central1')}"
}}

variable "environment" {{
  description = "Environment name"
  type        = string
  default     = "{self.config.environment}"
}}
"""

    def _generate_outputs_tf(self) -> str:
        return f"""output "service_url" {{
  description = "Cloud Run service URL"
  value       = google_cloud_run_v2_service.app.uri
}}

output "service_name" {{
  description = "Cloud Run service name"
  value       = google_cloud_run_v2_service.app.name
}}
"""

    def _generate_cloudbuild(self, is_python: bool) -> str:
        return f"""steps:
  # Build Docker image
  - name: 'gcr.io/cloud-builders/docker'
    args:
      - 'build'
      - '-t'
      - '${{_REGION}}-docker.pkg.dev/$PROJECT_ID/{self.config.project_name}-repo/{self.config.project_name}:$COMMIT_SHA'
      - '-t'
      - '${{_REGION}}-docker.pkg.dev/$PROJECT_ID/{self.config.project_name}-repo/{self.config.project_name}:latest'
      - '.'

  # Push to Artifact Registry
  - name: 'gcr.io/cloud-builders/docker'
    args:
      - 'push'
      - '--all-tags'
      - '${{_REGION}}-docker.pkg.dev/$PROJECT_ID/{self.config.project_name}-repo/{self.config.project_name}'

  # Deploy to Cloud Run
  - name: 'gcr.io/cloud-builders/gcloud'
    args:
      - 'run'
      - 'deploy'
      - '{self.config.project_name}'
      - '--image'
      - '${{_REGION}}-docker.pkg.dev/$PROJECT_ID/{self.config.project_name}-repo/{self.config.project_name}:$COMMIT_SHA'
      - '--region'
      - '${{_REGION}}'
      - '--platform'
      - 'managed'
      - '--allow-unauthenticated'

substitutions:
  _REGION: us-central1

options:
  logging: CLOUD_LOGGING_ONLY
"""


class AzureGenerator:
    """Generate Azure ARM templates"""

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Azure deployment files"""
        files = {}

        # ARM template
        files["azuredeploy.json"] = self._generate_arm_template()

        # Parameters file
        files["azuredeploy.parameters.json"] = self._generate_parameters()

        # Azure Pipelines
        files["azure-pipelines.yml"] = self._generate_pipeline()

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            (output / filename).write_text(content)

        return files

    def _generate_arm_template(self) -> str:
        template = {
            "$schema": "https://schema.management.azure.com/schemas/2019-04-01/deploymentTemplate.json#",
            "contentVersion": "1.0.0.0",
            "parameters": {
                "appName": {
                    "type": "string",
                    "defaultValue": self.config.project_name
                },
                "location": {
                    "type": "string",
                    "defaultValue": "[resourceGroup().location]"
                },
                "sku": {
                    "type": "string",
                    "defaultValue": "B1"
                }
            },
            "variables": {
                "appServicePlanName": f"[concat(parameters('appName'), '-plan')]",
                "webAppName": "[parameters('appName')]"
            },
            "resources": [
                {
                    "type": "Microsoft.Web/serverfarms",
                    "apiVersion": "2022-03-01",
                    "name": "[variables('appServicePlanName')]",
                    "location": "[parameters('location')]",
                    "sku": {
                        "name": "[parameters('sku')]"
                    },
                    "kind": "linux",
                    "properties": {
                        "reserved": True
                    }
                },
                {
                    "type": "Microsoft.Web/sites",
                    "apiVersion": "2022-03-01",
                    "name": "[variables('webAppName')]",
                    "location": "[parameters('location')]",
                    "dependsOn": [
                        "[resourceId('Microsoft.Web/serverfarms', variables('appServicePlanName'))]"
                    ],
                    "properties": {
                        "serverFarmId": "[resourceId('Microsoft.Web/serverfarms', variables('appServicePlanName'))]",
                        "siteConfig": {
                            "linuxFxVersion": "PYTHON|3.11",
                            "alwaysOn": True,
                            "healthCheckPath": self.config.health_check_path
                        },
                        "httpsOnly": True
                    }
                }
            ],
            "outputs": {
                "webAppUrl": {
                    "type": "string",
                    "value": "[concat('https://', variables('webAppName'), '.azurewebsites.net')]"
                }
            }
        }

        return json.dumps(template, indent=2)

    def _generate_parameters(self) -> str:
        params = {
            "$schema": "https://schema.management.azure.com/schemas/2019-04-01/deploymentParameters.json#",
            "contentVersion": "1.0.0.0",
            "parameters": {
                "appName": {
                    "value": self.config.project_name
                },
                "sku": {
                    "value": "B1"
                }
            }
        }
        return json.dumps(params, indent=2)

    def _generate_pipeline(self) -> str:
        return f"""trigger:
  - main

pool:
  vmImage: 'ubuntu-latest'

variables:
  azureSubscription: 'your-azure-subscription'
  resourceGroup: '{self.config.project_name}-rg'
  webAppName: '{self.config.project_name}'

stages:
  - stage: Build
    jobs:
      - job: Build
        steps:
          - task: UsePythonVersion@0
            inputs:
              versionSpec: '3.11'

          - script: |
              python -m pip install --upgrade pip
              pip install -r requirements.txt
            displayName: 'Install dependencies'

          - task: ArchiveFiles@2
            inputs:
              rootFolderOrFile: '$(Build.SourcesDirectory)'
              includeRootFolder: false
              archiveType: 'zip'
              archiveFile: '$(Build.ArtifactStagingDirectory)/$(Build.BuildId).zip'

          - publish: $(Build.ArtifactStagingDirectory)/$(Build.BuildId).zip
            artifact: drop

  - stage: Deploy
    dependsOn: Build
    jobs:
      - deployment: Deploy
        environment: '{self.config.environment}'
        strategy:
          runOnce:
            deploy:
              steps:
                - task: AzureRmWebAppDeployment@4
                  inputs:
                    ConnectionType: 'AzureRM'
                    azureSubscription: '$(azureSubscription)'
                    appType: 'webAppLinux'
                    WebAppName: '$(webAppName)'
                    packageForLinux: '$(Pipeline.Workspace)/drop/$(Build.BuildId).zip'
"""


class VercelGenerator:
    """Generate Vercel configuration"""

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Vercel deployment files"""
        files = {}

        is_python = "python" in [l.lower() for l in structure.languages]

        # vercel.json
        files["vercel.json"] = self._generate_vercel_json(is_python)

        # API route example
        if is_python:
            files["api/index.py"] = self._generate_python_api()
        else:
            files["api/index.ts"] = self._generate_node_api()

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            filepath = output / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)

        return files

    def _generate_vercel_json(self, is_python: bool) -> str:
        config = {
            "version": 2,
            "name": self.config.project_name,
            "builds": [
                {
                    "src": "api/**/*.py" if is_python else "api/**/*.ts",
                    "use": "@vercel/python" if is_python else "@vercel/node"
                }
            ],
            "routes": [
                {
                    "src": "/api/(.*)",
                    "dest": "/api/$1"
                },
                {
                    "src": "/(.*)",
                    "dest": "/api/index"
                }
            ],
            "env": {
                "ENVIRONMENT": self.config.environment
            },
            "regions": ["iad1"]
        }
        return json.dumps(config, indent=2)

    def _generate_python_api(self) -> str:
        return '''from http.server import BaseHTTPRequestHandler
import json


class handler(BaseHTTPRequestHandler):
    def do_GET(self):
        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()

        response = {
            "message": "Hello from Vercel!",
            "path": self.path
        }
        self.wfile.write(json.dumps(response).encode())

    def do_POST(self):
        content_length = int(self.headers['Content-Length'])
        post_data = self.rfile.read(content_length)

        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()

        response = {
            "message": "Received POST",
            "data": post_data.decode()
        }
        self.wfile.write(json.dumps(response).encode())
'''

    def _generate_node_api(self) -> str:
        return '''import type { VercelRequest, VercelResponse } from '@vercel/node';

export default function handler(req: VercelRequest, res: VercelResponse) {
  const { method, url, body } = req;

  if (method === 'GET') {
    return res.status(200).json({
      message: 'Hello from Vercel!',
      path: url
    });
  }

  if (method === 'POST') {
    return res.status(200).json({
      message: 'Received POST',
      data: body
    });
  }

  return res.status(405).json({ error: 'Method not allowed' });
}
'''


class RailwayGenerator:
    """Generate Railway configuration"""

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Railway deployment files"""
        files = {}

        is_python = "python" in [l.lower() for l in structure.languages]

        # railway.json
        files["railway.json"] = self._generate_railway_json(is_python)

        # Procfile
        files["Procfile"] = self._generate_procfile(is_python)

        # nixpacks.toml for build config
        files["nixpacks.toml"] = self._generate_nixpacks(is_python)

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            (output / filename).write_text(content)

        return files

    def _generate_railway_json(self, is_python: bool) -> str:
        config = {
            "$schema": "https://railway.app/railway.schema.json",
            "build": {
                "builder": "NIXPACKS"
            },
            "deploy": {
                "numReplicas": self.config.min_instances,
                "sleepApplication": False,
                "restartPolicyType": "ON_FAILURE",
                "restartPolicyMaxRetries": 10,
                "healthcheckPath": self.config.health_check_path,
                "healthcheckTimeout": 30
            }
        }
        return json.dumps(config, indent=2)

    def _generate_procfile(self, is_python: bool) -> str:
        if is_python:
            return f"web: uvicorn main:app --host 0.0.0.0 --port $PORT"
        else:
            return f"web: npm start"

    def _generate_nixpacks(self, is_python: bool) -> str:
        if is_python:
            return f"""[phases.setup]
nixPkgs = ["python311"]

[phases.install]
cmds = ["pip install -r requirements.txt"]

[start]
cmd = "uvicorn main:app --host 0.0.0.0 --port ${{PORT:-{self.config.port}}}"
"""
        else:
            return f"""[phases.setup]
nixPkgs = ["nodejs-20_x"]

[phases.install]
cmds = ["npm ci"]

[phases.build]
cmds = ["npm run build"]

[start]
cmd = "npm start"
"""


class CloudGenerator:
    """
    Unified cloud deployment generator.

    Supports multiple platforms with consistent interface.
    """

    PLATFORMS = {
        "aws": AWSGenerator,
        "gcp": GCPGenerator,
        "azure": AzureGenerator,
        "vercel": VercelGenerator,
        "railway": RailwayGenerator,
    }

    def __init__(self, config: Optional[CloudConfig] = None):
        self.config = config or CloudConfig()

    def generate(
        self,
        structure,
        output_path: str,
        platform: str
    ) -> Dict[str, str]:
        """
        Generate deployment configuration for specified platform.

        Args:
            structure: Project structure from analysis
            output_path: Output directory
            platform: Target platform (aws, gcp, azure, vercel, railway)

        Returns:
            Dictionary of generated files
        """
        if platform not in self.PLATFORMS:
            raise ValueError(f"Unknown platform: {platform}. Supported: {list(self.PLATFORMS.keys())}")

        generator_class = self.PLATFORMS[platform]
        generator = generator_class(self.config)

        return generator.generate(structure, output_path)

    def generate_all(self, structure, output_path: str) -> Dict[str, Dict[str, str]]:
        """Generate deployment configs for all platforms"""
        results = {}
        base = Path(output_path)

        for platform in self.PLATFORMS:
            platform_path = base / platform
            try:
                results[platform] = self.generate(structure, str(platform_path), platform)
            except Exception as e:
                results[platform] = {"error": str(e)}

        return results

    @classmethod
    def list_platforms(cls) -> List[Dict[str, str]]:
        """List available platforms"""
        return [
            {"name": "aws", "description": "AWS CloudFormation/SAM templates"},
            {"name": "gcp", "description": "GCP Terraform + Cloud Run"},
            {"name": "azure", "description": "Azure ARM templates"},
            {"name": "vercel", "description": "Vercel serverless deployment"},
            {"name": "railway", "description": "Railway PaaS deployment"},
        ]
