"""
Docker Generator - Generate Dockerfile and docker-compose.yml for projects
"""
from pathlib import Path
from typing import Dict, List, Optional
from dataclasses import dataclass

from ucts.core.models import ProjectStructure


@dataclass
class DockerConfig:
    """Configuration for Docker generation"""
    python_version: str = "3.11"
    node_version: str = "20"
    expose_port: int = 8000
    include_dev: bool = True
    use_alpine: bool = False
    multi_stage: bool = True


class DockerGenerator:
    """Generate Docker and docker-compose configuration"""

    def __init__(self, config: Optional[DockerConfig] = None):
        self.config = config or DockerConfig()

    def generate(self, structure: ProjectStructure, output_path: str) -> Dict[str, str]:
        """
        Generate Docker files for a project structure.

        Args:
            structure: Analyzed project structure
            output_path: Where to write the files

        Returns:
            Dict mapping filename to content
        """
        root = Path(output_path)
        root.mkdir(parents=True, exist_ok=True)

        files = {}

        # Determine project type and generate appropriate Dockerfile
        if 'python' in structure.languages:
            dockerfile = self._generate_python_dockerfile(structure)
            files['Dockerfile'] = dockerfile
        elif any(lang in structure.languages for lang in ('javascript', 'typescript')):
            dockerfile = self._generate_node_dockerfile(structure)
            files['Dockerfile'] = dockerfile

        # Generate docker-compose.yml
        compose = self._generate_compose(structure)
        files['docker-compose.yml'] = compose

        # Generate .dockerignore
        dockerignore = self._generate_dockerignore(structure)
        files['.dockerignore'] = dockerignore

        # Write files
        for filename, content in files.items():
            (root / filename).write_text(content, encoding='utf-8')

        return files

    def _generate_python_dockerfile(self, structure: ProjectStructure) -> str:
        """Generate Dockerfile for Python projects"""
        base_image = f"python:{self.config.python_version}"
        if self.config.use_alpine:
            base_image += "-alpine"
        else:
            base_image += "-slim"

        deps = structure.dependencies.get('python', [])
        has_flask = 'flask' in [d.lower() for d in deps]
        has_fastapi = 'fastapi' in [d.lower() for d in deps]
        has_django = 'django' in [d.lower() for d in deps]

        # Determine entry point
        if has_fastapi:
            cmd = 'CMD ["uvicorn", "main:app", "--host", "0.0.0.0", "--port", "8000"]'
        elif has_flask:
            cmd = 'CMD ["python", "-m", "flask", "run", "--host=0.0.0.0", "--port=8000"]'
        elif has_django:
            cmd = 'CMD ["python", "manage.py", "runserver", "0.0.0.0:8000"]'
        else:
            cmd = 'CMD ["python", "main.py"]'

        if self.config.multi_stage:
            return f'''# Build stage
FROM {base_image} AS builder

WORKDIR /app

# Install build dependencies
RUN pip install --no-cache-dir --upgrade pip setuptools wheel

# Copy requirements first for caching
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Production stage
FROM {base_image}

WORKDIR /app

# Create non-root user
RUN useradd --create-home --shell /bin/bash appuser

# Copy installed packages from builder
COPY --from=builder /usr/local/lib/python{self.config.python_version}/site-packages /usr/local/lib/python{self.config.python_version}/site-packages
COPY --from=builder /usr/local/bin /usr/local/bin

# Copy application code
COPY --chown=appuser:appuser . .

USER appuser

EXPOSE {self.config.expose_port}

{cmd}
'''
        else:
            return f'''FROM {base_image}

WORKDIR /app

# Install dependencies
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copy application
COPY . .

EXPOSE {self.config.expose_port}

{cmd}
'''

    def _generate_node_dockerfile(self, structure: ProjectStructure) -> str:
        """Generate Dockerfile for Node.js projects"""
        base_image = f"node:{self.config.node_version}"
        if self.config.use_alpine:
            base_image += "-alpine"
        else:
            base_image += "-slim"

        deps = structure.dependencies.get('node', [])
        has_next = 'next' in deps
        has_vite = 'vite' in deps

        if has_next:
            cmd = 'CMD ["npm", "run", "start"]'
            build_cmd = 'RUN npm run build'
        elif has_vite:
            cmd = 'CMD ["npm", "run", "preview"]'
            build_cmd = 'RUN npm run build'
        else:
            cmd = 'CMD ["npm", "start"]'
            build_cmd = ''

        if self.config.multi_stage:
            return f'''# Build stage
FROM {base_image} AS builder

WORKDIR /app

# Copy package files
COPY package*.json ./

# Install dependencies
RUN npm ci

# Copy source
COPY . .

# Build application
{build_cmd}

# Production stage
FROM {base_image}

WORKDIR /app

# Create non-root user
RUN addgroup --system --gid 1001 nodejs && \\
    adduser --system --uid 1001 appuser

COPY --from=builder /app/node_modules ./node_modules
COPY --from=builder /app/package*.json ./
COPY --from=builder --chown=appuser:nodejs /app .

USER appuser

EXPOSE {self.config.expose_port}

{cmd}
'''
        else:
            return f'''FROM {base_image}

WORKDIR /app

# Copy package files and install
COPY package*.json ./
RUN npm ci --only=production

# Copy application
COPY . .

EXPOSE {self.config.expose_port}

{cmd}
'''

    def _generate_compose(self, structure: ProjectStructure) -> str:
        """Generate docker-compose.yml"""
        services = []
        deps = structure.dependencies

        # Main application service
        app_service = f'''  app:
    build: .
    ports:
      - "{self.config.expose_port}:{self.config.expose_port}"
    environment:
      - NODE_ENV=production
      - PYTHONUNBUFFERED=1
    volumes:
      - .:/app:ro
    restart: unless-stopped'''

        services.append(app_service)

        # Add database services based on dependencies
        python_deps = [d.lower() for d in deps.get('python', [])]
        node_deps = [d.lower() for d in deps.get('node', [])]
        all_deps = python_deps + node_deps

        # PostgreSQL
        if any(d in all_deps for d in ['psycopg2', 'psycopg2-binary', 'asyncpg', 'pg', 'postgres']):
            services.append('''
  postgres:
    image: postgres:16-alpine
    environment:
      POSTGRES_USER: app
      POSTGRES_PASSWORD: app_password
      POSTGRES_DB: app_db
    ports:
      - "5432:5432"
    volumes:
      - postgres_data:/var/lib/postgresql/data
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U app"]
      interval: 5s
      timeout: 5s
      retries: 5''')

        # Redis
        if any(d in all_deps for d in ['redis', 'ioredis', 'aioredis']):
            services.append('''
  redis:
    image: redis:7-alpine
    ports:
      - "6379:6379"
    volumes:
      - redis_data:/data
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 5s
      timeout: 5s
      retries: 5''')

        # MongoDB
        if any(d in all_deps for d in ['pymongo', 'motor', 'mongodb', 'mongoose']):
            services.append('''
  mongodb:
    image: mongo:7
    ports:
      - "27017:27017"
    environment:
      MONGO_INITDB_ROOT_USERNAME: root
      MONGO_INITDB_ROOT_PASSWORD: root_password
    volumes:
      - mongo_data:/data/db''')

        # MySQL/MariaDB
        if any(d in all_deps for d in ['mysql', 'mysql-connector', 'pymysql', 'mysql2']):
            services.append('''
  mysql:
    image: mysql:8
    environment:
      MYSQL_ROOT_PASSWORD: root_password
      MYSQL_DATABASE: app_db
      MYSQL_USER: app
      MYSQL_PASSWORD: app_password
    ports:
      - "3306:3306"
    volumes:
      - mysql_data:/var/lib/mysql''')

        # Build the compose file
        compose = f'''version: "3.8"

services:
{chr(10).join(services)}
'''

        # Add volumes if needed
        volumes = []
        if 'postgres:' in compose:
            volumes.append('  postgres_data:')
        if 'redis:' in compose:
            volumes.append('  redis_data:')
        if 'mongodb:' in compose:
            volumes.append('  mongo_data:')
        if 'mysql:' in compose:
            volumes.append('  mysql_data:')

        if volumes:
            compose += f'''
volumes:
{chr(10).join(volumes)}
'''

        return compose

    def _generate_dockerignore(self, structure: ProjectStructure) -> str:
        """Generate .dockerignore file"""
        patterns = [
            '# Git',
            '.git',
            '.gitignore',
            '',
            '# IDE',
            '.idea/',
            '.vscode/',
            '*.swp',
            '*.swo',
            '',
            '# Docker',
            'Dockerfile*',
            'docker-compose*.yml',
            '.docker/',
            '',
            '# Documentation',
            '*.md',
            'docs/',
            '',
            '# Tests',
            'tests/',
            'test/',
            '__tests__/',
            'coverage/',
            '.coverage',
            'htmlcov/',
            '',
        ]

        if 'python' in structure.languages:
            patterns.extend([
                '# Python',
                '__pycache__/',
                '*.py[cod]',
                '*$py.class',
                '.Python',
                'build/',
                'dist/',
                '*.egg-info/',
                '.eggs/',
                '.venv/',
                'venv/',
                'env/',
                '.env',
                '.mypy_cache/',
                '.pytest_cache/',
                '',
            ])

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            patterns.extend([
                '# Node.js',
                'node_modules/',
                'npm-debug.log*',
                'yarn-debug.log*',
                'yarn-error.log*',
                '.npm/',
                '.yarn/',
                '',
            ])

        return '\n'.join(patterns)


def add_docker_to_project(structure: ProjectStructure, output_path: str,
                          config: Optional[DockerConfig] = None) -> Dict[str, str]:
    """
    Convenience function to add Docker support to a project.

    Args:
        structure: The project structure
        output_path: Where to write Docker files
        config: Optional Docker configuration

    Returns:
        Dict of generated files
    """
    generator = DockerGenerator(config)
    return generator.generate(structure, output_path)
