"""
GitHub Repository Generator

Creates a GitHub repository from a ProjectStructure.
"""
import logging
import subprocess
import tempfile
from pathlib import Path
from typing import Optional, List

import requests

from ucts.core.models import ProjectStructure
from ucts.generators.vscode import VSCodeGenerator

# Configure logging
logger = logging.getLogger(__name__)


class GitHubGeneratorError(Exception):
    """Base exception for GitHub generator errors"""
    pass


class GitHubAPIError(GitHubGeneratorError):
    """Error from GitHub API"""
    pass


class GitCommandError(GitHubGeneratorError):
    """Error running git command"""
    pass


class GitHubGenerator:
    """Generate GitHub repository from project structure"""

    def __init__(self, token: str):
        """
        Initialize GitHub generator.

        Args:
            token: GitHub personal access token

        Raises:
            ValueError: If token is empty
        """
        if not token:
            raise ValueError("token cannot be empty")

        self.token = token
        self.api_url = "https://api.github.com"
        self.headers = {
            "Authorization": f"Bearer {token}",
            "Accept": "application/vnd.github+json",
            "X-GitHub-Api-Version": "2022-11-28"
        }
        logger.debug("Initialized GitHub generator")

    def generate(self, structure: ProjectStructure,
                 org: Optional[str] = None,
                 private: bool = True) -> str:
        """
        Create GitHub repository from project structure.

        Args:
            structure: ProjectStructure to generate from
            org: Optional organization name
            private: Whether to create a private repository

        Returns:
            URL of the created GitHub repository

        Raises:
            ValueError: If structure is None
            GitHubAPIError: If API calls fail
            GitCommandError: If git operations fail
        """
        if structure is None:
            raise ValueError("structure cannot be None")

        logger.info(f"Generating GitHub repository: {structure.name}")

        # First, generate local project using VSCodeGenerator
        with tempfile.TemporaryDirectory() as tmpdir:
            local_path = Path(tmpdir) / structure.name
            vscode_gen = VSCodeGenerator()
            vscode_gen.generate(structure, str(local_path))

            # Add GitHub Actions workflow
            self._add_github_actions(local_path, structure)

            # Initialize git
            self._run_git(['init'], local_path)
            self._run_git(['add', '.'], local_path)
            self._run_git(['commit', '-m', 'Initial commit from UCTS'], local_path)

            # Create GitHub repository via API
            repo = self._create_github_repo(
                structure.name,
                structure.description,
                org,
                private
            )

            # Push to GitHub
            remote_url = repo['clone_url']
            # Embed token for HTTPS push
            remote_url = remote_url.replace(
                'https://',
                f'https://{self.token}@'
            )

            self._run_git(['remote', 'add', 'origin', remote_url], local_path)
            self._run_git(['branch', '-M', 'main'], local_path)
            self._run_git(['push', '-u', 'origin', 'main'], local_path)

            logger.info(f"Created GitHub repository: {repo['html_url']}")
            return repo['html_url']

    def _create_github_repo(self, name: str, description: str,
                            org: Optional[str], private: bool) -> dict:
        """Create GitHub repository via API"""
        data = {
            'name': name,
            'description': description[:255] if description else '',
            'private': private,
            'auto_init': False,
        }

        if org:
            url = f"{self.api_url}/orgs/{org}/repos"
        else:
            url = f"{self.api_url}/user/repos"

        try:
            response = requests.post(url, headers=self.headers, json=data, timeout=60)
        except requests.RequestException as e:
            raise GitHubAPIError(f"Failed to connect to GitHub API: {e}")

        if not response.ok:
            raise GitHubAPIError(f"Failed to create GitHub repo: {response.text}")

        return response.json()

    def _add_github_actions(self, path: Path, structure: ProjectStructure):
        """Add production-grade GitHub Actions workflows"""
        workflows_dir = path / '.github' / 'workflows'
        workflows_dir.mkdir(parents=True, exist_ok=True)

        languages = structure.languages if structure.languages else []
        has_python = 'python' in languages
        has_node = any(lang in languages for lang in ('javascript', 'typescript'))

        # Create CI workflow
        ci_workflow = '''# UCTS Generated GitHub Actions CI/CD Configuration
# Production-ready pipeline with linting, testing, building, and publishing

name: CI

on:
  push:
    branches: [main, develop]
  pull_request:
    branches: [main]

concurrency:
  group: ${{ github.workflow }}-${{ github.ref }}
  cancel-in-progress: true

jobs:
'''

        if has_python:
            ci_workflow += '''  lint-python:
    name: Lint Python
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Set up Python
        uses: actions/setup-python@v5
        with:
          python-version: '3.11'
          cache: 'pip'

      - name: Install linting tools
        run: pip install black isort ruff mypy

      - name: Run Black
        run: black --check . || echo "::warning::Run 'black .' to fix formatting"
        continue-on-error: true

      - name: Run isort
        run: isort --check-only . || echo "::warning::Run 'isort .' to fix imports"
        continue-on-error: true

      - name: Run Ruff
        run: ruff check .
        continue-on-error: true

  test-python:
    name: Test Python ${{ matrix.python-version }}
    runs-on: ubuntu-latest
    strategy:
      fail-fast: false
      matrix:
        python-version: ['3.9', '3.10', '3.11', '3.12']

    steps:
      - uses: actions/checkout@v4

      - name: Set up Python ${{ matrix.python-version }}
        uses: actions/setup-python@v5
        with:
          python-version: ${{ matrix.python-version }}
          cache: 'pip'

      - name: Install dependencies
        run: |
          python -m pip install --upgrade pip
          pip install -e ".[dev]" || pip install -r requirements.txt
          pip install pytest pytest-cov

      - name: Run tests
        run: pytest tests/ -v --cov=. --cov-report=xml --cov-report=term

      - name: Upload coverage to Codecov
        uses: codecov/codecov-action@v4
        with:
          files: ./coverage.xml
          fail_ci_if_error: false

  build-python:
    name: Build Python Package
    runs-on: ubuntu-latest
    needs: [test-python]
    steps:
      - uses: actions/checkout@v4

      - name: Set up Python
        uses: actions/setup-python@v5
        with:
          python-version: '3.11'
          cache: 'pip'

      - name: Install build tools
        run: pip install build twine

      - name: Build package
        run: python -m build

      - name: Check package
        run: twine check dist/*

      - name: Upload artifacts
        uses: actions/upload-artifact@v4
        with:
          name: python-dist
          path: dist/
          retention-days: 7

'''

        if has_node:
            ci_workflow += '''  lint-node:
    name: Lint Node.js
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Use Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'
          cache: 'npm'

      - name: Install dependencies
        run: npm ci

      - name: Run linter
        run: npm run lint || echo "::warning::Linting issues found"
        continue-on-error: true

  test-node:
    name: Test Node.js ${{ matrix.node-version }}
    runs-on: ubuntu-latest
    strategy:
      fail-fast: false
      matrix:
        node-version: ['18', '20', '22']

    steps:
      - uses: actions/checkout@v4

      - name: Use Node.js ${{ matrix.node-version }}
        uses: actions/setup-node@v4
        with:
          node-version: ${{ matrix.node-version }}
          cache: 'npm'

      - name: Install dependencies
        run: npm ci

      - name: Run tests
        run: npm test

  build-node:
    name: Build Node.js
    runs-on: ubuntu-latest
    needs: [test-node]
    steps:
      - uses: actions/checkout@v4

      - name: Use Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'
          cache: 'npm'

      - name: Install dependencies
        run: npm ci

      - name: Build
        run: npm run build

      - name: Upload artifacts
        uses: actions/upload-artifact@v4
        with:
          name: node-dist
          path: |
            dist/
            build/
          retention-days: 7

'''

        # If no specific languages detected, add generic job
        if not has_python and not has_node:
            ci_workflow += '''  test:
    name: Test
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Run tests
        run: echo "Add your test commands here"

  build:
    name: Build
    runs-on: ubuntu-latest
    needs: [test]
    steps:
      - uses: actions/checkout@v4

      - name: Build
        run: echo "Add your build commands here"

'''

        (workflows_dir / 'ci.yml').write_text(ci_workflow, encoding='utf-8')

        # Create release/publish workflow
        publish_workflow = '''# UCTS Generated Release/Publish Workflow

name: Release

on:
  push:
    tags:
      - 'v*'

jobs:
'''

        if has_python:
            publish_workflow += '''  publish-pypi:
    name: Publish to PyPI
    runs-on: ubuntu-latest
    environment:
      name: pypi
      url: https://pypi.org/p/${{ github.event.repository.name }}
    permissions:
      id-token: write  # For trusted publishing

    steps:
      - uses: actions/checkout@v4

      - name: Set up Python
        uses: actions/setup-python@v5
        with:
          python-version: '3.11'

      - name: Install build tools
        run: pip install build

      - name: Build package
        run: python -m build

      - name: Publish to PyPI
        uses: pypa/gh-action-pypi-publish@release/v1

'''

        if has_node:
            publish_workflow += '''  publish-npm:
    name: Publish to npm
    runs-on: ubuntu-latest
    environment:
      name: npm
      url: https://www.npmjs.com/package/${{ github.event.repository.name }}

    steps:
      - uses: actions/checkout@v4

      - name: Use Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'
          registry-url: 'https://registry.npmjs.org'

      - name: Install dependencies
        run: npm ci

      - name: Build
        run: npm run build

      - name: Publish to npm
        run: npm publish --access public
        env:
          NODE_AUTH_TOKEN: ${{ secrets.NPM_TOKEN }}

'''

        # Add GitHub release job
        publish_workflow += '''  create-release:
    name: Create GitHub Release
    runs-on: ubuntu-latest
    permissions:
      contents: write

    steps:
      - uses: actions/checkout@v4

      - name: Create Release
        uses: softprops/action-gh-release@v2
        with:
          generate_release_notes: true
          draft: false
          prerelease: ${{ contains(github.ref, 'alpha') || contains(github.ref, 'beta') || contains(github.ref, 'rc') }}
'''

        (workflows_dir / 'release.yml').write_text(publish_workflow, encoding='utf-8')
        logger.debug("Added production GitHub Actions workflows")

    def _run_git(self, args: List[str], cwd: Path, timeout: int = 120):
        """Run git command with timeout and error handling"""
        try:
            result = subprocess.run(
                ['git'] + args,
                cwd=cwd,
                capture_output=True,
                text=True,
                timeout=timeout
            )
            if result.returncode != 0:
                raise GitCommandError(f"Git command failed: {result.stderr}")
            return result.stdout
        except subprocess.TimeoutExpired:
            raise GitCommandError(f"Git command timed out after {timeout}s: git {' '.join(args)}")
        except FileNotFoundError:
            raise GitCommandError("git executable not found in PATH")
