"""
GitLab Repository Generator

Creates a GitLab repository from a ProjectStructure.
"""
import logging
import subprocess
import tempfile
from pathlib import Path
from typing import Optional, List

import requests

from ucts.core.models import ProjectStructure
from ucts.generators.vscode import VSCodeGenerator

# Configure logging
logger = logging.getLogger(__name__)


class GitLabGeneratorError(Exception):
    """Base exception for GitLab generator errors"""
    pass


class GitLabAPIError(GitLabGeneratorError):
    """Error from GitLab API"""
    pass


class GitCommandError(GitLabGeneratorError):
    """Error running git command"""
    pass


class GitLabGenerator:
    """Generate GitLab repository from project structure"""

    def __init__(self, gitlab_url: str, token: str):
        """
        Initialize GitLab generator.

        Args:
            gitlab_url: GitLab server URL (e.g., https://gitlab.com)
            token: GitLab API access token

        Raises:
            ValueError: If gitlab_url or token is empty
        """
        if not gitlab_url:
            raise ValueError("gitlab_url cannot be empty")
        if not token:
            raise ValueError("token cannot be empty")

        self.gitlab_url = gitlab_url.rstrip('/')
        self.token = token
        self.api_url = f"{self.gitlab_url}/api/v4"
        self.headers = {"PRIVATE-TOKEN": token}
        logger.debug(f"Initialized GitLab generator for {self.gitlab_url}")

    def generate(self, structure: ProjectStructure,
                 group: Optional[str] = None,
                 visibility: str = 'private') -> str:
        """
        Create GitLab repository from project structure.

        Args:
            structure: ProjectStructure to generate from
            group: Optional group/namespace for the project
            visibility: 'private', 'internal', or 'public'

        Returns:
            URL of the created GitLab project

        Raises:
            ValueError: If structure is None or visibility is invalid
            GitLabAPIError: If API calls fail
            GitCommandError: If git operations fail
        """
        if structure is None:
            raise ValueError("structure cannot be None")

        if visibility not in ('private', 'internal', 'public'):
            raise ValueError(f"Invalid visibility: {visibility}")

        logger.info(f"Generating GitLab repository: {structure.name}")

        # First, generate local project using VSCodeGenerator
        with tempfile.TemporaryDirectory() as tmpdir:
            local_path = Path(tmpdir) / structure.name
            vscode_gen = VSCodeGenerator()
            vscode_gen.generate(structure, str(local_path))

            # Add GitLab CI configuration
            self._add_gitlab_ci(local_path, structure)

            # Initialize git
            self._run_git(['init'], local_path)
            self._run_git(['add', '.'], local_path)
            self._run_git(['commit', '-m', 'Initial commit from UCTS'], local_path)

            # Create GitLab project via API
            project = self._create_gitlab_project(
                structure.name,
                structure.description,
                group,
                visibility
            )

            # Push to GitLab
            remote_url = project['ssh_url_to_repo']

            # Try HTTPS if SSH not configured
            if not self._ssh_available():
                remote_url = project['http_url_to_repo']
                # Embed token for HTTPS push
                remote_url = remote_url.replace(
                    'https://',
                    f'https://oauth2:{self.token}@'
                )

            self._run_git(['remote', 'add', 'origin', remote_url], local_path)
            self._run_git(['branch', '-M', 'main'], local_path)
            self._run_git(['push', '-u', 'origin', 'main'], local_path)

            logger.info(f"Created GitLab repository: {project['web_url']}")
            return project['web_url']

    def _create_gitlab_project(self, name: str, description: str,
                               group: Optional[str], visibility: str) -> dict:
        """Create GitLab project via API"""
        data = {
            'name': name,
            'description': description[:255] if description else '',
            'visibility': visibility,
            'initialize_with_readme': False,
        }

        if group:
            # Get namespace ID
            try:
                ns_response = requests.get(
                    f"{self.api_url}/namespaces",
                    headers=self.headers,
                    params={'search': group},
                    timeout=30
                )
                if ns_response.ok and ns_response.json():
                    data['namespace_id'] = ns_response.json()[0]['id']
            except requests.RequestException as e:
                logger.warning(f"Could not find namespace {group}: {e}")

        try:
            response = requests.post(
                f"{self.api_url}/projects",
                headers=self.headers,
                json=data,
                timeout=60
            )
        except requests.RequestException as e:
            raise GitLabAPIError(f"Failed to connect to GitLab API: {e}")

        if not response.ok:
            raise GitLabAPIError(f"Failed to create GitLab project: {response.text}")

        return response.json()

    def _add_gitlab_ci(self, path: Path, structure: ProjectStructure):
        """Add production-grade .gitlab-ci.yml configuration"""
        languages = structure.languages if structure.languages else []
        has_python = 'python' in languages
        has_node = any(lang in languages for lang in ('javascript', 'typescript'))

        ci_config = '''# UCTS Generated GitLab CI/CD Configuration
# Production-ready pipeline with linting, testing, building, and publishing

stages:
  - lint
  - test
  - build
  - publish

variables:
  PIP_CACHE_DIR: "$CI_PROJECT_DIR/.pip-cache"
  PYTHON_VERSION: "3.11"

# Global cache configuration
cache:
  key: "${CI_COMMIT_REF_SLUG}"
  paths:
    - .pip-cache/
    - node_modules/
    - .npm/

'''

        if has_python:
            ci_config += '''# ======================
# Python Jobs
# ======================

lint-python:
  stage: lint
  image: python:${PYTHON_VERSION}
  script:
    - pip install black isort ruff mypy
    - black --check . || echo "Run 'black .' to fix formatting"
    - isort --check-only . || echo "Run 'isort .' to fix imports"
    - ruff check . || true
  allow_failure: true
  rules:
    - exists:
        - "**/*.py"

test-python:
  stage: test
  image: python:${PYTHON_VERSION}
  parallel:
    matrix:
      - PYTHON_VERSION: ['3.9', '3.10', '3.11', '3.12']
  script:
    - pip install -e ".[dev]" || pip install -r requirements.txt
    - pip install pytest pytest-cov
    - pytest tests/ -v --cov=. --cov-report=xml --cov-report=term
  coverage: '/TOTAL.*\\s+(\\d+%)$/'
  artifacts:
    reports:
      coverage_report:
        coverage_format: cobertura
        path: coverage.xml
    expire_in: 1 week
  rules:
    - exists:
        - "**/*.py"

build-python:
  stage: build
  image: python:${PYTHON_VERSION}
  script:
    - pip install build twine
    - python -m build
  artifacts:
    paths:
      - dist/
    expire_in: 1 week
  rules:
    - exists:
        - pyproject.toml
    - exists:
        - setup.py

publish-pypi-test:
  stage: publish
  image: python:${PYTHON_VERSION}
  script:
    - pip install twine
    - twine upload --repository testpypi dist/*
  dependencies:
    - build-python
  rules:
    - if: $CI_COMMIT_TAG
      when: manual
  environment:
    name: testpypi
    url: https://test.pypi.org/project/${CI_PROJECT_NAME}/

publish-pypi:
  stage: publish
  image: python:${PYTHON_VERSION}
  script:
    - pip install twine
    - twine upload dist/*
  dependencies:
    - build-python
  rules:
    - if: $CI_COMMIT_TAG
      when: manual
  environment:
    name: pypi
    url: https://pypi.org/project/${CI_PROJECT_NAME}/

'''

        if has_node:
            ci_config += '''# ======================
# Node.js Jobs
# ======================

lint-node:
  stage: lint
  image: node:20
  script:
    - npm ci --cache .npm --prefer-offline
    - npm run lint || echo "Linting issues found"
  allow_failure: true
  rules:
    - exists:
        - package.json

test-node:
  stage: test
  image: node:20
  parallel:
    matrix:
      - NODE_VERSION: ['18', '20', '22']
  script:
    - npm ci --cache .npm --prefer-offline
    - npm test
  coverage: '/All files[^|]*\\|[^|]*\\s+([\\d\\.]+)/'
  artifacts:
    reports:
      coverage_report:
        coverage_format: cobertura
        path: coverage/cobertura-coverage.xml
    expire_in: 1 week
  rules:
    - exists:
        - package.json

build-node:
  stage: build
  image: node:20
  script:
    - npm ci --cache .npm --prefer-offline
    - npm run build
  artifacts:
    paths:
      - dist/
      - build/
    expire_in: 1 week
  rules:
    - exists:
        - package.json

publish-npm:
  stage: publish
  image: node:20
  script:
    - echo "//registry.npmjs.org/:_authToken=${NPM_TOKEN}" > .npmrc
    - npm publish --access public
  dependencies:
    - build-node
  rules:
    - if: $CI_COMMIT_TAG
      when: manual
  environment:
    name: npm
    url: https://www.npmjs.com/package/${CI_PROJECT_NAME}

'''

        # If no specific languages detected, add generic jobs
        if not has_python and not has_node:
            ci_config += '''# ======================
# Generic Jobs
# ======================

test:
  stage: test
  image: alpine:latest
  script:
    - echo "Add your test commands here"
    - echo "No specific language detected"
  allow_failure: true

build:
  stage: build
  image: alpine:latest
  script:
    - echo "Add your build commands here"
  artifacts:
    paths:
      - dist/
    expire_in: 1 week

'''

        (path / '.gitlab-ci.yml').write_text(ci_config, encoding='utf-8')
        logger.debug("Added production GitLab CI configuration")

    def _run_git(self, args: List[str], cwd: Path, timeout: int = 120):
        """Run git command with timeout and error handling"""
        try:
            result = subprocess.run(
                ['git'] + args,
                cwd=cwd,
                capture_output=True,
                text=True,
                timeout=timeout
            )
            if result.returncode != 0:
                raise GitCommandError(f"Git command failed: {result.stderr}")
            return result.stdout
        except subprocess.TimeoutExpired:
            raise GitCommandError(f"Git command timed out after {timeout}s: git {' '.join(args)}")
        except FileNotFoundError:
            raise GitCommandError("git executable not found in PATH")

    def _ssh_available(self) -> bool:
        """Check if SSH is available for git operations"""
        try:
            result = subprocess.run(
                ['ssh', '-T', '-o', 'BatchMode=yes', '-o', 'ConnectTimeout=5',
                 f'git@{self.gitlab_url.replace("https://", "").replace("http://", "").split("/")[0]}'],
                capture_output=True,
                timeout=10
            )
            # SSH auth failure returns 1, but connection works
            return True
        except (subprocess.TimeoutExpired, FileNotFoundError, Exception):
            return False
