"""
Kubernetes Deployment Generator

Generates Kubernetes manifests and Helm charts:
- Deployment, Service, ConfigMap, Secret
- Helm chart with values
- Kustomize overlays
- Ingress configuration
"""

import json
import yaml
from pathlib import Path
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, field


@dataclass
class K8sConfig:
    """Configuration for Kubernetes deployment generation"""
    name: str = "app"
    namespace: str = "default"
    # Container
    image: str = ""
    tag: str = "latest"
    port: int = 8000
    replicas: int = 2
    # Resources
    cpu_request: str = "100m"
    cpu_limit: str = "500m"
    memory_request: str = "128Mi"
    memory_limit: str = "512Mi"
    # Health
    health_path: str = "/health"
    readiness_path: str = "/ready"
    # Ingress
    enable_ingress: bool = True
    ingress_host: str = ""
    ingress_class: str = "nginx"
    enable_tls: bool = True
    # Autoscaling
    enable_hpa: bool = True
    min_replicas: int = 2
    max_replicas: int = 10
    target_cpu: int = 80
    # Environment
    env_vars: Dict[str, str] = field(default_factory=dict)
    secrets: Dict[str, str] = field(default_factory=dict)


class KubernetesGenerator:
    """Generate Kubernetes manifests"""

    def __init__(self, config: Optional[K8sConfig] = None):
        self.config = config or K8sConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Kubernetes manifests"""
        files = {}

        # Update config from structure
        if not self.config.image:
            self.config.image = f"{self.config.name}"
        if not self.config.ingress_host:
            self.config.ingress_host = f"{self.config.name}.example.com"

        # Namespace
        files["namespace.yaml"] = self._generate_namespace()

        # ConfigMap
        files["configmap.yaml"] = self._generate_configmap()

        # Secret
        files["secret.yaml"] = self._generate_secret()

        # Deployment
        files["deployment.yaml"] = self._generate_deployment()

        # Service
        files["service.yaml"] = self._generate_service()

        # Ingress
        if self.config.enable_ingress:
            files["ingress.yaml"] = self._generate_ingress()

        # HPA
        if self.config.enable_hpa:
            files["hpa.yaml"] = self._generate_hpa()

        # Kustomization
        files["kustomization.yaml"] = self._generate_kustomization()

        # Write files
        output = Path(output_path)
        output.mkdir(parents=True, exist_ok=True)

        for filename, content in files.items():
            (output / filename).write_text(content)

        return files

    def _generate_namespace(self) -> str:
        manifest = {
            "apiVersion": "v1",
            "kind": "Namespace",
            "metadata": {
                "name": self.config.namespace,
                "labels": {
                    "app.kubernetes.io/name": self.config.name
                }
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_configmap(self) -> str:
        manifest = {
            "apiVersion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": f"{self.config.name}-config",
                "namespace": self.config.namespace
            },
            "data": {
                "ENVIRONMENT": "production",
                "LOG_LEVEL": "info",
                **self.config.env_vars
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_secret(self) -> str:
        manifest = {
            "apiVersion": "v1",
            "kind": "Secret",
            "metadata": {
                "name": f"{self.config.name}-secrets",
                "namespace": self.config.namespace
            },
            "type": "Opaque",
            "stringData": {
                "placeholder": "replace-with-actual-secrets",
                **self.config.secrets
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_deployment(self) -> str:
        manifest = {
            "apiVersion": "apps/v1",
            "kind": "Deployment",
            "metadata": {
                "name": self.config.name,
                "namespace": self.config.namespace,
                "labels": {
                    "app": self.config.name
                }
            },
            "spec": {
                "replicas": self.config.replicas,
                "selector": {
                    "matchLabels": {
                        "app": self.config.name
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "app": self.config.name
                        }
                    },
                    "spec": {
                        "containers": [
                            {
                                "name": self.config.name,
                                "image": f"{self.config.image}:{self.config.tag}",
                                "ports": [
                                    {
                                        "containerPort": self.config.port,
                                        "protocol": "TCP"
                                    }
                                ],
                                "envFrom": [
                                    {
                                        "configMapRef": {
                                            "name": f"{self.config.name}-config"
                                        }
                                    },
                                    {
                                        "secretRef": {
                                            "name": f"{self.config.name}-secrets"
                                        }
                                    }
                                ],
                                "resources": {
                                    "requests": {
                                        "cpu": self.config.cpu_request,
                                        "memory": self.config.memory_request
                                    },
                                    "limits": {
                                        "cpu": self.config.cpu_limit,
                                        "memory": self.config.memory_limit
                                    }
                                },
                                "livenessProbe": {
                                    "httpGet": {
                                        "path": self.config.health_path,
                                        "port": self.config.port
                                    },
                                    "initialDelaySeconds": 10,
                                    "periodSeconds": 10,
                                    "failureThreshold": 3
                                },
                                "readinessProbe": {
                                    "httpGet": {
                                        "path": self.config.readiness_path,
                                        "port": self.config.port
                                    },
                                    "initialDelaySeconds": 5,
                                    "periodSeconds": 5,
                                    "failureThreshold": 3
                                }
                            }
                        ]
                    }
                }
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_service(self) -> str:
        manifest = {
            "apiVersion": "v1",
            "kind": "Service",
            "metadata": {
                "name": self.config.name,
                "namespace": self.config.namespace
            },
            "spec": {
                "type": "ClusterIP",
                "selector": {
                    "app": self.config.name
                },
                "ports": [
                    {
                        "port": 80,
                        "targetPort": self.config.port,
                        "protocol": "TCP"
                    }
                ]
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_ingress(self) -> str:
        manifest = {
            "apiVersion": "networking.k8s.io/v1",
            "kind": "Ingress",
            "metadata": {
                "name": self.config.name,
                "namespace": self.config.namespace,
                "annotations": {
                    "kubernetes.io/ingress.class": self.config.ingress_class,
                    "nginx.ingress.kubernetes.io/ssl-redirect": "true" if self.config.enable_tls else "false"
                }
            },
            "spec": {
                "rules": [
                    {
                        "host": self.config.ingress_host,
                        "http": {
                            "paths": [
                                {
                                    "path": "/",
                                    "pathType": "Prefix",
                                    "backend": {
                                        "service": {
                                            "name": self.config.name,
                                            "port": {
                                                "number": 80
                                            }
                                        }
                                    }
                                }
                            ]
                        }
                    }
                ]
            }
        }

        if self.config.enable_tls:
            manifest["spec"]["tls"] = [
                {
                    "hosts": [self.config.ingress_host],
                    "secretName": f"{self.config.name}-tls"
                }
            ]

        return yaml.dump(manifest, default_flow_style=False)

    def _generate_hpa(self) -> str:
        manifest = {
            "apiVersion": "autoscaling/v2",
            "kind": "HorizontalPodAutoscaler",
            "metadata": {
                "name": self.config.name,
                "namespace": self.config.namespace
            },
            "spec": {
                "scaleTargetRef": {
                    "apiVersion": "apps/v1",
                    "kind": "Deployment",
                    "name": self.config.name
                },
                "minReplicas": self.config.min_replicas,
                "maxReplicas": self.config.max_replicas,
                "metrics": [
                    {
                        "type": "Resource",
                        "resource": {
                            "name": "cpu",
                            "target": {
                                "type": "Utilization",
                                "averageUtilization": self.config.target_cpu
                            }
                        }
                    }
                ]
            }
        }
        return yaml.dump(manifest, default_flow_style=False)

    def _generate_kustomization(self) -> str:
        resources = [
            "namespace.yaml",
            "configmap.yaml",
            "secret.yaml",
            "deployment.yaml",
            "service.yaml"
        ]
        if self.config.enable_ingress:
            resources.append("ingress.yaml")
        if self.config.enable_hpa:
            resources.append("hpa.yaml")

        manifest = {
            "apiVersion": "kustomize.config.k8s.io/v1beta1",
            "kind": "Kustomization",
            "namespace": self.config.namespace,
            "resources": resources,
            "commonLabels": {
                "app.kubernetes.io/name": self.config.name,
                "app.kubernetes.io/managed-by": "ucts"
            }
        }
        return yaml.dump(manifest, default_flow_style=False)


class HelmGenerator:
    """Generate Helm charts"""

    def __init__(self, config: Optional[K8sConfig] = None):
        self.config = config or K8sConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Helm chart"""
        files = {}

        chart_name = self.config.name

        # Chart.yaml
        files["Chart.yaml"] = self._generate_chart_yaml()

        # values.yaml
        files["values.yaml"] = self._generate_values()

        # templates/deployment.yaml
        files["templates/deployment.yaml"] = self._generate_deployment_template()

        # templates/service.yaml
        files["templates/service.yaml"] = self._generate_service_template()

        # templates/ingress.yaml
        files["templates/ingress.yaml"] = self._generate_ingress_template()

        # templates/configmap.yaml
        files["templates/configmap.yaml"] = self._generate_configmap_template()

        # templates/secret.yaml
        files["templates/secret.yaml"] = self._generate_secret_template()

        # templates/hpa.yaml
        files["templates/hpa.yaml"] = self._generate_hpa_template()

        # templates/_helpers.tpl
        files["templates/_helpers.tpl"] = self._generate_helpers()

        # templates/NOTES.txt
        files["templates/NOTES.txt"] = self._generate_notes()

        # Write files
        output = Path(output_path)

        for filename, content in files.items():
            filepath = output / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)

        return files

    def _generate_chart_yaml(self) -> str:
        chart = {
            "apiVersion": "v2",
            "name": self.config.name,
            "description": f"Helm chart for {self.config.name} - Generated by UCTS",
            "type": "application",
            "version": "0.1.0",
            "appVersion": "1.0.0",
            "maintainers": [
                {
                    "name": "UCTS",
                    "url": "https://github.com/ucts"
                }
            ]
        }
        return yaml.dump(chart, default_flow_style=False)

    def _generate_values(self) -> str:
        values = {
            "replicaCount": self.config.replicas,
            "image": {
                "repository": self.config.image or self.config.name,
                "tag": self.config.tag,
                "pullPolicy": "IfNotPresent"
            },
            "service": {
                "type": "ClusterIP",
                "port": 80,
                "targetPort": self.config.port
            },
            "ingress": {
                "enabled": self.config.enable_ingress,
                "className": self.config.ingress_class,
                "hosts": [
                    {
                        "host": self.config.ingress_host,
                        "paths": [
                            {
                                "path": "/",
                                "pathType": "Prefix"
                            }
                        ]
                    }
                ],
                "tls": [
                    {
                        "secretName": f"{self.config.name}-tls",
                        "hosts": [self.config.ingress_host]
                    }
                ] if self.config.enable_tls else []
            },
            "resources": {
                "requests": {
                    "cpu": self.config.cpu_request,
                    "memory": self.config.memory_request
                },
                "limits": {
                    "cpu": self.config.cpu_limit,
                    "memory": self.config.memory_limit
                }
            },
            "autoscaling": {
                "enabled": self.config.enable_hpa,
                "minReplicas": self.config.min_replicas,
                "maxReplicas": self.config.max_replicas,
                "targetCPUUtilizationPercentage": self.config.target_cpu
            },
            "livenessProbe": {
                "httpGet": {
                    "path": self.config.health_path,
                    "port": "http"
                },
                "initialDelaySeconds": 10,
                "periodSeconds": 10
            },
            "readinessProbe": {
                "httpGet": {
                    "path": self.config.readiness_path,
                    "port": "http"
                },
                "initialDelaySeconds": 5,
                "periodSeconds": 5
            },
            "env": self.config.env_vars,
            "secrets": {}
        }
        return yaml.dump(values, default_flow_style=False)

    def _generate_deployment_template(self) -> str:
        return '''apiVersion: apps/v1
kind: Deployment
metadata:
  name: {{ include "chart.fullname" . }}
  labels:
    {{- include "chart.labels" . | nindent 4 }}
spec:
  {{- if not .Values.autoscaling.enabled }}
  replicas: {{ .Values.replicaCount }}
  {{- end }}
  selector:
    matchLabels:
      {{- include "chart.selectorLabels" . | nindent 6 }}
  template:
    metadata:
      labels:
        {{- include "chart.selectorLabels" . | nindent 8 }}
    spec:
      containers:
        - name: {{ .Chart.Name }}
          image: "{{ .Values.image.repository }}:{{ .Values.image.tag }}"
          imagePullPolicy: {{ .Values.image.pullPolicy }}
          ports:
            - name: http
              containerPort: {{ .Values.service.targetPort }}
              protocol: TCP
          envFrom:
            - configMapRef:
                name: {{ include "chart.fullname" . }}-config
            - secretRef:
                name: {{ include "chart.fullname" . }}-secrets
          livenessProbe:
            {{- toYaml .Values.livenessProbe | nindent 12 }}
          readinessProbe:
            {{- toYaml .Values.readinessProbe | nindent 12 }}
          resources:
            {{- toYaml .Values.resources | nindent 12 }}
'''

    def _generate_service_template(self) -> str:
        return '''apiVersion: v1
kind: Service
metadata:
  name: {{ include "chart.fullname" . }}
  labels:
    {{- include "chart.labels" . | nindent 4 }}
spec:
  type: {{ .Values.service.type }}
  ports:
    - port: {{ .Values.service.port }}
      targetPort: {{ .Values.service.targetPort }}
      protocol: TCP
      name: http
  selector:
    {{- include "chart.selectorLabels" . | nindent 4 }}
'''

    def _generate_ingress_template(self) -> str:
        return '''{{- if .Values.ingress.enabled -}}
apiVersion: networking.k8s.io/v1
kind: Ingress
metadata:
  name: {{ include "chart.fullname" . }}
  labels:
    {{- include "chart.labels" . | nindent 4 }}
  {{- with .Values.ingress.annotations }}
  annotations:
    {{- toYaml . | nindent 4 }}
  {{- end }}
spec:
  ingressClassName: {{ .Values.ingress.className }}
  {{- if .Values.ingress.tls }}
  tls:
    {{- range .Values.ingress.tls }}
    - hosts:
        {{- range .hosts }}
        - {{ . | quote }}
        {{- end }}
      secretName: {{ .secretName }}
    {{- end }}
  {{- end }}
  rules:
    {{- range .Values.ingress.hosts }}
    - host: {{ .host | quote }}
      http:
        paths:
          {{- range .paths }}
          - path: {{ .path }}
            pathType: {{ .pathType }}
            backend:
              service:
                name: {{ include "chart.fullname" $ }}
                port:
                  number: {{ $.Values.service.port }}
          {{- end }}
    {{- end }}
{{- end }}
'''

    def _generate_configmap_template(self) -> str:
        return '''apiVersion: v1
kind: ConfigMap
metadata:
  name: {{ include "chart.fullname" . }}-config
  labels:
    {{- include "chart.labels" . | nindent 4 }}
data:
  ENVIRONMENT: "production"
  {{- range $key, $value := .Values.env }}
  {{ $key }}: {{ $value | quote }}
  {{- end }}
'''

    def _generate_secret_template(self) -> str:
        return '''apiVersion: v1
kind: Secret
metadata:
  name: {{ include "chart.fullname" . }}-secrets
  labels:
    {{- include "chart.labels" . | nindent 4 }}
type: Opaque
stringData:
  {{- range $key, $value := .Values.secrets }}
  {{ $key }}: {{ $value | quote }}
  {{- end }}
'''

    def _generate_hpa_template(self) -> str:
        return '''{{- if .Values.autoscaling.enabled }}
apiVersion: autoscaling/v2
kind: HorizontalPodAutoscaler
metadata:
  name: {{ include "chart.fullname" . }}
  labels:
    {{- include "chart.labels" . | nindent 4 }}
spec:
  scaleTargetRef:
    apiVersion: apps/v1
    kind: Deployment
    name: {{ include "chart.fullname" . }}
  minReplicas: {{ .Values.autoscaling.minReplicas }}
  maxReplicas: {{ .Values.autoscaling.maxReplicas }}
  metrics:
    - type: Resource
      resource:
        name: cpu
        target:
          type: Utilization
          averageUtilization: {{ .Values.autoscaling.targetCPUUtilizationPercentage }}
{{- end }}
'''

    def _generate_helpers(self) -> str:
        return f'''{{{{/*
Expand the name of the chart.
*/}}}}
{{{{- define "chart.name" -}}}}
{{{{- default .Chart.Name .Values.nameOverride | trunc 63 | trimSuffix "-" }}}}
{{{{- end }}}}

{{{{/*
Create a default fully qualified app name.
*/}}}}
{{{{- define "chart.fullname" -}}}}
{{{{- if .Values.fullnameOverride }}}}
{{{{- .Values.fullnameOverride | trunc 63 | trimSuffix "-" }}}}
{{{{- else }}}}
{{{{- $name := default .Chart.Name .Values.nameOverride }}}}
{{{{- if contains $name .Release.Name }}}}
{{{{- .Release.Name | trunc 63 | trimSuffix "-" }}}}
{{{{- else }}}}
{{{{- printf "%s-%s" .Release.Name $name | trunc 63 | trimSuffix "-" }}}}
{{{{- end }}}}
{{{{- end }}}}
{{{{- end }}}}

{{{{/*
Common labels
*/}}}}
{{{{- define "chart.labels" -}}}}
helm.sh/chart: {{{{ include "chart.name" . }}}}-{{{{ .Chart.Version | replace "+" "_" }}}}
{{{{ include "chart.selectorLabels" . }}}}
app.kubernetes.io/version: {{{{ .Chart.AppVersion | quote }}}}
app.kubernetes.io/managed-by: {{{{ .Release.Service }}}}
{{{{- end }}}}

{{{{/*
Selector labels
*/}}}}
{{{{- define "chart.selectorLabels" -}}}}
app.kubernetes.io/name: {{{{ include "chart.name" . }}}}
app.kubernetes.io/instance: {{{{ .Release.Name }}}}
{{{{- end }}}}
'''

    def _generate_notes(self) -> str:
        return f'''1. Get the application URL by running these commands:
{{{{- if .Values.ingress.enabled }}}}
  http{{{{- if .Values.ingress.tls }}}}s{{{{- end }}}}://{{{{ (index .Values.ingress.hosts 0).host }}}}
{{{{- else if contains "NodePort" .Values.service.type }}}}
  export NODE_PORT=$(kubectl get --namespace {{{{ .Release.Namespace }}}} -o jsonpath="{{{{.spec.ports[0].nodePort}}}}" services {{{{ include "chart.fullname" . }}}})
  export NODE_IP=$(kubectl get nodes --namespace {{{{ .Release.Namespace }}}} -o jsonpath="{{{{.items[0].status.addresses[0].address}}}}")
  echo http://$NODE_IP:$NODE_PORT
{{{{- else if contains "LoadBalancer" .Values.service.type }}}}
  export SERVICE_IP=$(kubectl get svc --namespace {{{{ .Release.Namespace }}}} {{{{ include "chart.fullname" . }}}} --template "{{{{{{ range (index .status.loadBalancer.ingress 0) }}}}{{{{{{ . }}}}}}{{{{{{ end }}}}}}}}")
  echo http://$SERVICE_IP:{{{{ .Values.service.port }}}}
{{{{- else if contains "ClusterIP" .Values.service.type }}}}
  kubectl port-forward --namespace {{{{ .Release.Namespace }}}} svc/{{{{ include "chart.fullname" . }}}} 8080:{{{{ .Values.service.port }}}}
  echo "Visit http://127.0.0.1:8080"
{{{{- end }}}}

Generated by UCTS - Universal Context Transfer System
'''
