"""
Project Merger - Merge UCTS output into existing projects

Handles intelligent merging of generated code into existing repositories.
"""
import json
import os
from pathlib import Path
from typing import Dict, List, Any, Optional, Tuple
from dataclasses import dataclass
from enum import Enum

from ucts.core.models import ProjectStructure


class MergeStrategy(Enum):
    """How to handle file conflicts"""
    SKIP = "skip"           # Skip existing files
    OVERWRITE = "overwrite" # Replace existing files
    APPEND = "append"       # Append to existing files
    SMART = "smart"         # Intelligent merge based on file type


@dataclass
class MergeResult:
    """Result of a merge operation"""
    files_created: List[str]
    files_updated: List[str]
    files_skipped: List[str]
    conflicts: List[str]
    dependencies_added: Dict[str, List[str]]


class ProjectMerger:
    """Merge generated project structure into existing projects"""

    def __init__(self, strategy: MergeStrategy = MergeStrategy.SMART):
        self.strategy = strategy

    def merge(self, structure: ProjectStructure, target_path: str) -> MergeResult:
        """
        Merge project structure into existing project.

        Args:
            structure: ProjectStructure to merge
            target_path: Path to existing project

        Returns:
            MergeResult with details of the merge
        """
        root = Path(target_path)

        if not root.exists():
            raise ValueError(f"Target path does not exist: {target_path}")

        result = MergeResult(
            files_created=[],
            files_updated=[],
            files_skipped=[],
            conflicts=[],
            dependencies_added={}
        )

        # Create new directories
        for dir_path in structure.directories:
            full_path = root / dir_path
            if not full_path.exists():
                full_path.mkdir(parents=True, exist_ok=True)

        # Merge files
        for file_path, content in structure.files.items():
            self._merge_file(root, file_path, content, result)

        # Merge dependencies
        if 'python' in structure.languages:
            self._merge_python_deps(root, structure, result)

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            self._merge_node_deps(root, structure, result)

        # Update .gitignore if needed
        self._merge_gitignore(root, structure)

        return result

    def _merge_file(self, root: Path, file_path: str, content: str,
                    result: MergeResult) -> None:
        """Merge a single file"""
        full_path = root / file_path

        # Ensure parent directory exists
        full_path.parent.mkdir(parents=True, exist_ok=True)

        if not full_path.exists():
            # New file - just create it
            full_path.write_text(content, encoding='utf-8')
            result.files_created.append(file_path)
            return

        # File exists - apply strategy
        if self.strategy == MergeStrategy.SKIP:
            result.files_skipped.append(file_path)

        elif self.strategy == MergeStrategy.OVERWRITE:
            full_path.write_text(content, encoding='utf-8')
            result.files_updated.append(file_path)

        elif self.strategy == MergeStrategy.APPEND:
            existing = full_path.read_text(encoding='utf-8')
            merged = existing + "\n\n# === UCTS APPENDED ===\n\n" + content
            full_path.write_text(merged, encoding='utf-8')
            result.files_updated.append(file_path)

        elif self.strategy == MergeStrategy.SMART:
            self._smart_merge_file(full_path, content, result, file_path)

    def _smart_merge_file(self, full_path: Path, new_content: str,
                          result: MergeResult, rel_path: str) -> None:
        """Intelligently merge based on file type"""
        suffix = full_path.suffix.lower()
        existing = full_path.read_text(encoding='utf-8')

        # JSON files - merge objects
        if suffix == '.json':
            merged = self._merge_json(existing, new_content)
            if merged:
                full_path.write_text(merged, encoding='utf-8')
                result.files_updated.append(rel_path)
            else:
                result.conflicts.append(rel_path)
            return

        # Python files - append with imports deduplication
        if suffix == '.py':
            merged = self._merge_python(existing, new_content)
            full_path.write_text(merged, encoding='utf-8')
            result.files_updated.append(rel_path)
            return

        # Markdown - append sections
        if suffix == '.md':
            merged = self._merge_markdown(existing, new_content)
            full_path.write_text(merged, encoding='utf-8')
            result.files_updated.append(rel_path)
            return

        # Requirements/dependencies - merge lines
        if full_path.name in ('requirements.txt', '.gitignore'):
            merged = self._merge_lines(existing, new_content)
            full_path.write_text(merged, encoding='utf-8')
            result.files_updated.append(rel_path)
            return

        # Default: mark as conflict
        result.conflicts.append(rel_path)

    def _merge_json(self, existing: str, new: str) -> Optional[str]:
        """Merge two JSON objects"""
        try:
            existing_obj = json.loads(existing)
            new_obj = json.loads(new)

            if isinstance(existing_obj, dict) and isinstance(new_obj, dict):
                merged = self._deep_merge_dict(existing_obj, new_obj)
                return json.dumps(merged, indent=2)

            return None
        except json.JSONDecodeError:
            return None

    def _deep_merge_dict(self, base: Dict, override: Dict) -> Dict:
        """Deep merge two dictionaries"""
        result = base.copy()

        for key, value in override.items():
            if key in result and isinstance(result[key], dict) and isinstance(value, dict):
                result[key] = self._deep_merge_dict(result[key], value)
            elif key in result and isinstance(result[key], list) and isinstance(value, list):
                # Merge lists, avoiding duplicates
                existing_set = set(str(x) for x in result[key] if not isinstance(x, dict))
                for item in value:
                    if isinstance(item, dict) or str(item) not in existing_set:
                        result[key].append(item)
            else:
                result[key] = value

        return result

    def _merge_python(self, existing: str, new: str) -> str:
        """Merge Python files, deduplicating imports"""
        import re

        # Extract imports from both
        existing_imports = set(re.findall(r'^(?:from .+ import .+|import .+)$',
                                          existing, re.MULTILINE))
        new_imports = set(re.findall(r'^(?:from .+ import .+|import .+)$',
                                     new, re.MULTILINE))

        # Remove imports from new content
        new_without_imports = re.sub(r'^(?:from .+ import .+|import .+)\n?',
                                     '', new, flags=re.MULTILINE).strip()

        # Check if the new code already exists in existing
        if new_without_imports in existing:
            return existing  # Already present

        # Add new imports to existing
        combined_imports = existing_imports | new_imports

        # Find where imports end in existing
        import_end = 0
        for match in re.finditer(r'^(?:from .+ import .+|import .+)$',
                                 existing, re.MULTILINE):
            import_end = max(import_end, match.end())

        # Build merged content
        merged = existing[:import_end]
        for imp in new_imports - existing_imports:
            merged = merged.rstrip() + f"\n{imp}"

        merged += existing[import_end:]

        # Append new code if not empty
        if new_without_imports:
            merged = merged.rstrip() + f"\n\n\n# === Added by UCTS ===\n\n{new_without_imports}\n"

        return merged

    def _merge_markdown(self, existing: str, new: str) -> str:
        """Merge markdown by appending new sections"""
        # Extract h2 headings from new content
        import re
        new_sections = re.split(r'^## ', new, flags=re.MULTILINE)

        result = existing.rstrip()

        for section in new_sections[1:]:  # Skip content before first heading
            heading = section.split('\n')[0].strip()
            if f"## {heading}" not in existing:
                result += f"\n\n## {section.rstrip()}"

        return result + "\n"

    def _merge_lines(self, existing: str, new: str) -> str:
        """Merge line-based files (requirements.txt, .gitignore)"""
        existing_lines = set(line.strip() for line in existing.split('\n') if line.strip())
        new_lines = [line.strip() for line in new.split('\n') if line.strip()]

        result = existing.rstrip()

        added = []
        for line in new_lines:
            if line not in existing_lines and not line.startswith('#'):
                added.append(line)

        if added:
            result += "\n\n# Added by UCTS\n"
            result += '\n'.join(added)
            result += "\n"

        return result

    def _merge_python_deps(self, root: Path, structure: ProjectStructure,
                           result: MergeResult) -> None:
        """Merge Python dependencies"""
        deps = structure.dependencies.get('python', [])
        if not deps:
            return

        req_file = root / 'requirements.txt'
        if req_file.exists():
            existing = req_file.read_text(encoding='utf-8')
            existing_deps = set(line.split('==')[0].split('>=')[0].strip().lower()
                               for line in existing.split('\n')
                               if line.strip() and not line.startswith('#'))

            new_deps = [d for d in deps if d.lower() not in existing_deps]
            if new_deps:
                content = existing.rstrip() + "\n\n# Added by UCTS\n"
                content += '\n'.join(new_deps) + "\n"
                req_file.write_text(content, encoding='utf-8')
                result.dependencies_added['python'] = new_deps
        else:
            req_file.write_text('\n'.join(deps) + '\n', encoding='utf-8')
            result.dependencies_added['python'] = deps

    def _merge_node_deps(self, root: Path, structure: ProjectStructure,
                         result: MergeResult) -> None:
        """Merge Node.js dependencies"""
        deps = structure.dependencies.get('node', [])
        if not deps:
            return

        pkg_file = root / 'package.json'
        if pkg_file.exists():
            try:
                pkg = json.loads(pkg_file.read_text(encoding='utf-8'))
                existing_deps = set(pkg.get('dependencies', {}).keys())
                existing_deps |= set(pkg.get('devDependencies', {}).keys())

                new_deps = [d for d in deps if d not in existing_deps]
                if new_deps:
                    if 'dependencies' not in pkg:
                        pkg['dependencies'] = {}
                    for dep in new_deps:
                        pkg['dependencies'][dep] = "*"

                    pkg_file.write_text(json.dumps(pkg, indent=2), encoding='utf-8')
                    result.dependencies_added['node'] = new_deps
            except json.JSONDecodeError:
                pass

    def _merge_gitignore(self, root: Path, structure: ProjectStructure) -> None:
        """Merge .gitignore patterns"""
        gitignore = root / '.gitignore'
        patterns = []

        if 'python' in structure.languages:
            patterns.extend(['__pycache__/', '*.pyc', '.venv/', 'dist/', 'build/'])

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            patterns.extend(['node_modules/', 'dist/', 'coverage/'])

        if not gitignore.exists():
            gitignore.write_text('\n'.join(patterns) + '\n', encoding='utf-8')
        else:
            existing = gitignore.read_text(encoding='utf-8')
            new_patterns = [p for p in patterns if p not in existing]
            if new_patterns:
                content = existing.rstrip() + "\n\n# Added by UCTS\n"
                content += '\n'.join(new_patterns) + "\n"
                gitignore.write_text(content, encoding='utf-8')
