"""
Test Generation

Auto-generates test files from analyzed project structure:
- Python: pytest test files
- JavaScript/TypeScript: Jest/Vitest tests
- API endpoint tests
"""

from pathlib import Path
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, field
import re


@dataclass
class TestConfig:
    """Configuration for test generation"""
    framework: str = "auto"  # pytest, jest, vitest, auto
    coverage_target: int = 80
    include_mocks: bool = True
    include_fixtures: bool = True
    include_integration: bool = True


class PytestGenerator:
    """Generate pytest test files"""

    def __init__(self, config: Optional[TestConfig] = None):
        self.config = config or TestConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate pytest files from project structure"""
        files = {}

        # conftest.py
        files["tests/conftest.py"] = self._generate_conftest(structure)

        # Test files for each source file
        for filepath, content in structure.files.items():
            if filepath.endswith(".py") and not filepath.startswith("test_"):
                test_file = self._generate_test_file(filepath, content, structure)
                if test_file:
                    test_path = f"tests/test_{Path(filepath).stem}.py"
                    files[test_path] = test_file

        # pytest.ini
        files["pytest.ini"] = self._generate_pytest_ini()

        # Write files
        output = Path(output_path)
        for filename, content in files.items():
            filepath = output / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)

        return files

    def _generate_conftest(self, structure) -> str:
        fixtures = []

        # Detect if FastAPI
        has_fastapi = any(
            "fastapi" in content.lower()
            for content in structure.files.values()
        )

        if has_fastapi:
            fixtures.append('''
@pytest.fixture
def client():
    """Test client for FastAPI app"""
    from fastapi.testclient import TestClient
    from main import app
    return TestClient(app)
''')

        # Detect if Flask
        has_flask = any(
            "flask" in content.lower()
            for content in structure.files.values()
        )

        if has_flask:
            fixtures.append('''
@pytest.fixture
def client():
    """Test client for Flask app"""
    from main import app
    app.config['TESTING'] = True
    with app.test_client() as client:
        yield client
''')

        # Detect database usage
        has_db = any(
            "sqlalchemy" in content.lower() or "database" in content.lower()
            for content in structure.files.values()
        )

        if has_db:
            fixtures.append('''
@pytest.fixture
def db_session():
    """Database session for testing"""
    # Setup
    from database import SessionLocal, engine, Base
    Base.metadata.create_all(bind=engine)
    session = SessionLocal()
    yield session
    # Teardown
    session.rollback()
    session.close()
''')

        return f'''"""
Test fixtures and configuration
Generated by UCTS
"""
import pytest
from unittest.mock import Mock, patch, MagicMock
from typing import Generator
{chr(10).join(fixtures)}

@pytest.fixture
def mock_env(monkeypatch):
    """Mock environment variables"""
    monkeypatch.setenv("ENVIRONMENT", "test")
    monkeypatch.setenv("DEBUG", "true")


@pytest.fixture
def sample_data():
    """Sample test data"""
    return {{
        "id": 1,
        "name": "Test Item",
        "value": 100
    }}
'''

    def _generate_test_file(self, filepath: str, content: str, structure) -> Optional[str]:
        """Generate test file for a source file"""
        module_name = Path(filepath).stem

        # Extract functions and classes
        functions = re.findall(r'def\s+(\w+)\s*\(', content)
        classes = re.findall(r'class\s+(\w+)\s*[\(:]', content)

        if not functions and not classes:
            return None

        tests = []

        # Generate class tests
        for cls in classes:
            if cls.startswith('_'):
                continue
            tests.append(f'''
class Test{cls}:
    """Tests for {cls} class"""

    def test_{cls.lower()}_initialization(self):
        """Test {cls} can be initialized"""
        # Arrange
        # Act
        instance = {cls}()
        # Assert
        assert instance is not None

    def test_{cls.lower()}_attributes(self):
        """Test {cls} has expected attributes"""
        instance = {cls}()
        # Add attribute assertions
        pass
''')

        # Generate function tests
        for func in functions:
            if func.startswith('_') or func in ['__init__', '__str__', '__repr__']:
                continue
            tests.append(f'''
class Test{func.title().replace("_", "")}:
    """Tests for {func} function"""

    def test_{func}_basic(self):
        """Test {func} with basic input"""
        # Arrange
        # Act
        result = {func}()
        # Assert
        assert result is not None

    def test_{func}_edge_cases(self):
        """Test {func} with edge cases"""
        # Test with None, empty values, etc.
        pass

    def test_{func}_error_handling(self):
        """Test {func} handles errors correctly"""
        # Test error conditions
        pass
''')

        if not tests:
            return None

        imports = f'from {module_name} import *'

        return f'''"""
Tests for {module_name}
Generated by UCTS
"""
import pytest
from unittest.mock import Mock, patch
{imports}
{chr(10).join(tests)}
'''

    def _generate_pytest_ini(self) -> str:
        return f'''[pytest]
testpaths = tests
python_files = test_*.py
python_classes = Test*
python_functions = test_*
addopts = -v --tb=short --cov=. --cov-report=html --cov-report=term-missing --cov-fail-under={self.config.coverage_target}
filterwarnings =
    ignore::DeprecationWarning
    ignore::PendingDeprecationWarning
markers =
    slow: marks tests as slow (deselect with '-m "not slow"')
    integration: marks tests as integration tests
    unit: marks tests as unit tests
'''


class JestGenerator:
    """Generate Jest test files"""

    def __init__(self, config: Optional[TestConfig] = None):
        self.config = config or TestConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Jest files from project structure"""
        files = {}

        # Detect TypeScript
        is_typescript = any(
            f.endswith('.ts') or f.endswith('.tsx')
            for f in structure.files.keys()
        )

        # jest.config.js
        files["jest.config.js" if not is_typescript else "jest.config.ts"] = self._generate_jest_config(is_typescript)

        # Test files
        for filepath, content in structure.files.items():
            if (filepath.endswith('.js') or filepath.endswith('.ts') or
                filepath.endswith('.jsx') or filepath.endswith('.tsx')):
                if '.test.' not in filepath and '.spec.' not in filepath:
                    test_file = self._generate_test_file(filepath, content, is_typescript)
                    if test_file:
                        ext = '.test.ts' if is_typescript else '.test.js'
                        base = Path(filepath).stem
                        test_path = f"__tests__/{base}{ext}"
                        files[test_path] = test_file

        # Setup file
        files["jest.setup.js"] = self._generate_setup()

        # Write files
        output = Path(output_path)
        for filename, content in files.items():
            filepath = output / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)

        return files

    def _generate_jest_config(self, is_typescript: bool) -> str:
        if is_typescript:
            return '''import type { Config } from 'jest';

const config: Config = {
  preset: 'ts-jest',
  testEnvironment: 'node',
  roots: ['<rootDir>/__tests__'],
  testMatch: ['**/*.test.ts', '**/*.spec.ts'],
  transform: {
    '^.+\\.tsx?$': 'ts-jest',
  },
  collectCoverageFrom: [
    'src/**/*.{ts,tsx}',
    '!src/**/*.d.ts',
  ],
  coverageThreshold: {
    global: {
      branches: 80,
      functions: 80,
      lines: 80,
      statements: 80,
    },
  },
  setupFilesAfterEnv: ['<rootDir>/jest.setup.js'],
  moduleNameMapper: {
    '^@/(.*)$': '<rootDir>/src/$1',
  },
};

export default config;
'''
        else:
            return '''module.exports = {
  testEnvironment: 'node',
  roots: ['<rootDir>/__tests__'],
  testMatch: ['**/*.test.js', '**/*.spec.js'],
  collectCoverageFrom: [
    'src/**/*.js',
    '!src/**/*.test.js',
  ],
  coverageThreshold: {
    global: {
      branches: 80,
      functions: 80,
      lines: 80,
      statements: 80,
    },
  },
  setupFilesAfterEnv: ['<rootDir>/jest.setup.js'],
};
'''

    def _generate_test_file(self, filepath: str, content: str, is_typescript: bool) -> Optional[str]:
        """Generate test file for a source file"""
        module_name = Path(filepath).stem

        # Extract exports
        exports = re.findall(r'export\s+(?:const|function|class)\s+(\w+)', content)
        default_exports = re.findall(r'export\s+default\s+(?:function\s+)?(\w+)?', content)

        if not exports and not default_exports:
            # Try CommonJS exports
            exports = re.findall(r'module\.exports\s*=\s*\{([^}]+)\}', content)
            if exports:
                exports = [e.strip() for e in exports[0].split(',')]

        if not exports and not default_exports:
            return None

        tests = []

        for exp in exports:
            if exp.startswith('_'):
                continue
            tests.append(f'''
describe('{exp}', () => {{
  it('should be defined', () => {{
    expect({exp}).toBeDefined();
  }});

  it('should work with basic input', () => {{
    // Arrange
    const input = {{}};

    // Act
    const result = {exp}(input);

    // Assert
    expect(result).toBeDefined();
  }});

  it('should handle edge cases', () => {{
    // Test edge cases
  }});

  it('should throw on invalid input', () => {{
    expect(() => {exp}(null)).toThrow();
  }});
}});
''')

        if not tests:
            return None

        type_annotation = ': ' if is_typescript else ''
        import_ext = filepath.replace('\\', '/')

        return f'''/**
 * Tests for {module_name}
 * Generated by UCTS
 */
import {{ {', '.join(exports)} }} from '../{import_ext}';

{'// Mock dependencies' if self.config.include_mocks else ''}
jest.mock('../config', () => ({{
  apiUrl: 'http://test.local',
}}));
{chr(10).join(tests)}
'''

    def _generate_setup(self) -> str:
        return '''/**
 * Jest setup file
 * Generated by UCTS
 */

// Extend Jest matchers
// import '@testing-library/jest-dom';

// Global mocks
global.fetch = jest.fn(() =>
  Promise.resolve({
    json: () => Promise.resolve({}),
    ok: true,
    status: 200,
  })
);

// Console error/warn suppression in tests
const originalError = console.error;
const originalWarn = console.warn;

beforeAll(() => {
  console.error = (...args) => {
    if (args[0]?.includes?.('Warning:')) return;
    originalError.call(console, ...args);
  };
  console.warn = (...args) => {
    if (args[0]?.includes?.('Warning:')) return;
    originalWarn.call(console, ...args);
  };
});

afterAll(() => {
  console.error = originalError;
  console.warn = originalWarn;
});

// Clean up after each test
afterEach(() => {
  jest.clearAllMocks();
});
'''


class VitestGenerator:
    """Generate Vitest test files"""

    def __init__(self, config: Optional[TestConfig] = None):
        self.config = config or TestConfig()

    def generate(self, structure, output_path: str) -> Dict[str, str]:
        """Generate Vitest files from project structure"""
        files = {}

        # vitest.config.ts
        files["vitest.config.ts"] = self._generate_vitest_config()

        # Test files
        for filepath, content in structure.files.items():
            if (filepath.endswith('.ts') or filepath.endswith('.tsx') or
                filepath.endswith('.js') or filepath.endswith('.jsx')):
                if '.test.' not in filepath and '.spec.' not in filepath:
                    test_file = self._generate_test_file(filepath, content)
                    if test_file:
                        base = Path(filepath).stem
                        test_path = f"tests/{base}.test.ts"
                        files[test_path] = test_file

        # Setup file
        files["tests/setup.ts"] = self._generate_setup()

        # Write files
        output = Path(output_path)
        for filename, content in files.items():
            filepath = output / filename
            filepath.parent.mkdir(parents=True, exist_ok=True)
            filepath.write_text(content)

        return files

    def _generate_vitest_config(self) -> str:
        return '''import { defineConfig } from 'vitest/config';

export default defineConfig({
  test: {
    globals: true,
    environment: 'node',
    include: ['tests/**/*.test.ts'],
    coverage: {
      provider: 'v8',
      reporter: ['text', 'html', 'lcov'],
      exclude: [
        'node_modules/',
        'tests/',
        '**/*.d.ts',
      ],
      thresholds: {
        lines: 80,
        functions: 80,
        branches: 80,
        statements: 80,
      },
    },
    setupFiles: ['./tests/setup.ts'],
  },
  resolve: {
    alias: {
      '@': './src',
    },
  },
});
'''

    def _generate_test_file(self, filepath: str, content: str) -> Optional[str]:
        """Generate test file for a source file"""
        module_name = Path(filepath).stem

        # Extract exports
        exports = re.findall(r'export\s+(?:const|function|class)\s+(\w+)', content)

        if not exports:
            return None

        tests = []

        for exp in exports:
            if exp.startswith('_'):
                continue
            tests.append(f'''
describe('{exp}', () => {{
  it('should be defined', () => {{
    expect({exp}).toBeDefined();
  }});

  it('should work with basic input', () => {{
    // Arrange
    const input = {{}};

    // Act
    const result = {exp}(input);

    // Assert
    expect(result).toBeDefined();
  }});
}});
''')

        if not tests:
            return None

        import_path = filepath.replace('\\', '/')

        return f'''/**
 * Tests for {module_name}
 * Generated by UCTS
 */
import {{ describe, it, expect, vi, beforeEach, afterEach }} from 'vitest';
import {{ {', '.join(exports)} }} from '../{import_path}';
{chr(10).join(tests)}
'''

    def _generate_setup(self) -> str:
        return '''/**
 * Vitest setup file
 * Generated by UCTS
 */
import { vi } from 'vitest';

// Global mocks
global.fetch = vi.fn(() =>
  Promise.resolve({
    json: () => Promise.resolve({}),
    ok: true,
    status: 200,
  } as Response)
);

// Clean up after each test
afterEach(() => {
  vi.clearAllMocks();
});
'''


class TestGenerator:
    """
    Unified test generator.

    Auto-detects the appropriate test framework and generates tests.
    """

    def __init__(self, config: Optional[TestConfig] = None):
        self.config = config or TestConfig()

    def generate(self, structure, output_path: str, framework: Optional[str] = None) -> Dict[str, str]:
        """
        Generate tests for the project.

        Args:
            structure: Project structure from analysis
            output_path: Output directory
            framework: Test framework (pytest, jest, vitest, auto)

        Returns:
            Dictionary of generated files
        """
        # Detect framework if auto
        if framework is None or framework == "auto":
            framework = self._detect_framework(structure)

        generators = {
            "pytest": PytestGenerator,
            "jest": JestGenerator,
            "vitest": VitestGenerator,
        }

        if framework not in generators:
            raise ValueError(f"Unknown framework: {framework}")

        generator = generators[framework](self.config)
        return generator.generate(structure, output_path)

    def _detect_framework(self, structure) -> str:
        """Detect appropriate test framework from project"""
        languages = [l.lower() for l in structure.languages]

        if "python" in languages:
            return "pytest"

        # Check for existing config files
        files = structure.files.keys()

        if any("vitest.config" in f for f in files):
            return "vitest"
        if any("jest.config" in f for f in files):
            return "jest"

        # Default based on language
        if any(l in ["typescript", "javascript"] for l in languages):
            # Prefer vitest for modern projects
            return "vitest"

        return "pytest"

    @staticmethod
    def list_frameworks() -> List[Dict[str, str]]:
        """List available test frameworks"""
        return [
            {"name": "pytest", "language": "Python", "description": "Python testing framework"},
            {"name": "jest", "language": "JavaScript/TypeScript", "description": "JavaScript testing framework"},
            {"name": "vitest", "language": "TypeScript", "description": "Fast Vite-native testing framework"},
        ]
