"""
VS Code Workspace Generator

Generates a complete VS Code workspace from a ProjectStructure.
"""
import json
import sys
from pathlib import Path
from typing import Dict, List, Any

from ucts.core.models import ProjectStructure


class VSCodeGenerator:
    """Generate VS Code workspace from project structure"""

    def generate(self, structure: ProjectStructure, output_path: str) -> Path:
        """
        Generate VS Code workspace at the specified path.

        Args:
            structure: ProjectStructure to generate from
            output_path: Directory path for the workspace

        Returns:
            Path to the generated workspace
        """
        root = Path(output_path)
        root.mkdir(parents=True, exist_ok=True)

        # Create directory structure
        for dir_path in structure.directories:
            (root / dir_path).mkdir(parents=True, exist_ok=True)

        # Write files
        for file_path, content in structure.files.items():
            full_path = root / file_path
            full_path.parent.mkdir(parents=True, exist_ok=True)
            full_path.write_text(content, encoding='utf-8')

        # Create .vscode directory
        vscode_dir = root / '.vscode'
        vscode_dir.mkdir(exist_ok=True)

        # settings.json
        settings = self._generate_settings(structure)
        (vscode_dir / 'settings.json').write_text(
            json.dumps(settings, indent=2),
            encoding='utf-8'
        )

        # extensions.json
        extensions = self._recommend_extensions(structure)
        (vscode_dir / 'extensions.json').write_text(
            json.dumps(extensions, indent=2),
            encoding='utf-8'
        )

        # tasks.json
        tasks = self._generate_tasks(structure)
        (vscode_dir / 'tasks.json').write_text(
            json.dumps(tasks, indent=2),
            encoding='utf-8'
        )

        # launch.json (debug configurations)
        launch = self._generate_launch_config(structure)
        (vscode_dir / 'launch.json').write_text(
            json.dumps(launch, indent=2),
            encoding='utf-8'
        )

        # README.md
        (root / 'README.md').write_text(
            structure.readme_content,
            encoding='utf-8'
        )

        # Language-specific setup
        if 'python' in structure.languages:
            self._setup_python(root, structure)
        if any(lang in structure.languages for lang in ('javascript', 'typescript', 'jsx', 'tsx')):
            self._setup_node(root, structure)

        # .gitignore
        gitignore = self._generate_gitignore(structure)
        (root / '.gitignore').write_text(gitignore, encoding='utf-8')

        # Workspace file
        workspace = {
            "folders": [{"path": "."}],
            "settings": settings
        }
        workspace_file = root / f'{structure.name}.code-workspace'
        workspace_file.write_text(
            json.dumps(workspace, indent=2),
            encoding='utf-8'
        )

        return root

    def _generate_settings(self, structure: ProjectStructure) -> Dict[str, Any]:
        """Generate VS Code settings.json"""
        settings: Dict[str, Any] = {
            "editor.formatOnSave": True,
            "files.trimTrailingWhitespace": True,
            "files.insertFinalNewline": True,
        }

        if 'python' in structure.languages:
            # Use platform-appropriate venv path
            if sys.platform == 'win32':
                interpreter_path = ".venv/Scripts/python.exe"
            else:
                interpreter_path = ".venv/bin/python"

            settings.update({
                "python.defaultInterpreterPath": interpreter_path,
                "python.formatting.provider": "black",
                "python.linting.enabled": True,
                "python.linting.pylintEnabled": False,
                "python.linting.flake8Enabled": True,
                "[python]": {
                    "editor.defaultFormatter": "ms-python.black-formatter",
                    "editor.formatOnSave": True
                }
            })

        if any(lang in structure.languages for lang in ('typescript', 'tsx')):
            settings.update({
                "typescript.preferences.quoteStyle": "single",
                "typescript.preferences.importModuleSpecifier": "relative",
                "[typescript]": {
                    "editor.defaultFormatter": "esbenp.prettier-vscode"
                },
                "[typescriptreact]": {
                    "editor.defaultFormatter": "esbenp.prettier-vscode"
                }
            })

        if any(lang in structure.languages for lang in ('javascript', 'jsx')):
            settings.update({
                "javascript.preferences.quoteStyle": "single",
                "[javascript]": {
                    "editor.defaultFormatter": "esbenp.prettier-vscode"
                },
                "[javascriptreact]": {
                    "editor.defaultFormatter": "esbenp.prettier-vscode"
                }
            })

        return settings

    def _recommend_extensions(self, structure: ProjectStructure) -> Dict[str, List[str]]:
        """Generate recommended extensions based on languages"""
        recommendations = []

        if 'python' in structure.languages:
            recommendations.extend([
                "ms-python.python",
                "ms-python.vscode-pylance",
                "ms-python.black-formatter",
            ])

        if any(lang in structure.languages for lang in ('javascript', 'typescript', 'jsx', 'tsx')):
            recommendations.extend([
                "esbenp.prettier-vscode",
                "dbaeumer.vscode-eslint",
            ])

        if 'rust' in structure.languages:
            recommendations.extend([
                "rust-lang.rust-analyzer",
            ])

        if 'go' in structure.languages:
            recommendations.extend([
                "golang.go",
            ])

        # Always useful
        recommendations.extend([
            "eamodio.gitlens",
            "usernamehw.errorlens",
        ])

        return {"recommendations": recommendations}

    def _generate_tasks(self, structure: ProjectStructure) -> Dict[str, Any]:
        """Generate tasks.json for common operations"""
        tasks: List[Dict[str, Any]] = []

        if 'python' in structure.languages:
            tasks.extend([
                {
                    "label": "Python: Run main",
                    "type": "shell",
                    "command": "python",
                    "args": ["src/main.py"],
                    "group": "build"
                },
                {
                    "label": "Python: Run tests",
                    "type": "shell",
                    "command": "pytest",
                    "args": ["tests/", "-v"],
                    "group": "test"
                },
                {
                    "label": "Python: Install dependencies",
                    "type": "shell",
                    "command": "pip",
                    "args": ["install", "-r", "requirements.txt"],
                    "group": "build"
                }
            ])

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            tasks.extend([
                {
                    "label": "npm: install",
                    "type": "shell",
                    "command": "npm",
                    "args": ["install"],
                    "group": "build"
                },
                {
                    "label": "npm: run build",
                    "type": "shell",
                    "command": "npm",
                    "args": ["run", "build"],
                    "group": "build"
                },
                {
                    "label": "npm: test",
                    "type": "shell",
                    "command": "npm",
                    "args": ["test"],
                    "group": "test"
                }
            ])

        return {
            "version": "2.0.0",
            "tasks": tasks
        }

    def _generate_launch_config(self, structure: ProjectStructure) -> Dict[str, Any]:
        """Generate debug launch configurations"""
        configurations: List[Dict[str, Any]] = []

        if 'python' in structure.languages:
            configurations.append({
                "name": "Python: Current File",
                "type": "debugpy",
                "request": "launch",
                "program": "${file}",
                "console": "integratedTerminal"
            })

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            configurations.append({
                "name": "Node: Run Current File",
                "type": "node",
                "request": "launch",
                "program": "${file}",
                "console": "integratedTerminal"
            })

        return {
            "version": "0.2.0",
            "configurations": configurations
        }

    def _setup_python(self, root: Path, structure: ProjectStructure):
        """Set up Python-specific files"""
        # requirements.txt
        deps = structure.dependencies.get('python', [])
        requirements = '\n'.join(deps) + '\n' if deps else '# Add dependencies here\n'
        (root / 'requirements.txt').write_text(requirements, encoding='utf-8')

        # pyproject.toml (basic)
        pyproject = f'''[build-system]
requires = ["setuptools>=61.0"]
build-backend = "setuptools.build_meta"

[project]
name = "{structure.name}"
version = "0.1.0"
description = "{structure.description[:100] if structure.description else 'A Python project'}"
requires-python = ">=3.9"
dependencies = {json.dumps(deps)}

[tool.pytest.ini_options]
testpaths = ["tests"]
'''
        (root / 'pyproject.toml').write_text(pyproject, encoding='utf-8')

        # src/__init__.py
        src_dir = root / 'src'
        src_dir.mkdir(exist_ok=True)
        (src_dir / '__init__.py').write_text('', encoding='utf-8')

    def _setup_node(self, root: Path, structure: ProjectStructure):
        """Set up Node.js-specific files"""
        deps = structure.dependencies.get('node', [])

        package_json = {
            "name": structure.name,
            "version": "0.1.0",
            "description": structure.description[:100] if structure.description else '',
            "main": "src/index.js",
            "scripts": {
                "start": "node src/index.js",
                "test": "jest",
                "build": "tsc" if 'typescript' in structure.languages else "echo 'No build step'"
            },
            "dependencies": {dep: "*" for dep in deps},
            "devDependencies": {}
        }

        if 'typescript' in structure.languages:
            package_json["devDependencies"]["typescript"] = "^5.0.0"
            package_json["devDependencies"]["@types/node"] = "^20.0.0"

        (root / 'package.json').write_text(
            json.dumps(package_json, indent=2),
            encoding='utf-8'
        )

        # tsconfig.json if TypeScript
        if 'typescript' in structure.languages:
            tsconfig = {
                "compilerOptions": {
                    "target": "ES2020",
                    "module": "commonjs",
                    "strict": True,
                    "esModuleInterop": True,
                    "skipLibCheck": True,
                    "forceConsistentCasingInFileNames": True,
                    "outDir": "./dist",
                    "rootDir": "./src"
                },
                "include": ["src/**/*"],
                "exclude": ["node_modules"]
            }
            (root / 'tsconfig.json').write_text(
                json.dumps(tsconfig, indent=2),
                encoding='utf-8'
            )

    def _generate_gitignore(self, structure: ProjectStructure) -> str:
        """Generate .gitignore content"""
        lines = [
            "# IDE",
            ".idea/",
            "*.swp",
            "*.swo",
            ".DS_Store",
            "",
            "# Dependencies",
        ]

        if 'python' in structure.languages:
            lines.extend([
                "__pycache__/",
                "*.py[cod]",
                ".venv/",
                "venv/",
                "*.egg-info/",
                "dist/",
                "build/",
                ".pytest_cache/",
                ".coverage",
                "htmlcov/",
            ])

        if any(lang in structure.languages for lang in ('javascript', 'typescript')):
            lines.extend([
                "node_modules/",
                "dist/",
                "coverage/",
                "*.log",
            ])

        lines.extend([
            "",
            "# Environment",
            ".env",
            ".env.local",
            "*.local",
        ])

        return '\n'.join(lines) + '\n'
