"""UCTS Ingestion Module - Session capture from multiple sources"""

from ucts.ingestion.base import SessionIngester
from ucts.ingestion.claude_desktop import ClaudeDesktopIngester
from ucts.ingestion.vscode import VSCodeIngester
from ucts.ingestion.web_chat import WebChatIngester
from ucts.ingestion.transcript import TranscriptIngester

__all__ = [
    "SessionIngester",
    "ClaudeDesktopIngester",
    "VSCodeIngester",
    "WebChatIngester",
    "TranscriptIngester",
    "get_ingester",
]


def get_ingester(format_type: str, source_path: str = None) -> SessionIngester:
    """Get appropriate ingester for format type"""
    if format_type == "auto" and source_path:
        # Auto-detect based on file extension and content
        if source_path.endswith('.json'):
            # Try to determine JSON type
            try:
                import json
                with open(source_path, 'r') as f:
                    data = json.load(f)
                if 'conversation' in data or 'claude' in str(data).lower():
                    return ClaudeDesktopIngester()
                elif 'vscode' in str(data).lower() or 'copilot' in str(data).lower():
                    return VSCodeIngester()
                else:
                    return WebChatIngester()
            except Exception:
                return TranscriptIngester()
        elif source_path.endswith('.md') or source_path.endswith('.txt'):
            return TranscriptIngester()
        else:
            return TranscriptIngester()

    ingesters = {
        "claude": ClaudeDesktopIngester,
        "claude_desktop": ClaudeDesktopIngester,
        "vscode": VSCodeIngester,
        "web": WebChatIngester,
        "transcript": TranscriptIngester,
    }

    ingester_class = ingesters.get(format_type, TranscriptIngester)
    return ingester_class()
