"""
Coherence Integration

Integration with the Coherence cross-project orchestrator for
workflow handoffs and state synchronization.
"""
import json
import logging
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime

# Configure logging
logger = logging.getLogger(__name__)


class CoherenceError(Exception):
    """Base exception for Coherence integration errors"""
    pass


class CoherenceConnectionError(CoherenceError):
    """Error connecting to or finding Coherence state"""
    pass


class CoherenceStateError(CoherenceError):
    """Error reading or writing Coherence state"""
    pass


class CoherenceIntegration:
    """Integration with Coherence orchestrator"""

    def __init__(self, coherence_path: Optional[str] = None):
        """
        Initialize Coherence integration.

        Args:
            coherence_path: Path to Coherence state file

        Raises:
            CoherenceConnectionError: If coherence_path is invalid
        """
        if coherence_path:
            self.coherence_path = Path(coherence_path)
            if not self.coherence_path.parent.exists():
                logger.warning(f"Coherence parent directory does not exist: {self.coherence_path.parent}")
        else:
            # Try common locations
            possible_paths = [
                Path.cwd() / "execution" / "coherence_state.json",
                Path.cwd() / "coherence_state.json",
                Path.home() / ".coherence" / "state.json",
            ]
            self.coherence_path = next(
                (p for p in possible_paths if p.exists()),
                possible_paths[0]
            )
            logger.debug(f"Using Coherence path: {self.coherence_path}")

    def export_state(self, project_id: str, status: str,
                    completed_work: List[str], available_actions: List[str]) -> Dict[str, Any]:
        """
        Export UCTS state to Coherence.

        Args:
            project_id: Unique project identifier
            status: Current status (operational, pending, error)
            completed_work: List of completed tasks
            available_actions: List of available actions

        Returns:
            Exported state dictionary

        Raises:
            ValueError: If project_id is empty
            CoherenceStateError: If state cannot be written
        """
        # Validate inputs
        if not project_id or not isinstance(project_id, str):
            raise ValueError("project_id must be a non-empty string")

        if status not in ("operational", "pending", "error", "unknown"):
            logger.warning(f"Unexpected status value: {status}")

        if not isinstance(completed_work, list):
            completed_work = list(completed_work) if completed_work else []

        if not isinstance(available_actions, list):
            available_actions = list(available_actions) if available_actions else []

        state = {
            "project_id": project_id,
            "system": "ucts",
            "version": "2.0.0",
            "exported_at": datetime.now().isoformat(),
            "status": status,
            "completed_work": completed_work,
            "available_actions": available_actions,
            "capabilities": [
                "session_ingestion",
                "project_generation",
                "context_transfer",
                "vscode_workspace",
                "gitlab_repo",
                "github_repo"
            ]
        }

        # Save to Coherence path
        try:
            self.coherence_path.parent.mkdir(parents=True, exist_ok=True)
            with open(self.coherence_path, 'w', encoding='utf-8') as f:
                json.dump(state, f, indent=2)
            logger.info(f"Exported state for project {project_id} to {self.coherence_path}")
        except PermissionError as e:
            raise CoherenceStateError(f"Permission denied writing to {self.coherence_path}: {e}")
        except OSError as e:
            raise CoherenceStateError(f"Failed to write Coherence state: {e}")

        return state

    def import_state(self) -> Optional[Dict[str, Any]]:
        """
        Import state from Coherence.

        Returns:
            Coherence state dictionary or None if not available

        Raises:
            CoherenceStateError: If state file exists but cannot be read
        """
        if not self.coherence_path.exists():
            logger.debug(f"Coherence state file not found: {self.coherence_path}")
            return None

        try:
            with open(self.coherence_path, 'r', encoding='utf-8') as f:
                state = json.load(f)
            logger.debug(f"Imported Coherence state from {self.coherence_path}")
            return state
        except json.JSONDecodeError as e:
            raise CoherenceStateError(f"Invalid JSON in Coherence state file: {e}")
        except PermissionError as e:
            raise CoherenceStateError(f"Permission denied reading {self.coherence_path}: {e}")
        except OSError as e:
            raise CoherenceStateError(f"Failed to read Coherence state: {e}")

    def register_workflow(self, workflow_id: str, workflow_type: str,
                         source_project: str, target_project: str) -> Dict[str, Any]:
        """
        Register a workflow handoff with Coherence.

        Args:
            workflow_id: Unique workflow identifier
            workflow_type: Type of workflow (forge, transfer, sync)
            source_project: Source project ID
            target_project: Target project ID

        Returns:
            Workflow registration

        Raises:
            ValueError: If required parameters are empty
            CoherenceStateError: If workflow cannot be registered
        """
        # Validate inputs
        if not workflow_id:
            raise ValueError("workflow_id must be non-empty")
        if not workflow_type:
            raise ValueError("workflow_type must be non-empty")
        if not source_project:
            raise ValueError("source_project must be non-empty")
        if not target_project:
            raise ValueError("target_project must be non-empty")

        if workflow_type not in ("forge", "transfer", "sync", "custom"):
            logger.warning(f"Non-standard workflow type: {workflow_type}")

        workflow = {
            "workflow_id": workflow_id,
            "type": workflow_type,
            "source": source_project,
            "target": target_project,
            "status": "registered",
            "registered_at": datetime.now().isoformat(),
        }

        # Append to workflows log
        workflows_path = self.coherence_path.parent / "workflows.json"
        workflows: List[Dict[str, Any]] = []

        try:
            if workflows_path.exists():
                with open(workflows_path, 'r', encoding='utf-8') as f:
                    workflows = json.load(f)
                    if not isinstance(workflows, list):
                        logger.warning("Workflows file is not a list, resetting")
                        workflows = []
        except json.JSONDecodeError as e:
            logger.warning(f"Invalid JSON in workflows file, resetting: {e}")
            workflows = []
        except OSError as e:
            logger.warning(f"Could not read workflows file: {e}")

        workflows.append(workflow)

        try:
            workflows_path.parent.mkdir(parents=True, exist_ok=True)
            with open(workflows_path, 'w', encoding='utf-8') as f:
                json.dump(workflows, f, indent=2)
            logger.info(f"Registered workflow {workflow_id} ({workflow_type})")
        except PermissionError as e:
            raise CoherenceStateError(f"Permission denied writing to {workflows_path}: {e}")
        except OSError as e:
            raise CoherenceStateError(f"Failed to write workflows: {e}")

        return workflow

    def notify_completion(self, workflow_id: str, result: Dict[str, Any]) -> bool:
        """
        Notify Coherence of workflow completion.

        Args:
            workflow_id: Workflow identifier
            result: Workflow result data

        Returns:
            True if workflow was found and updated, False otherwise

        Raises:
            ValueError: If workflow_id is empty
            CoherenceStateError: If workflows cannot be updated
        """
        if not workflow_id:
            raise ValueError("workflow_id must be non-empty")

        if not isinstance(result, dict):
            result = {"value": result}

        workflows_path = self.coherence_path.parent / "workflows.json"

        if not workflows_path.exists():
            logger.warning(f"Workflows file not found: {workflows_path}")
            return False

        try:
            with open(workflows_path, 'r', encoding='utf-8') as f:
                workflows = json.load(f)
        except json.JSONDecodeError as e:
            raise CoherenceStateError(f"Invalid JSON in workflows file: {e}")
        except OSError as e:
            raise CoherenceStateError(f"Failed to read workflows: {e}")

        found = False
        for workflow in workflows:
            if workflow.get("workflow_id") == workflow_id:
                workflow["status"] = "completed"
                workflow["completed_at"] = datetime.now().isoformat()
                workflow["result"] = result
                found = True
                break

        if not found:
            logger.warning(f"Workflow not found: {workflow_id}")
            return False

        try:
            with open(workflows_path, 'w', encoding='utf-8') as f:
                json.dump(workflows, f, indent=2)
            logger.info(f"Workflow {workflow_id} marked as completed")
        except OSError as e:
            raise CoherenceStateError(f"Failed to update workflows: {e}")

        return True

    def get_pending_workflows(self) -> List[Dict[str, Any]]:
        """
        Get all pending (non-completed) workflows.

        Returns:
            List of pending workflow dictionaries
        """
        workflows_path = self.coherence_path.parent / "workflows.json"

        if not workflows_path.exists():
            return []

        try:
            with open(workflows_path, 'r', encoding='utf-8') as f:
                workflows = json.load(f)
            return [w for w in workflows if w.get("status") != "completed"]
        except (json.JSONDecodeError, OSError) as e:
            logger.warning(f"Could not read workflows: {e}")
            return []
