"""
DanAI Agent Marketplace Integration

Provides:
- UCTS registration as code generation agent
- Template discovery as agent capabilities
- Task routing integration
- LOCAL AI FIRST agent selection
"""

import json
import logging
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime
from dataclasses import dataclass, field
from enum import Enum

logger = logging.getLogger(__name__)


class CostTier(Enum):
    """Cost tiers for LOCAL AI FIRST routing"""
    FREE = "free"           # Ollama, local models
    ECONOMY = "economy"     # DeepSeek
    LIGHTWEIGHT = "lightweight"  # Haiku
    STANDARD = "standard"   # Claude
    PREMIUM = "premium"     # Opus


class AgentStatus(Enum):
    """Agent availability status"""
    AVAILABLE = "available"
    BUSY = "busy"
    OFFLINE = "offline"
    MAINTENANCE = "maintenance"


@dataclass
class UCTSAgent:
    """UCTS as an agent in the marketplace"""
    agent_id: str
    name: str
    capabilities: List[str]
    cost_tier: CostTier
    status: AgentStatus = AgentStatus.AVAILABLE
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class TaskRequest:
    """Incoming task request from ecosystem"""
    task_id: str
    task_type: str
    payload: Dict[str, Any]
    requester: str
    priority: int = 2


class DanAIIntegration:
    """
    Integration with DanAI agent marketplace.

    Registers UCTS templates and generators as discoverable agents,
    enables task routing from other ecosystem projects.
    """

    def __init__(self, config=None):
        from ucts.integrations.ecosystem import EcosystemConfig
        self.config = config or EcosystemConfig()
        self._registered_agents: Dict[str, UCTSAgent] = {}
        self._pending_tasks: List[TaskRequest] = []
        self._status = AgentStatus.AVAILABLE

    @property
    def base_path(self) -> Path:
        return Path(self.config.base_path)

    @property
    def danai_path(self) -> Path:
        return self.base_path / "danai"

    def health_check(self) -> Dict[str, Any]:
        """Perform health check"""
        return {
            "status": "healthy" if self._status == AgentStatus.AVAILABLE else "degraded",
            "project": "danai",
            "registered_agents": len(self._registered_agents),
            "pending_tasks": len(self._pending_tasks),
            "timestamp": datetime.now().isoformat()
        }

    def register_ucts_agents(self) -> Dict[str, Any]:
        """
        Register UCTS and its templates as DanAI agents.

        Returns:
            Registration result with agent IDs
        """
        # Main UCTS agent
        main_agent = UCTSAgent(
            agent_id="ucts-forge",
            name="UCTS Project Forge",
            capabilities=[
                "code-generation",
                "project-scaffolding",
                "conversation-to-code",
                "multi-language",
            ],
            cost_tier=CostTier.FREE,  # UCTS runs locally
            metadata={
                "version": "2.0.0",
                "source_formats": ["claude_desktop", "vscode", "web_chat"],
                "output_targets": ["vscode", "gitlab", "github", "docker"]
            }
        )
        self._registered_agents["ucts-forge"] = main_agent

        # Register each template as an agent
        try:
            from ucts.templates import get_registry
            registry = get_registry()
            templates = registry.list_all()

            for template in templates:
                agent = UCTSAgent(
                    agent_id=f"ucts-template-{template['name']}",
                    name=f"UCTS {template['name'].title()} Generator",
                    capabilities=[
                        "code-generation",
                        f"language-{template.get('language', 'unknown')}",
                        f"framework-{template['name']}",
                        "project-scaffolding"
                    ],
                    cost_tier=CostTier.FREE,
                    metadata={
                        "template": template["name"],
                        "description": template.get("description", ""),
                        "language": template.get("language", "unknown")
                    }
                )
                self._registered_agents[agent.agent_id] = agent
        except ImportError:
            logger.warning("Template registry not available")

        # Docker generator agent
        docker_agent = UCTSAgent(
            agent_id="ucts-docker",
            name="UCTS Docker Generator",
            capabilities=[
                "docker",
                "containerization",
                "docker-compose",
                "devops"
            ],
            cost_tier=CostTier.FREE,
            metadata={"generates": ["Dockerfile", "docker-compose.yml"]}
        )
        self._registered_agents["ucts-docker"] = docker_agent

        # Write registration to file
        self._save_registrations()

        logger.info(f"Registered {len(self._registered_agents)} UCTS agents with DanAI")

        return {
            "status": "registered",
            "agent_count": len(self._registered_agents),
            "agents": list(self._registered_agents.keys())
        }

    def _save_registrations(self):
        """Save agent registrations to file"""
        reg_path = self.danai_path / "ucts_agents.json"
        reg_path.parent.mkdir(parents=True, exist_ok=True)

        registrations = {
            agent_id: {
                "agent_id": agent.agent_id,
                "name": agent.name,
                "capabilities": agent.capabilities,
                "cost_tier": agent.cost_tier.value,
                "status": agent.status.value,
                "metadata": agent.metadata,
                "registered_at": datetime.now().isoformat()
            }
            for agent_id, agent in self._registered_agents.items()
        }

        with open(reg_path, 'w') as f:
            json.dump(registrations, f, indent=2)

    def update_status(self, status: AgentStatus) -> Dict[str, Any]:
        """
        Update UCTS agent status.

        Args:
            status: New status

        Returns:
            Status update result
        """
        old_status = self._status
        self._status = status

        # Update all registered agents
        for agent in self._registered_agents.values():
            agent.status = status

        self._save_registrations()

        return {
            "old_status": old_status.value,
            "new_status": status.value,
            "agents_updated": len(self._registered_agents)
        }

    def discover_agents(
        self,
        capability: Optional[str] = None,
        max_cost_tier: Optional[CostTier] = None,
        status: Optional[AgentStatus] = None
    ) -> List[Dict[str, Any]]:
        """
        Discover agents from ecosystem.

        Args:
            capability: Filter by capability
            max_cost_tier: Maximum cost tier (LOCAL AI FIRST)
            status: Filter by status

        Returns:
            List of matching agents
        """
        agents_path = self.danai_path / "ecosystem_agents.json"

        if not agents_path.exists():
            # Return UCTS agents only if no ecosystem file
            return [
                {
                    "agent_id": a.agent_id,
                    "name": a.name,
                    "capabilities": a.capabilities,
                    "cost_tier": a.cost_tier.value,
                    "status": a.status.value
                }
                for a in self._registered_agents.values()
            ]

        try:
            with open(agents_path) as f:
                all_agents = json.load(f)
        except (json.JSONDecodeError, OSError):
            return []

        # Filter agents
        results = []
        cost_order = [CostTier.FREE, CostTier.ECONOMY, CostTier.LIGHTWEIGHT, CostTier.STANDARD, CostTier.PREMIUM]

        for agent in all_agents:
            # Capability filter
            if capability and capability not in agent.get("capabilities", []):
                continue

            # Cost tier filter (LOCAL AI FIRST)
            if max_cost_tier:
                agent_tier = CostTier(agent.get("cost_tier", "standard"))
                if cost_order.index(agent_tier) > cost_order.index(max_cost_tier):
                    continue

            # Status filter
            if status and agent.get("status") != status.value:
                continue

            results.append(agent)

        # Sort by cost tier (LOCAL AI FIRST)
        def cost_sort_key(a):
            try:
                return cost_order.index(CostTier(a.get("cost_tier", "standard")))
            except ValueError:
                return 99

        results.sort(key=cost_sort_key)

        return results

    def get_routing_recommendation(
        self,
        task_type: str,
        required_capabilities: Optional[List[str]] = None
    ) -> Dict[str, Any]:
        """
        Get LOCAL AI FIRST routing recommendation.

        Args:
            task_type: Type of task
            required_capabilities: Required agent capabilities

        Returns:
            Routing recommendation
        """
        # First try to find a FREE tier agent (local)
        agents = self.discover_agents(max_cost_tier=CostTier.FREE)

        if required_capabilities:
            agents = [
                a for a in agents
                if all(cap in a.get("capabilities", []) for cap in required_capabilities)
            ]

        if agents:
            return {
                "recommended_agent": agents[0]["agent_id"],
                "cost_tier": agents[0]["cost_tier"],
                "reasoning": "LOCAL AI FIRST - using free/local agent",
                "alternatives": [a["agent_id"] for a in agents[1:3]]
            }

        # Fall back to economy tier
        agents = self.discover_agents(max_cost_tier=CostTier.ECONOMY)
        if required_capabilities:
            agents = [
                a for a in agents
                if all(cap in a.get("capabilities", []) for cap in required_capabilities)
            ]

        if agents:
            return {
                "recommended_agent": agents[0]["agent_id"],
                "cost_tier": agents[0]["cost_tier"],
                "reasoning": "LOCAL AI FIRST - using economy tier",
                "alternatives": [a["agent_id"] for a in agents[1:3]]
            }

        # Default to UCTS if no matches
        return {
            "recommended_agent": "ucts-forge",
            "cost_tier": CostTier.FREE.value,
            "reasoning": "Defaulting to UCTS local processing",
            "alternatives": []
        }

    def accept_task(self, task: TaskRequest) -> Dict[str, Any]:
        """
        Accept an incoming task from ecosystem.

        Args:
            task: Task request

        Returns:
            Acceptance result
        """
        self._pending_tasks.append(task)

        # Write to pending tasks file
        tasks_path = self.danai_path / "pending_tasks.json"
        tasks_path.parent.mkdir(parents=True, exist_ok=True)

        tasks_data = []
        if tasks_path.exists():
            try:
                with open(tasks_path) as f:
                    tasks_data = json.load(f)
            except (json.JSONDecodeError, OSError):
                tasks_data = []

        tasks_data.append({
            "task_id": task.task_id,
            "task_type": task.task_type,
            "payload": task.payload,
            "requester": task.requester,
            "priority": task.priority,
            "accepted_at": datetime.now().isoformat(),
            "status": "pending"
        })

        with open(tasks_path, 'w') as f:
            json.dump(tasks_data, f, indent=2)

        return {
            "status": "accepted",
            "task_id": task.task_id,
            "position": len(self._pending_tasks)
        }

    def complete_task(self, task_id: str, result: Dict[str, Any]) -> Dict[str, Any]:
        """
        Mark a task as complete.

        Args:
            task_id: Task identifier
            result: Task result

        Returns:
            Completion result
        """
        tasks_path = self.danai_path / "pending_tasks.json"

        if not tasks_path.exists():
            return {"status": "not_found", "task_id": task_id}

        try:
            with open(tasks_path) as f:
                tasks_data = json.load(f)
        except (json.JSONDecodeError, OSError):
            return {"status": "error", "task_id": task_id}

        found = False
        for task in tasks_data:
            if task.get("task_id") == task_id:
                task["status"] = "completed"
                task["completed_at"] = datetime.now().isoformat()
                task["result"] = result
                found = True
                break

        if not found:
            return {"status": "not_found", "task_id": task_id}

        with open(tasks_path, 'w') as f:
            json.dump(tasks_data, f, indent=2)

        # Remove from pending list
        self._pending_tasks = [t for t in self._pending_tasks if t.task_id != task_id]

        return {"status": "completed", "task_id": task_id}

    def report_task_completion(
        self,
        task_id: str,
        agent_id: str,
        success: bool,
        metrics: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Report task completion to DanAI.

        Args:
            task_id: Task identifier
            agent_id: Agent that completed the task
            success: Whether task succeeded
            metrics: Optional metrics data

        Returns:
            Completion report
        """
        completion = {
            "task_id": task_id,
            "agent_id": agent_id,
            "success": success,
            "metrics": metrics or {},
            "timestamp": datetime.now().isoformat()
        }

        completions_path = self.danai_path / "completions.json"
        completions_path.parent.mkdir(parents=True, exist_ok=True)

        completions = []
        if completions_path.exists():
            try:
                with open(completions_path) as f:
                    completions = json.load(f)
            except (json.JSONDecodeError, OSError):
                completions = []

        completions.append(completion)
        completions = completions[-500:]

        with open(completions_path, 'w') as f:
            json.dump(completions, f, indent=2)

        logger.debug(f"Reported task completion: {task_id}")
        return completion

    def report_task_metrics(
        self,
        task_type: str,
        success: bool,
        duration_seconds: float,
        cost_tier_used: CostTier
    ) -> Dict[str, Any]:
        """
        Report task completion metrics for virtuous cycle.

        Args:
            task_type: Type of task
            success: Whether task succeeded
            duration_seconds: Task duration
            cost_tier_used: Cost tier that was used

        Returns:
            Metrics report
        """
        metrics = {
            "task_type": task_type,
            "success": success,
            "duration_seconds": duration_seconds,
            "cost_tier": cost_tier_used.value,
            "agent": "ucts",
            "timestamp": datetime.now().isoformat()
        }

        metrics_path = self.danai_path / "task_metrics.json"
        metrics_path.parent.mkdir(parents=True, exist_ok=True)

        all_metrics = []
        if metrics_path.exists():
            try:
                with open(metrics_path) as f:
                    all_metrics = json.load(f)
            except (json.JSONDecodeError, OSError):
                all_metrics = []

        all_metrics.append(metrics)
        all_metrics = all_metrics[-1000:]

        with open(metrics_path, 'w') as f:
            json.dump(all_metrics, f, indent=2)

        return metrics
