"""
Coherence Ecosystem Integration Manager

Coordinates all ecosystem project integrations:
- GCU (Ground Control Ultra) - Trust & health monitoring
- Sentinel - Constitutional AI & policies
- DanAI - Agent marketplace & routing
- MMB - Semantic memory bridge
- WB2 - Dashboard & visualization
"""

import os
import json
import logging
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime
from dataclasses import dataclass, field
from enum import Enum

logger = logging.getLogger(__name__)


class IntegrationMode(Enum):
    """Integration connection mode"""
    SDK = "sdk"           # Use coherence-sdk (async, network)
    FILE = "file"         # File-based integration (sync, local)
    HYBRID = "hybrid"     # Try SDK, fall back to file


@dataclass
class EcosystemConfig:
    """Configuration for ecosystem integrations"""
    mode: IntegrationMode = IntegrationMode.FILE
    coherence_url: str = field(default_factory=lambda: os.getenv("COHERENCE_URL", "http://localhost:5000"))
    api_key: Optional[str] = field(default_factory=lambda: os.getenv("COHERENCE_API_KEY"))
    base_path: str = field(default_factory=lambda: os.getenv("COHERENCE_BASE_PATH", str(Path.home() / ".coherence")))
    project_id: str = "UCTS"
    auto_register: bool = True
    event_broadcasting: bool = True


@dataclass
class IntegrationStatus:
    """Status of an integration"""
    name: str
    available: bool
    mode: IntegrationMode
    last_sync: Optional[datetime] = None
    error: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)


class EcosystemManager:
    """
    Central manager for Coherence ecosystem integrations.

    Provides unified access to all ecosystem projects and coordinates
    cross-project operations.
    """

    def __init__(self, config: Optional[EcosystemConfig] = None):
        self.config = config or EcosystemConfig()
        self._integrations: Dict[str, Any] = {}
        self._status_cache: Dict[str, IntegrationStatus] = {}
        self._initialized = False

        # SDK client (lazy loaded)
        self._sdk_client = None
        self._sdk_available = None

    @property
    def base_path(self) -> Path:
        return Path(self.config.base_path)

    @property
    def sdk_available(self) -> bool:
        """Check if coherence-sdk is available"""
        if self._sdk_available is None:
            try:
                import coherence_sdk
                self._sdk_available = True
            except ImportError:
                self._sdk_available = False
        return self._sdk_available

    def get_effective_mode(self) -> IntegrationMode:
        """Get the effective integration mode based on config and availability"""
        if self.config.mode == IntegrationMode.SDK:
            return IntegrationMode.SDK if self.sdk_available else IntegrationMode.FILE
        elif self.config.mode == IntegrationMode.HYBRID:
            return IntegrationMode.SDK if self.sdk_available else IntegrationMode.FILE
        return IntegrationMode.FILE

    def initialize(self) -> Dict[str, IntegrationStatus]:
        """
        Initialize all ecosystem integrations.

        Returns:
            Dictionary of integration statuses
        """
        if self._initialized:
            return self._status_cache

        # Ensure base path exists
        self.base_path.mkdir(parents=True, exist_ok=True)

        # Initialize each integration
        from ucts.integrations.gcu import GCUIntegration
        from ucts.integrations.danai import DanAIIntegration
        from ucts.integrations.mmb import MMBIntegration
        from ucts.integrations.wb2 import WB2Integration
        from ucts.integrations.sentinel import SentinelIntegration

        integrations = [
            ("gcu", GCUIntegration),
            ("sentinel", SentinelIntegration),
            ("danai", DanAIIntegration),
            ("mmb", MMBIntegration),
            ("wb2", WB2Integration),
        ]

        mode = self.get_effective_mode()

        for name, cls in integrations:
            try:
                if name == "sentinel":
                    # Sentinel uses different constructor
                    self._integrations[name] = cls()
                else:
                    self._integrations[name] = cls(self.config)

                self._status_cache[name] = IntegrationStatus(
                    name=name,
                    available=True,
                    mode=mode,
                    last_sync=datetime.now()
                )
            except Exception as e:
                logger.warning(f"Failed to initialize {name} integration: {e}")
                self._status_cache[name] = IntegrationStatus(
                    name=name,
                    available=False,
                    mode=mode,
                    error=str(e)
                )

        self._initialized = True

        # Auto-register if configured
        if self.config.auto_register:
            self._register_ucts()

        return self._status_cache

    def _register_ucts(self):
        """Register UCTS with ecosystem"""
        registration = {
            "project_id": self.config.project_id,
            "name": "Universal Context Transfer System",
            "type": "code-generation",
            "version": "2.0.0",
            "capabilities": [
                "session_ingestion",
                "code_extraction",
                "project_generation",
                "project_merge",
                "docker_generation",
                "template_generation",
            ],
            "source_formats": ["claude_desktop", "vscode", "web_chat", "transcript"],
            "output_targets": ["vscode", "gitlab", "github", "docker"],
            "registered_at": datetime.now().isoformat()
        }

        # Write to local registration file
        reg_path = self.base_path / "ucts" / "registration.json"
        reg_path.parent.mkdir(parents=True, exist_ok=True)

        with open(reg_path, 'w') as f:
            json.dump(registration, f, indent=2)

        logger.info(f"UCTS registered with ecosystem at {reg_path}")

    def get_integration(self, name: str) -> Optional[Any]:
        """Get a specific integration by name"""
        if not self._initialized:
            self.initialize()
        return self._integrations.get(name)

    @property
    def gcu(self):
        """Get GCU integration"""
        return self.get_integration("gcu")

    @property
    def sentinel(self):
        """Get Sentinel integration"""
        return self.get_integration("sentinel")

    @property
    def danai(self):
        """Get DanAI integration"""
        return self.get_integration("danai")

    @property
    def mmb(self):
        """Get MMB integration"""
        return self.get_integration("mmb")

    @property
    def wb2(self):
        """Get WB2 integration"""
        return self.get_integration("wb2")

    def health_check(self) -> Dict[str, Any]:
        """
        Perform health check on all integrations.

        Returns:
            Health status for all integrations
        """
        if not self._initialized:
            self.initialize()

        results = {
            "timestamp": datetime.now().isoformat(),
            "mode": self.get_effective_mode().value,
            "integrations": {}
        }

        for name, integration in self._integrations.items():
            try:
                if hasattr(integration, 'health_check'):
                    results["integrations"][name] = integration.health_check()
                else:
                    results["integrations"][name] = {"status": "unknown", "available": True}
            except Exception as e:
                results["integrations"][name] = {"status": "error", "error": str(e)}

        # Overall health
        all_healthy = all(
            r.get("status") != "error"
            for r in results["integrations"].values()
        )
        results["healthy"] = all_healthy

        return results

    def broadcast_event(self, event_type: str, payload: Dict[str, Any]) -> Dict[str, Any]:
        """
        Broadcast an event to all ecosystem projects.

        Args:
            event_type: Type of event (forge_complete, merge_complete, etc.)
            payload: Event payload

        Returns:
            Broadcast result
        """
        if not self.config.event_broadcasting:
            return {"status": "disabled", "broadcast": False}

        event = {
            "event_id": f"ucts-{datetime.now().strftime('%Y%m%d%H%M%S%f')}",
            "event_type": event_type,
            "source": self.config.project_id,
            "timestamp": datetime.now().isoformat(),
            "payload": payload
        }

        # Write to events file
        events_path = self.base_path / "ucts" / "events.json"
        events_path.parent.mkdir(parents=True, exist_ok=True)

        events = []
        if events_path.exists():
            try:
                with open(events_path) as f:
                    events = json.load(f)
            except (json.JSONDecodeError, OSError):
                events = []

        events.append(event)

        # Keep last 1000 events
        events = events[-1000:]

        with open(events_path, 'w') as f:
            json.dump(events, f, indent=2)

        logger.info(f"Broadcast event: {event_type}")

        return {"status": "broadcast", "event_id": event["event_id"]}

    def get_ecosystem_status(self) -> Dict[str, Any]:
        """Get status of entire ecosystem"""
        if not self._initialized:
            self.initialize()

        return {
            "ucts": {
                "status": "operational",
                "version": "2.0.0",
                "mode": self.get_effective_mode().value,
            },
            "integrations": {
                name: {
                    "available": status.available,
                    "mode": status.mode.value,
                    "last_sync": status.last_sync.isoformat() if status.last_sync else None,
                    "error": status.error
                }
                for name, status in self._status_cache.items()
            },
            "timestamp": datetime.now().isoformat()
        }


# Singleton instance
_ecosystem: Optional[EcosystemManager] = None


def get_ecosystem(config: Optional[EcosystemConfig] = None) -> EcosystemManager:
    """Get the ecosystem manager singleton"""
    global _ecosystem
    if _ecosystem is None:
        _ecosystem = EcosystemManager(config)
    return _ecosystem


def reset_ecosystem():
    """Reset the ecosystem singleton (for testing)"""
    global _ecosystem
    _ecosystem = None
