"""
UCTS Forge Pipeline with Ecosystem Integration

Implements the synergies from SYNERGIES.md:
- MMB session storage integration
- Sentinel pre-forge validation
- DanAI agent registration
- GCU health reporting
- MMB pattern discovery
- DanAI task routing
- Cross-project template sharing
- Trust-gated operations
- Real-time event streaming
"""

import asyncio
import json
import logging
import time
from datetime import datetime
from pathlib import Path
from typing import Dict, Any, Optional, List, Callable
from dataclasses import dataclass, field
from enum import Enum

logger = logging.getLogger(__name__)


class PipelinePhase(Enum):
    """Forge pipeline phases"""
    PRE_INGEST = "pre_ingest"
    INGEST = "ingest"
    POST_INGEST = "post_ingest"
    PRE_ANALYZE = "pre_analyze"
    ANALYZE = "analyze"
    POST_ANALYZE = "post_analyze"
    PRE_FORGE = "pre_forge"
    FORGE = "forge"
    POST_FORGE = "post_forge"
    PRE_MERGE = "pre_merge"
    MERGE = "merge"
    POST_MERGE = "post_merge"


class PipelineStatus(Enum):
    """Pipeline execution status"""
    PENDING = "pending"
    RUNNING = "running"
    SUCCESS = "success"
    FAILED = "failed"
    BLOCKED = "blocked"
    SKIPPED = "skipped"


@dataclass
class PipelineEvent:
    """Event emitted during pipeline execution"""
    phase: PipelinePhase
    status: PipelineStatus
    timestamp: datetime = field(default_factory=datetime.now)
    data: Dict[str, Any] = field(default_factory=dict)
    error: Optional[str] = None


@dataclass
class PipelineContext:
    """Context passed through pipeline phases"""
    session_id: str
    source_path: Optional[str] = None
    output_path: Optional[str] = None
    target: str = "vscode"
    languages: List[str] = field(default_factory=list)
    dependencies: Dict[str, List[str]] = field(default_factory=dict)
    code_blocks: int = 0
    patterns_detected: List[str] = field(default_factory=list)
    security_scan_passed: bool = True
    trust_tier: int = 1
    user_id: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)


class ForgePipeline:
    """
    Forge pipeline with full ecosystem integration.

    Implements all synergies from SYNERGIES.md:
    - Pre-forge: Trust checks, policy validation, pattern search
    - Post-forge: Security scan, session storage, metrics recording
    """

    def __init__(self):
        self._hooks: Dict[PipelinePhase, List[Callable]] = {
            phase: [] for phase in PipelinePhase
        }
        self._event_listeners: List[Callable[[PipelineEvent], None]] = []
        self._setup_ecosystem_hooks()

    def _setup_ecosystem_hooks(self):
        """Register default ecosystem integration hooks"""
        # Pre-forge hooks
        self.register_hook(PipelinePhase.PRE_FORGE, self._gcu_trust_check)
        self.register_hook(PipelinePhase.PRE_FORGE, self._sentinel_policy_validation)
        self.register_hook(PipelinePhase.PRE_FORGE, self._mmb_pattern_search)

        # Post-forge hooks
        self.register_hook(PipelinePhase.POST_FORGE, self._sentinel_security_scan)
        self.register_hook(PipelinePhase.POST_FORGE, self._mmb_session_storage)
        self.register_hook(PipelinePhase.POST_FORGE, self._gcu_metrics_report)
        self.register_hook(PipelinePhase.POST_FORGE, self._mmb_pattern_extraction)
        self.register_hook(PipelinePhase.POST_FORGE, self._danai_task_complete)
        self.register_hook(PipelinePhase.POST_FORGE, self._wb2_activity_feed)

    def register_hook(self, phase: PipelinePhase, hook: Callable):
        """Register a hook for a pipeline phase"""
        self._hooks[phase].append(hook)
        logger.debug(f"Registered hook for {phase.value}: {hook.__name__}")

    def add_event_listener(self, listener: Callable[[PipelineEvent], None]):
        """Add listener for pipeline events (real-time streaming)"""
        self._event_listeners.append(listener)

    def _emit_event(self, event: PipelineEvent):
        """Emit event to all listeners"""
        for listener in self._event_listeners:
            try:
                listener(event)
            except Exception as e:
                logger.warning(f"Event listener failed: {e}")

    async def execute(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Execute the full forge pipeline.

        Args:
            context: Pipeline context with session info

        Returns:
            Pipeline execution result
        """
        start_time = time.time()
        results: Dict[str, Any] = {
            "session_id": context.session_id,
            "phases": {},
            "status": "pending",
        }

        try:
            # Pre-forge phase
            await self._run_phase(PipelinePhase.PRE_FORGE, context, results)

            if results["phases"].get(PipelinePhase.PRE_FORGE.value, {}).get("status") == "blocked":
                results["status"] = "blocked"
                return results

            # Forge phase
            await self._run_phase(PipelinePhase.FORGE, context, results)

            # Post-forge phase
            await self._run_phase(PipelinePhase.POST_FORGE, context, results)

            results["status"] = "success"
            results["duration_seconds"] = time.time() - start_time

        except Exception as e:
            logger.error(f"Pipeline failed: {e}")
            results["status"] = "failed"
            results["error"] = str(e)

        return results

    async def _run_phase(
        self,
        phase: PipelinePhase,
        context: PipelineContext,
        results: Dict
    ):
        """Run all hooks for a pipeline phase"""
        phase_results = []
        phase_status = PipelineStatus.RUNNING

        self._emit_event(PipelineEvent(
            phase=phase,
            status=PipelineStatus.RUNNING,
            data={"session_id": context.session_id}
        ))

        for hook in self._hooks[phase]:
            try:
                result = await self._call_hook(hook, context)
                phase_results.append({
                    "hook": hook.__name__,
                    "status": "success",
                    "result": result
                })

                # Check for blocking result
                if isinstance(result, dict) and result.get("blocked"):
                    phase_status = PipelineStatus.BLOCKED
                    break

            except Exception as e:
                logger.error(f"Hook {hook.__name__} failed: {e}")
                phase_results.append({
                    "hook": hook.__name__,
                    "status": "failed",
                    "error": str(e)
                })

        if phase_status != PipelineStatus.BLOCKED:
            phase_status = PipelineStatus.SUCCESS

        results["phases"][phase.value] = {
            "status": phase_status.value,
            "hooks": phase_results
        }

        self._emit_event(PipelineEvent(
            phase=phase,
            status=phase_status,
            data={"results": phase_results}
        ))

    async def _call_hook(self, hook: Callable, context: PipelineContext) -> Any:
        """Call a hook, handling both sync and async functions"""
        if asyncio.iscoroutinefunction(hook):
            return await hook(context)
        else:
            return hook(context)

    # =========================================================================
    # GCU Integration Hooks (Trust & Health)
    # =========================================================================

    async def _gcu_trust_check(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Check trust tier before forge operation.

        Implements: SYNERGIES Section 1 - GCU → UCTS Trust-Gated Operations
        """
        try:
            from ucts.integrations.gcu import GCUIntegration

            gcu = GCUIntegration()

            # Determine required trust based on operation
            operation = "forge"
            if context.metadata.get("has_deploy_scripts"):
                operation = "forge_deploy"
            elif context.metadata.get("has_auth_code"):
                operation = "forge_auth"
            elif context.metadata.get("has_secrets"):
                operation = "forge_secrets"

            agent_id = context.user_id or "ucts-cli"
            result = gcu.check_trust(agent_id, operation)

            context.trust_tier = result.get("agent_tier", 1)

            if not result.get("approved"):
                logger.warning(f"Trust check failed: {result.get('reason')}")
                return {"blocked": True, "reason": result.get("reason")}

            return {"approved": True, "trust_tier": context.trust_tier}

        except ImportError:
            logger.debug("GCU integration not available")
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"GCU trust check failed: {e}")
            return {"error": str(e)}

    async def _gcu_metrics_report(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Report forge metrics to GCU.

        Implements: SYNERGIES Section 1 - UCTS → GCU Health Reporting
        """
        try:
            from ucts.integrations.gcu import GCUIntegration

            gcu = GCUIntegration()
            result = gcu.report_forge_metrics(
                success=True,
                duration_seconds=context.metadata.get("duration_seconds", 0),
                files_created=context.metadata.get("files_created", 0),
                languages=context.languages,
                target_type=context.target
            )

            return {"reported": True, "metrics": result}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"GCU metrics report failed: {e}")
            return {"error": str(e)}

    # =========================================================================
    # Sentinel Integration Hooks (Security)
    # =========================================================================

    async def _sentinel_policy_validation(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Validate forge operation against policies.

        Implements: SYNERGIES Section 2 - Sentinel → UCTS Policy Enforcement
        """
        try:
            from ucts.integrations.sentinel import SentinelIntegration

            sentinel = SentinelIntegration()
            result = sentinel.validate_forge_policy(
                languages=context.languages,
                has_auth_patterns=context.metadata.get("has_auth_code", False),
                has_db_operations=context.metadata.get("has_db_operations", False),
                has_api_keys=context.metadata.get("has_api_keys", False),
                estimated_complexity=context.metadata.get("complexity", "medium")
            )

            if not result.get("approved"):
                if result.get("decision") == "requires_review":
                    logger.warning("Forge requires human review")
                    return {"requires_review": True, "requirements": result.get("requirements")}
                return {"blocked": True, "reason": result.get("reasons")}

            return {"approved": True, "requirements": result.get("requirements", [])}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"Sentinel policy validation failed: {e}")
            return {"error": str(e)}

    async def _sentinel_security_scan(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Scan generated code for security vulnerabilities.

        Implements: SYNERGIES Section 2 - Sentinel → UCTS Security Scanning
        """
        try:
            from ucts.integrations.sentinel import SentinelIntegration

            sentinel = SentinelIntegration()

            # Get generated files
            output_path = Path(context.output_path) if context.output_path else None
            if not output_path or not output_path.exists():
                return {"skipped": True, "reason": "No output path"}

            # Collect files to scan
            files_to_scan: Dict[str, str] = {}
            for ext in [".py", ".js", ".ts", ".jsx", ".tsx", ".go", ".rs"]:
                for file_path in output_path.rglob(f"*{ext}"):
                    try:
                        with open(file_path, 'r', encoding='utf-8') as f:
                            files_to_scan[str(file_path)] = f.read()
                    except Exception:
                        continue

            if not files_to_scan:
                return {"skipped": True, "reason": "No scannable files"}

            # Run security scan
            result = sentinel.scan_project_security(files_to_scan)

            context.security_scan_passed = result.passed

            return {
                "passed": result.passed,
                "violations": len(result.violations),
                "warnings": len(result.warnings),
                "scanned_files": result.scanned_files
            }

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"Sentinel security scan failed: {e}")
            return {"error": str(e)}

    # =========================================================================
    # MMB Integration Hooks (Memory & Patterns)
    # =========================================================================

    async def _mmb_pattern_search(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Search for existing patterns before forging.

        Implements: SYNERGIES Section 4 - MMB → UCTS Pattern Retrieval
        """
        try:
            from ucts.integrations.mmb import MMBIntegration

            mmb = MMBIntegration()

            # Search for similar sessions
            sessions = mmb.search_sessions(
                languages=context.languages,
                limit=5
            )

            # Search for relevant patterns
            patterns = []
            for lang in context.languages[:3]:  # Limit to 3 languages
                lang_patterns = mmb.search_patterns(language=lang, limit=5)
                patterns.extend(lang_patterns)

            return {
                "similar_sessions": len(sessions),
                "relevant_patterns": len(patterns),
                "sessions": [s.get("session_id") for s in sessions],
                "patterns": [p.get("name") for p in patterns[:10]]
            }

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"MMB pattern search failed: {e}")
            return {"error": str(e)}

    async def _mmb_session_storage(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Store forge session in semantic memory.

        Implements: SYNERGIES Section 4 - UCTS → MMB Session Storage
        """
        try:
            from ucts.integrations.mmb import MMBIntegration

            mmb = MMBIntegration()
            result = mmb.store_session(
                session_id=context.session_id,
                summary=context.metadata.get("summary", "Forge session"),
                languages=context.languages,
                dependencies=context.dependencies,
                code_blocks=context.code_blocks,
                patterns_detected=context.patterns_detected,
                tags=context.metadata.get("tags", [])
            )

            return {"stored": True, "session_id": context.session_id}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"MMB session storage failed: {e}")
            return {"error": str(e)}

    async def _mmb_pattern_extraction(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Extract and store reusable patterns from session.

        Implements: SYNERGIES Section 4 - UCTS → MMB Pattern Library
        """
        try:
            from ucts.integrations.mmb import MMBIntegration

            mmb = MMBIntegration()

            # Extract patterns from code blocks
            patterns_stored = 0
            for i, pattern_name in enumerate(context.patterns_detected[:10]):
                result = mmb.store_code_pattern(
                    name=pattern_name,
                    pattern_type="snippet",
                    language=context.languages[0] if context.languages else "unknown",
                    code=f"# Pattern: {pattern_name}",  # Placeholder
                    description=f"Pattern extracted from session {context.session_id}",
                    tags=["auto-extracted", context.session_id]
                )
                if result.get("status") == "stored":
                    patterns_stored += 1

            return {"patterns_stored": patterns_stored}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"MMB pattern extraction failed: {e}")
            return {"error": str(e)}

    # =========================================================================
    # DanAI Integration Hooks (Agent Routing)
    # =========================================================================

    async def _danai_task_complete(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Report task completion to DanAI.

        Implements: SYNERGIES Section 3 - UCTS → DanAI Task Completion Reports
        """
        try:
            from ucts.integrations.danai import DanAIIntegration

            danai = DanAIIntegration()
            result = danai.report_task_completion(
                task_id=context.session_id,
                agent_id="ucts-forge",
                success=True,
                metrics={
                    "languages": context.languages,
                    "files_created": context.metadata.get("files_created", 0),
                    "code_blocks": context.code_blocks,
                    "target": context.target
                }
            )

            return {"reported": True}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"DanAI task report failed: {e}")
            return {"error": str(e)}

    # =========================================================================
    # WB2 Integration Hooks (Dashboard)
    # =========================================================================

    async def _wb2_activity_feed(self, context: PipelineContext) -> Dict[str, Any]:
        """
        Send forge activity to WB2 dashboard.

        Implements: SYNERGIES Section 5 - UCTS → WB2 Activity Feed
        """
        try:
            from ucts.integrations.wb2 import WB2Integration

            wb2 = WB2Integration()
            result = wb2.send_activity(
                activity_type="forge_complete",
                project="ucts",
                data={
                    "session_id": context.session_id,
                    "languages": context.languages,
                    "files_created": context.metadata.get("files_created", 0),
                    "target": context.target,
                    "timestamp": datetime.now().isoformat()
                }
            )

            return {"sent": True}

        except ImportError:
            return {"skipped": True}
        except Exception as e:
            logger.warning(f"WB2 activity feed failed: {e}")
            return {"error": str(e)}


# =============================================================================
# Pipeline Factory
# =============================================================================

_default_pipeline: Optional[ForgePipeline] = None


def get_forge_pipeline() -> ForgePipeline:
    """Get or create the default forge pipeline"""
    global _default_pipeline
    if _default_pipeline is None:
        _default_pipeline = ForgePipeline()
    return _default_pipeline


async def run_forge_pipeline(
    session_id: str,
    source_path: Optional[str] = None,
    output_path: Optional[str] = None,
    target: str = "vscode",
    user_id: Optional[str] = None,
    **metadata
) -> Dict[str, Any]:
    """
    Run the forge pipeline with ecosystem integration.

    Args:
        session_id: Unique session identifier
        source_path: Path to source conversation
        output_path: Output directory for generated project
        target: Target platform (vscode, github, gitlab, etc.)
        user_id: User ID for trust checks
        **metadata: Additional metadata

    Returns:
        Pipeline execution result
    """
    pipeline = get_forge_pipeline()
    context = PipelineContext(
        session_id=session_id,
        source_path=source_path,
        output_path=output_path,
        target=target,
        user_id=user_id,
        metadata=metadata
    )

    return await pipeline.execute(context)
