"""
Ground Control Ultra (GCU) Integration

Provides:
- Trust tier management for UCTS operations
- Health reporting to ecosystem
- Alert integration
- Trust-gated code generation
"""

import json
import logging
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime
from dataclasses import dataclass
from enum import Enum

logger = logging.getLogger(__name__)


class TrustTier(Enum):
    """Trust tiers for operations"""
    UNTRUSTED = 0
    BASIC = 1
    VERIFIED = 2
    TRUSTED = 3
    PRIVILEGED = 4


class AlertSeverity(Enum):
    """Alert severity levels"""
    INFO = "info"
    WARNING = "warning"
    ERROR = "error"
    CRITICAL = "critical"


@dataclass
class TrustRequirement:
    """Trust requirement for an operation"""
    operation: str
    min_tier: TrustTier
    reason: str


class GCUIntegration:
    """
    Integration with Ground Control Ultra for trust and health management.

    Provides:
    - Trust tier checking before operations
    - Health status reporting
    - Alert broadcasting
    - Operation audit logging
    """

    # Default trust requirements for UCTS operations
    TRUST_REQUIREMENTS = {
        "forge": TrustRequirement("forge", TrustTier.BASIC, "Basic code generation"),
        "forge_deploy": TrustRequirement("forge_deploy", TrustTier.TRUSTED, "Deploy script generation"),
        "forge_auth": TrustRequirement("forge_auth", TrustTier.VERIFIED, "Auth code generation"),
        "forge_secrets": TrustRequirement("forge_secrets", TrustTier.PRIVILEGED, "Secrets handling"),
        "merge": TrustRequirement("merge", TrustTier.VERIFIED, "Merge into existing project"),
        "docker": TrustRequirement("docker", TrustTier.VERIFIED, "Docker configuration"),
        "template": TrustRequirement("template", TrustTier.BASIC, "Template generation"),
    }

    def __init__(self, config=None):
        from ucts.integrations.ecosystem import EcosystemConfig
        self.config = config or EcosystemConfig()
        self._trust_cache: Dict[str, TrustTier] = {}
        self._health_status = "healthy"

    @property
    def base_path(self) -> Path:
        return Path(self.config.base_path)

    @property
    def gcu_path(self) -> Path:
        return self.base_path / "gcu"

    def health_check(self) -> Dict[str, Any]:
        """Perform health check"""
        return {
            "status": self._health_status,
            "project": "gcu",
            "trust_cache_size": len(self._trust_cache),
            "timestamp": datetime.now().isoformat()
        }

    def get_trust_tier(self, agent_id: str) -> TrustTier:
        """
        Get trust tier for an agent.

        Args:
            agent_id: Agent identifier

        Returns:
            Trust tier for the agent
        """
        if agent_id in self._trust_cache:
            return self._trust_cache[agent_id]

        # Check trust file
        trust_file = self.gcu_path / "trust_tiers.json"

        if trust_file.exists():
            try:
                with open(trust_file) as f:
                    trust_data = json.load(f)
                tier_value = trust_data.get(agent_id, 0)
                tier = TrustTier(tier_value)
                self._trust_cache[agent_id] = tier
                return tier
            except (json.JSONDecodeError, OSError, ValueError) as e:
                logger.warning(f"Failed to read trust tiers: {e}")

        # Default to BASIC for UCTS operations
        return TrustTier.BASIC

    def check_trust(self, agent_id: str, operation: str) -> Dict[str, Any]:
        """
        Check if agent has sufficient trust for operation.

        Args:
            agent_id: Agent identifier
            operation: Operation to perform

        Returns:
            Trust check result with approved/denied status
        """
        agent_tier = self.get_trust_tier(agent_id)
        requirement = self.TRUST_REQUIREMENTS.get(
            operation,
            TrustRequirement(operation, TrustTier.BASIC, "Default operation")
        )

        approved = agent_tier.value >= requirement.min_tier.value

        result = {
            "agent_id": agent_id,
            "operation": operation,
            "agent_tier": agent_tier.value,
            "required_tier": requirement.min_tier.value,
            "approved": approved,
            "reason": requirement.reason if approved else f"Requires {requirement.min_tier.name} tier",
            "timestamp": datetime.now().isoformat()
        }

        # Log the check
        self._log_trust_check(result)

        return result

    def _log_trust_check(self, result: Dict[str, Any]):
        """Log trust check to audit file"""
        audit_path = self.gcu_path / "trust_audit.json"
        audit_path.parent.mkdir(parents=True, exist_ok=True)

        audits = []
        if audit_path.exists():
            try:
                with open(audit_path) as f:
                    audits = json.load(f)
            except (json.JSONDecodeError, OSError):
                audits = []

        audits.append(result)
        audits = audits[-1000:]  # Keep last 1000

        with open(audit_path, 'w') as f:
            json.dump(audits, f, indent=2)

    def report_health(self, status: str, metrics: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """
        Report UCTS health to GCU.

        Args:
            status: Health status (healthy, degraded, unhealthy)
            metrics: Optional metrics to include

        Returns:
            Health report result
        """
        self._health_status = status

        report = {
            "project_id": "ucts",
            "status": status,
            "timestamp": datetime.now().isoformat(),
            "metrics": metrics or {},
            "capabilities": [
                "session_ingestion",
                "code_extraction",
                "project_generation",
            ]
        }

        # Write to health file
        health_path = self.gcu_path / "health" / "ucts.json"
        health_path.parent.mkdir(parents=True, exist_ok=True)

        with open(health_path, 'w') as f:
            json.dump(report, f, indent=2)

        logger.debug(f"Reported health: {status}")
        return report

    def send_alert(
        self,
        severity: AlertSeverity,
        title: str,
        message: str,
        context: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Send alert to GCU.

        Args:
            severity: Alert severity
            title: Alert title
            message: Alert message
            context: Optional context data

        Returns:
            Alert result
        """
        alert = {
            "alert_id": f"ucts-{datetime.now().strftime('%Y%m%d%H%M%S%f')}",
            "source": "ucts",
            "severity": severity.value,
            "title": title,
            "message": message,
            "context": context or {},
            "timestamp": datetime.now().isoformat()
        }

        # Write to alerts file
        alerts_path = self.gcu_path / "alerts.json"
        alerts_path.parent.mkdir(parents=True, exist_ok=True)

        alerts = []
        if alerts_path.exists():
            try:
                with open(alerts_path) as f:
                    alerts = json.load(f)
            except (json.JSONDecodeError, OSError):
                alerts = []

        alerts.append(alert)
        alerts = alerts[-500:]

        with open(alerts_path, 'w') as f:
            json.dump(alerts, f, indent=2)

        logger.info(f"Alert sent: [{severity.value}] {title}")
        return alert

    def report_forge_metrics(
        self,
        success: bool,
        duration_seconds: float,
        files_created: int,
        languages: List[str],
        target_type: str
    ) -> Dict[str, Any]:
        """
        Report forge operation metrics to GCU.

        Args:
            success: Whether forge succeeded
            duration_seconds: Operation duration
            files_created: Number of files created
            languages: Languages detected
            target_type: Target platform

        Returns:
            Metrics report
        """
        metrics = {
            "operation": "forge",
            "success": success,
            "duration_seconds": duration_seconds,
            "files_created": files_created,
            "languages": languages,
            "target_type": target_type,
            "timestamp": datetime.now().isoformat()
        }

        # Append to metrics file
        metrics_path = self.gcu_path / "metrics" / "ucts_forge.json"
        metrics_path.parent.mkdir(parents=True, exist_ok=True)

        all_metrics = []
        if metrics_path.exists():
            try:
                with open(metrics_path) as f:
                    all_metrics = json.load(f)
            except (json.JSONDecodeError, OSError):
                all_metrics = []

        all_metrics.append(metrics)
        all_metrics = all_metrics[-1000:]

        with open(metrics_path, 'w') as f:
            json.dump(all_metrics, f, indent=2)

        # Update health based on success rate
        recent = all_metrics[-100:]
        success_rate = sum(1 for m in recent if m.get("success")) / len(recent) if recent else 1.0

        if success_rate < 0.5:
            self.report_health("unhealthy", {"success_rate": success_rate})
        elif success_rate < 0.9:
            self.report_health("degraded", {"success_rate": success_rate})
        else:
            self.report_health("healthy", {"success_rate": success_rate})

        return metrics

    def get_ecosystem_health(self) -> Dict[str, Any]:
        """
        Get aggregated ecosystem health from GCU.

        Returns:
            Ecosystem health status
        """
        health_dir = self.gcu_path / "health"

        if not health_dir.exists():
            return {"status": "unknown", "projects": {}}

        projects = {}
        for health_file in health_dir.glob("*.json"):
            try:
                with open(health_file) as f:
                    data = json.load(f)
                projects[health_file.stem] = {
                    "status": data.get("status", "unknown"),
                    "timestamp": data.get("timestamp")
                }
            except (json.JSONDecodeError, OSError):
                projects[health_file.stem] = {"status": "error"}

        # Determine overall status
        statuses = [p["status"] for p in projects.values()]
        if "unhealthy" in statuses:
            overall = "unhealthy"
        elif "degraded" in statuses:
            overall = "degraded"
        elif all(s == "healthy" for s in statuses):
            overall = "healthy"
        else:
            overall = "unknown"

        return {
            "status": overall,
            "projects": projects,
            "timestamp": datetime.now().isoformat()
        }
