"""
MMB (Memory) Semantic Memory Integration

Provides:
- Session storage in semantic memory
- Code pattern storage and retrieval
- Template sharing across ecosystem
- Federated search for similar code
"""

import json
import logging
import hashlib
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime
from dataclasses import dataclass, field

logger = logging.getLogger(__name__)


@dataclass
class MemoryEntry:
    """Entry in semantic memory"""
    namespace: str
    key: str
    content: str
    metadata: Dict[str, Any] = field(default_factory=dict)
    created_at: datetime = field(default_factory=datetime.now)
    source_project: str = "ucts"


@dataclass
class SearchResult:
    """Result from semantic search"""
    entry: MemoryEntry
    score: float
    source: str


@dataclass
class CodePattern:
    """Reusable code pattern"""
    pattern_id: str
    name: str
    pattern_type: str  # function, class, module, snippet
    language: str
    code: str
    description: str
    tags: List[str] = field(default_factory=list)
    usage_count: int = 0


class MMBIntegration:
    """
    Integration with MMB semantic memory.

    Enables storage and retrieval of:
    - Forge sessions for ecosystem discovery
    - Code patterns for reuse
    - Project templates
    - Cross-project learning
    """

    def __init__(self, config=None):
        from ucts.integrations.ecosystem import EcosystemConfig
        self.config = config or EcosystemConfig()
        self._pattern_cache: Dict[str, CodePattern] = {}

    @property
    def base_path(self) -> Path:
        return Path(self.config.base_path)

    @property
    def mmb_path(self) -> Path:
        return self.base_path / "mmb"

    def health_check(self) -> Dict[str, Any]:
        """Perform health check"""
        sessions_count = len(list((self.mmb_path / "sessions").glob("*.json"))) if (self.mmb_path / "sessions").exists() else 0
        patterns_count = len(list((self.mmb_path / "patterns").glob("*.json"))) if (self.mmb_path / "patterns").exists() else 0

        return {
            "status": "healthy",
            "project": "mmb",
            "sessions_stored": sessions_count,
            "patterns_stored": patterns_count,
            "timestamp": datetime.now().isoformat()
        }

    def store_session(
        self,
        session_id: str,
        summary: str,
        languages: List[str],
        dependencies: Dict[str, List[str]],
        code_blocks: int,
        patterns_detected: List[str],
        tags: Optional[List[str]] = None
    ) -> Dict[str, Any]:
        """
        Store forge session in semantic memory.

        Args:
            session_id: Unique session identifier
            summary: Session summary/description
            languages: Languages detected
            dependencies: Dependencies by language
            code_blocks: Number of code blocks
            patterns_detected: Detected code patterns
            tags: Optional tags

        Returns:
            Storage result
        """
        session_data = {
            "session_id": session_id,
            "summary": summary,
            "languages": languages,
            "dependencies": dependencies,
            "code_blocks": code_blocks,
            "patterns_detected": patterns_detected,
            "tags": tags or [],
            "source_project": "ucts",
            "stored_at": datetime.now().isoformat()
        }

        # Store in sessions directory
        sessions_path = self.mmb_path / "sessions"
        sessions_path.mkdir(parents=True, exist_ok=True)

        session_file = sessions_path / f"{session_id}.json"
        with open(session_file, 'w') as f:
            json.dump(session_data, f, indent=2)

        # Update index
        self._update_session_index(session_id, session_data)

        logger.info(f"Stored session {session_id} in MMB")

        return {
            "status": "stored",
            "session_id": session_id,
            "path": str(session_file)
        }

    def _update_session_index(self, session_id: str, data: Dict[str, Any]):
        """Update session search index"""
        index_path = self.mmb_path / "sessions" / "_index.json"

        index = {}
        if index_path.exists():
            try:
                with open(index_path) as f:
                    index = json.load(f)
            except (json.JSONDecodeError, OSError):
                index = {}

        index[session_id] = {
            "summary": data.get("summary", "")[:200],
            "languages": data.get("languages", []),
            "tags": data.get("tags", []),
            "stored_at": data.get("stored_at")
        }

        with open(index_path, 'w') as f:
            json.dump(index, f, indent=2)

    def search_sessions(
        self,
        query: Optional[str] = None,
        languages: Optional[List[str]] = None,
        tags: Optional[List[str]] = None,
        limit: int = 10
    ) -> List[Dict[str, Any]]:
        """
        Search stored sessions.

        Args:
            query: Text query to match against summary
            languages: Filter by languages
            tags: Filter by tags
            limit: Maximum results

        Returns:
            Matching sessions
        """
        index_path = self.mmb_path / "sessions" / "_index.json"

        if not index_path.exists():
            return []

        try:
            with open(index_path) as f:
                index = json.load(f)
        except (json.JSONDecodeError, OSError):
            return []

        results = []
        query_lower = query.lower() if query else None

        for session_id, data in index.items():
            # Query filter
            if query_lower and query_lower not in data.get("summary", "").lower():
                continue

            # Language filter
            if languages:
                session_langs = data.get("languages", [])
                if not any(lang in session_langs for lang in languages):
                    continue

            # Tags filter
            if tags:
                session_tags = data.get("tags", [])
                if not any(tag in session_tags for tag in tags):
                    continue

            results.append({
                "session_id": session_id,
                **data
            })

        # Sort by stored_at (newest first)
        results.sort(key=lambda x: x.get("stored_at", ""), reverse=True)

        return results[:limit]

    def store_code_pattern(
        self,
        name: str,
        pattern_type: str,
        language: str,
        code: str,
        description: str,
        tags: Optional[List[str]] = None
    ) -> Dict[str, Any]:
        """
        Store a reusable code pattern.

        Args:
            name: Pattern name
            pattern_type: Type (function, class, module, snippet)
            language: Programming language
            code: The code pattern
            description: Description
            tags: Optional tags

        Returns:
            Storage result
        """
        # Generate pattern ID from content
        pattern_id = hashlib.sha256(f"{language}:{name}:{code}".encode()).hexdigest()[:16]

        pattern = CodePattern(
            pattern_id=pattern_id,
            name=name,
            pattern_type=pattern_type,
            language=language,
            code=code,
            description=description,
            tags=tags or []
        )

        self._pattern_cache[pattern_id] = pattern

        # Store to file
        patterns_path = self.mmb_path / "patterns"
        patterns_path.mkdir(parents=True, exist_ok=True)

        pattern_file = patterns_path / f"{pattern_id}.json"
        with open(pattern_file, 'w') as f:
            json.dump({
                "pattern_id": pattern.pattern_id,
                "name": pattern.name,
                "pattern_type": pattern.pattern_type,
                "language": pattern.language,
                "code": pattern.code,
                "description": pattern.description,
                "tags": pattern.tags,
                "usage_count": pattern.usage_count,
                "source_project": "ucts",
                "stored_at": datetime.now().isoformat()
            }, f, indent=2)

        # Update pattern index
        self._update_pattern_index(pattern)

        logger.info(f"Stored code pattern: {name} ({language})")

        return {
            "status": "stored",
            "pattern_id": pattern_id,
            "name": name
        }

    def _update_pattern_index(self, pattern: CodePattern):
        """Update pattern search index"""
        index_path = self.mmb_path / "patterns" / "_index.json"

        index = {}
        if index_path.exists():
            try:
                with open(index_path) as f:
                    index = json.load(f)
            except (json.JSONDecodeError, OSError):
                index = {}

        index[pattern.pattern_id] = {
            "name": pattern.name,
            "pattern_type": pattern.pattern_type,
            "language": pattern.language,
            "description": pattern.description[:200],
            "tags": pattern.tags
        }

        with open(index_path, 'w') as f:
            json.dump(index, f, indent=2)

    def search_patterns(
        self,
        query: Optional[str] = None,
        language: Optional[str] = None,
        pattern_type: Optional[str] = None,
        tags: Optional[List[str]] = None,
        limit: int = 10
    ) -> List[Dict[str, Any]]:
        """
        Search code patterns.

        Args:
            query: Text query
            language: Filter by language
            pattern_type: Filter by type
            tags: Filter by tags
            limit: Maximum results

        Returns:
            Matching patterns
        """
        index_path = self.mmb_path / "patterns" / "_index.json"

        if not index_path.exists():
            return []

        try:
            with open(index_path) as f:
                index = json.load(f)
        except (json.JSONDecodeError, OSError):
            return []

        results = []
        query_lower = query.lower() if query else None

        for pattern_id, data in index.items():
            # Query filter
            if query_lower:
                searchable = f"{data.get('name', '')} {data.get('description', '')}".lower()
                if query_lower not in searchable:
                    continue

            # Language filter
            if language and data.get("language") != language:
                continue

            # Type filter
            if pattern_type and data.get("pattern_type") != pattern_type:
                continue

            # Tags filter
            if tags:
                pattern_tags = data.get("tags", [])
                if not any(tag in pattern_tags for tag in tags):
                    continue

            results.append({
                "pattern_id": pattern_id,
                **data
            })

        return results[:limit]

    def get_pattern(self, pattern_id: str) -> Optional[Dict[str, Any]]:
        """
        Get full pattern by ID.

        Args:
            pattern_id: Pattern identifier

        Returns:
            Full pattern data or None
        """
        pattern_file = self.mmb_path / "patterns" / f"{pattern_id}.json"

        if not pattern_file.exists():
            return None

        try:
            with open(pattern_file) as f:
                data = json.load(f)

            # Increment usage count
            data["usage_count"] = data.get("usage_count", 0) + 1
            with open(pattern_file, 'w') as f:
                json.dump(data, f, indent=2)

            return data
        except (json.JSONDecodeError, OSError):
            return None

    def store_template(
        self,
        template_name: str,
        template_type: str,
        description: str,
        languages: List[str],
        files: Dict[str, str]
    ) -> Dict[str, Any]:
        """
        Share a project template with ecosystem.

        Args:
            template_name: Template name
            template_type: Type (fastapi, flask, etc.)
            description: Description
            languages: Languages used
            files: Template files

        Returns:
            Storage result
        """
        template_id = hashlib.sha256(f"{template_name}:{template_type}".encode()).hexdigest()[:16]

        template_data = {
            "template_id": template_id,
            "name": template_name,
            "template_type": template_type,
            "description": description,
            "languages": languages,
            "files": files,
            "source_project": "ucts",
            "shared_at": datetime.now().isoformat()
        }

        templates_path = self.mmb_path / "templates"
        templates_path.mkdir(parents=True, exist_ok=True)

        template_file = templates_path / f"{template_id}.json"
        with open(template_file, 'w') as f:
            json.dump(template_data, f, indent=2)

        logger.info(f"Shared template: {template_name}")

        return {
            "status": "shared",
            "template_id": template_id,
            "name": template_name
        }

    def get_ecosystem_templates(
        self,
        language: Optional[str] = None,
        limit: int = 20
    ) -> List[Dict[str, Any]]:
        """
        Get templates from ecosystem.

        Args:
            language: Filter by language
            limit: Maximum results

        Returns:
            Available templates
        """
        templates_path = self.mmb_path / "templates"

        if not templates_path.exists():
            return []

        results = []
        for template_file in templates_path.glob("*.json"):
            if template_file.name.startswith("_"):
                continue

            try:
                with open(template_file) as f:
                    data = json.load(f)

                if language and language not in data.get("languages", []):
                    continue

                results.append({
                    "template_id": data.get("template_id"),
                    "name": data.get("name"),
                    "template_type": data.get("template_type"),
                    "description": data.get("description"),
                    "languages": data.get("languages", []),
                    "source_project": data.get("source_project")
                })
            except (json.JSONDecodeError, OSError):
                continue

        return results[:limit]

    def find_similar_code(
        self,
        code_snippet: str,
        language: str,
        limit: int = 5
    ) -> List[Dict[str, Any]]:
        """
        Find similar code patterns to a given snippet.

        Args:
            code_snippet: Code to find similar patterns for
            language: Programming language
            limit: Maximum results

        Returns:
            Similar patterns
        """
        # Simple keyword-based search for now
        # In production, would use vector embeddings
        keywords = set(code_snippet.lower().split())

        patterns = self.search_patterns(language=language, limit=50)

        scored = []
        for pattern in patterns:
            pattern_data = self.get_pattern(pattern["pattern_id"])
            if not pattern_data:
                continue

            pattern_keywords = set(pattern_data.get("code", "").lower().split())
            overlap = len(keywords & pattern_keywords)

            if overlap > 0:
                scored.append({
                    "score": overlap / max(len(keywords), len(pattern_keywords)),
                    **pattern
                })

        scored.sort(key=lambda x: x["score"], reverse=True)
        return scored[:limit]

    def extract_patterns_from_session(self, session) -> List[Dict[str, Any]]:
        """
        Extract reusable patterns from a forged session.

        Args:
            session: Ingested session object

        Returns:
            Extracted patterns
        """
        extracted = []

        for block in getattr(session, 'code_blocks', []):
            code = block.get("code", "")
            language = block.get("language", "")

            if not code or len(code) < 50:  # Skip very short blocks
                continue

            # Detect pattern type
            pattern_type = "snippet"
            if "class " in code:
                pattern_type = "class"
            elif "def " in code or "function " in code or "const " in code:
                pattern_type = "function"
            elif len(code.split("\n")) > 20:
                pattern_type = "module"

            # Generate name from first meaningful line
            lines = [l.strip() for l in code.split("\n") if l.strip() and not l.strip().startswith("#")]
            name = lines[0][:50] if lines else f"pattern_{len(extracted)}"

            pattern = {
                "name": name,
                "pattern_type": pattern_type,
                "language": language,
                "code": code,
                "description": f"Extracted from session",
                "tags": ["auto-extracted"]
            }

            result = self.store_code_pattern(**pattern)
            extracted.append(result)

        return extracted
