"""
Ecosystem SDK Mode

Enable full async SDK integration with Coherence:
- Use coherence-sdk for network communication
- Real-time event broadcasting via WebSocket
- Agent-to-agent (A2A) messaging
- Virtuous cycle participation
"""

import asyncio
import json
import logging
import os
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum
from typing import Any, Callable, Dict, List, Optional, Set
from abc import ABC, abstractmethod

logger = logging.getLogger(__name__)


class SDKMode(Enum):
    """SDK operation modes"""
    FILE = "file"        # File-based integration (offline)
    NETWORK = "network"  # Network-based SDK integration
    HYBRID = "hybrid"    # Both file and network


class EventType(Enum):
    """Event types for ecosystem broadcasting"""
    FORGE_STARTED = "forge.started"
    FORGE_COMPLETED = "forge.completed"
    FORGE_FAILED = "forge.failed"
    ANALYSIS_COMPLETED = "analysis.completed"
    SECURITY_SCAN_COMPLETED = "security.scan.completed"
    SESSION_STORED = "session.stored"
    PATTERN_EXTRACTED = "pattern.extracted"
    AGENT_REGISTERED = "agent.registered"
    HEALTH_CHECK = "health.check"
    METRICS_RECORDED = "metrics.recorded"


@dataclass
class SDKConfig:
    """Configuration for SDK mode"""
    mode: SDKMode = SDKMode.FILE
    coherence_url: str = ""
    api_key: str = ""
    agent_id: str = "ucts-forge"
    reconnect_attempts: int = 3
    reconnect_delay: float = 1.0
    event_buffer_size: int = 100
    heartbeat_interval: int = 30
    enable_a2a: bool = True
    enable_virtuous_cycle: bool = True

    @classmethod
    def from_env(cls) -> "SDKConfig":
        """Create config from environment variables"""
        return cls(
            mode=SDKMode(os.getenv("UCTS_SDK_MODE", "file")),
            coherence_url=os.getenv("COHERENCE_URL", ""),
            api_key=os.getenv("COHERENCE_API_KEY", ""),
            agent_id=os.getenv("UCTS_AGENT_ID", "ucts-forge"),
        )


@dataclass
class Event:
    """An ecosystem event"""
    event_type: EventType
    source: str
    timestamp: datetime = field(default_factory=datetime.now)
    data: Dict[str, Any] = field(default_factory=dict)
    correlation_id: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        return {
            "type": self.event_type.value,
            "source": self.source,
            "timestamp": self.timestamp.isoformat(),
            "data": self.data,
            "correlation_id": self.correlation_id,
        }


@dataclass
class A2AMessage:
    """Agent-to-agent message"""
    from_agent: str
    to_agent: str
    message_type: str
    payload: Dict[str, Any]
    timestamp: datetime = field(default_factory=datetime.now)
    reply_to: Optional[str] = None
    message_id: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        return {
            "from": self.from_agent,
            "to": self.to_agent,
            "type": self.message_type,
            "payload": self.payload,
            "timestamp": self.timestamp.isoformat(),
            "reply_to": self.reply_to,
            "message_id": self.message_id,
        }


class EventHandler(ABC):
    """Abstract event handler"""

    @abstractmethod
    async def handle(self, event: Event) -> None:
        pass


class WebSocketClient:
    """WebSocket client for Coherence connection"""

    def __init__(self, config: SDKConfig):
        self.config = config
        self.ws = None
        self.connected = False
        self._message_queue: asyncio.Queue = asyncio.Queue()
        self._handlers: Dict[str, List[Callable]] = {}

    async def connect(self) -> bool:
        """Connect to Coherence WebSocket"""
        if not self.config.coherence_url:
            logger.warning("No Coherence URL configured")
            return False

        try:
            import websockets

            ws_url = self.config.coherence_url.replace("http", "ws")
            if not ws_url.endswith("/ws"):
                ws_url += "/ws"

            self.ws = await websockets.connect(
                ws_url,
                extra_headers={"Authorization": f"Bearer {self.config.api_key}"}
            )
            self.connected = True
            logger.info(f"Connected to Coherence: {ws_url}")

            # Start message handler
            asyncio.create_task(self._message_loop())

            return True
        except ImportError:
            logger.warning("websockets package not installed")
            return False
        except Exception as e:
            logger.error(f"Failed to connect: {e}")
            return False

    async def disconnect(self):
        """Disconnect from Coherence"""
        if self.ws:
            await self.ws.close()
        self.ws = None
        self.connected = False
        logger.info("Disconnected from Coherence")

    async def send(self, message: Dict[str, Any]) -> bool:
        """Send message to Coherence"""
        if not self.connected or not self.ws:
            return False

        try:
            await self.ws.send(json.dumps(message))
            return True
        except Exception as e:
            logger.error(f"Failed to send message: {e}")
            return False

    def on(self, event_type: str, handler: Callable):
        """Register event handler"""
        if event_type not in self._handlers:
            self._handlers[event_type] = []
        self._handlers[event_type].append(handler)

    async def _message_loop(self):
        """Handle incoming messages"""
        if not self.ws:
            return

        try:
            async for message in self.ws:
                try:
                    data = json.loads(message)
                    event_type = data.get("type", "unknown")

                    handlers = self._handlers.get(event_type, [])
                    handlers.extend(self._handlers.get("*", []))

                    for handler in handlers:
                        try:
                            if asyncio.iscoroutinefunction(handler):
                                await handler(data)
                            else:
                                handler(data)
                        except Exception as e:
                            logger.error(f"Handler error: {e}")

                except json.JSONDecodeError:
                    logger.warning(f"Invalid JSON: {message}")
        except Exception as e:
            logger.error(f"Message loop error: {e}")
            self.connected = False


class EcosystemSDK:
    """
    Full async SDK integration with Coherence ecosystem.

    Provides:
    - Real-time event broadcasting
    - Agent-to-agent messaging
    - Virtuous cycle participation
    - Health monitoring
    """

    def __init__(self, config: Optional[SDKConfig] = None):
        self.config = config or SDKConfig.from_env()
        self._ws_client: Optional[WebSocketClient] = None
        self._event_buffer: List[Event] = []
        self._a2a_handlers: Dict[str, Callable] = {}
        self._heartbeat_task: Optional[asyncio.Task] = None
        self._initialized = False

    async def initialize(self) -> bool:
        """
        Initialize SDK connection.

        Returns:
            True if connected successfully
        """
        if self._initialized:
            return True

        if self.config.mode == SDKMode.FILE:
            logger.info("SDK running in file mode (offline)")
            self._initialized = True
            return True

        # Network mode - connect to Coherence
        self._ws_client = WebSocketClient(self.config)
        connected = await self._ws_client.connect()

        if connected:
            # Register handlers
            self._ws_client.on("a2a.message", self._handle_a2a_message)
            self._ws_client.on("virtuous.cycle", self._handle_virtuous_cycle)

            # Start heartbeat
            self._heartbeat_task = asyncio.create_task(self._heartbeat_loop())

            # Register agent
            await self.broadcast_event(Event(
                event_type=EventType.AGENT_REGISTERED,
                source=self.config.agent_id,
                data={
                    "agent_id": self.config.agent_id,
                    "capabilities": ["forge", "analyze", "scan", "store"],
                    "version": "2.2.0",
                }
            ))

            self._initialized = True
            logger.info("SDK initialized in network mode")
            return True
        else:
            # Fallback to hybrid mode
            logger.warning("Network connection failed, falling back to hybrid mode")
            self.config.mode = SDKMode.HYBRID
            self._initialized = True
            return True

    async def shutdown(self):
        """Shutdown SDK connection"""
        if self._heartbeat_task:
            self._heartbeat_task.cancel()

        if self._ws_client:
            await self._ws_client.disconnect()

        # Flush event buffer
        await self._flush_events()

        self._initialized = False
        logger.info("SDK shutdown complete")

    async def broadcast_event(self, event: Event) -> bool:
        """
        Broadcast event to ecosystem.

        Args:
            event: Event to broadcast

        Returns:
            True if sent successfully
        """
        event.source = self.config.agent_id

        if self.config.mode == SDKMode.FILE:
            # Store in buffer for file-based sync
            self._event_buffer.append(event)
            if len(self._event_buffer) >= self.config.event_buffer_size:
                await self._flush_events()
            return True

        if self._ws_client and self._ws_client.connected:
            return await self._ws_client.send({
                "action": "broadcast",
                "event": event.to_dict(),
            })

        # Hybrid mode - buffer and try network
        self._event_buffer.append(event)
        return True

    async def send_a2a_message(
        self,
        to_agent: str,
        message_type: str,
        payload: Dict[str, Any]
    ) -> bool:
        """
        Send agent-to-agent message.

        Args:
            to_agent: Target agent ID
            message_type: Message type
            payload: Message payload

        Returns:
            True if sent successfully
        """
        if not self.config.enable_a2a:
            logger.warning("A2A messaging disabled")
            return False

        message = A2AMessage(
            from_agent=self.config.agent_id,
            to_agent=to_agent,
            message_type=message_type,
            payload=payload,
        )

        if self._ws_client and self._ws_client.connected:
            return await self._ws_client.send({
                "action": "a2a",
                "message": message.to_dict(),
            })

        logger.warning(f"Cannot send A2A message to {to_agent} - not connected")
        return False

    def on_a2a_message(self, message_type: str, handler: Callable):
        """Register A2A message handler"""
        self._a2a_handlers[message_type] = handler

    async def participate_virtuous_cycle(
        self,
        contribution_type: str,
        contribution: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Participate in Coherence virtuous cycle.

        Args:
            contribution_type: Type of contribution (pattern, knowledge, metric)
            contribution: Contribution data

        Returns:
            Cycle response with any rewards/benefits
        """
        if not self.config.enable_virtuous_cycle:
            logger.warning("Virtuous cycle participation disabled")
            return {}

        if not self._ws_client or not self._ws_client.connected:
            # Store contribution for later sync
            self._event_buffer.append(Event(
                event_type=EventType.PATTERN_EXTRACTED,
                source=self.config.agent_id,
                data={
                    "contribution_type": contribution_type,
                    "contribution": contribution,
                }
            ))
            return {"status": "buffered"}

        # Send contribution
        await self._ws_client.send({
            "action": "virtuous_cycle",
            "contribution_type": contribution_type,
            "contribution": contribution,
        })

        return {"status": "submitted"}

    async def request_from_ecosystem(
        self,
        request_type: str,
        params: Dict[str, Any]
    ) -> Optional[Dict[str, Any]]:
        """
        Request data from ecosystem.

        Args:
            request_type: Type of request (similar_patterns, health_status, etc.)
            params: Request parameters

        Returns:
            Response data or None
        """
        if not self._ws_client or not self._ws_client.connected:
            return None

        # This would use a request-response pattern in production
        # For now, return None to indicate unavailable
        return None

    async def report_forge_result(
        self,
        success: bool,
        details: Dict[str, Any]
    ):
        """Report forge operation result to ecosystem"""
        event_type = EventType.FORGE_COMPLETED if success else EventType.FORGE_FAILED

        await self.broadcast_event(Event(
            event_type=event_type,
            source=self.config.agent_id,
            data={
                "success": success,
                "timestamp": datetime.now().isoformat(),
                **details,
            }
        ))

        # Contribute to virtuous cycle if successful
        if success and self.config.enable_virtuous_cycle:
            await self.participate_virtuous_cycle(
                contribution_type="forge_completion",
                contribution={
                    "languages": details.get("languages", []),
                    "files_created": details.get("files_created", 0),
                    "patterns_used": details.get("patterns", []),
                }
            )

    async def report_health(self) -> Dict[str, Any]:
        """Report health status to ecosystem"""
        health = {
            "agent_id": self.config.agent_id,
            "status": "healthy",
            "mode": self.config.mode.value,
            "connected": self._ws_client.connected if self._ws_client else False,
            "buffered_events": len(self._event_buffer),
            "timestamp": datetime.now().isoformat(),
        }

        await self.broadcast_event(Event(
            event_type=EventType.HEALTH_CHECK,
            source=self.config.agent_id,
            data=health,
        ))

        return health

    async def _handle_a2a_message(self, data: Dict[str, Any]):
        """Handle incoming A2A message"""
        message_type = data.get("type")
        handler = self._a2a_handlers.get(message_type)

        if handler:
            try:
                if asyncio.iscoroutinefunction(handler):
                    await handler(data)
                else:
                    handler(data)
            except Exception as e:
                logger.error(f"A2A handler error: {e}")
        else:
            logger.debug(f"No handler for A2A message type: {message_type}")

    async def _handle_virtuous_cycle(self, data: Dict[str, Any]):
        """Handle virtuous cycle events"""
        cycle_type = data.get("cycle_type")
        logger.info(f"Virtuous cycle event: {cycle_type}")

        # Could receive patterns, knowledge, etc. from the cycle
        if cycle_type == "pattern_shared":
            pattern = data.get("pattern")
            logger.info(f"Received shared pattern: {pattern.get('name')}")

    async def _heartbeat_loop(self):
        """Send periodic heartbeats"""
        try:
            while True:
                await asyncio.sleep(self.config.heartbeat_interval)
                await self.report_health()
        except asyncio.CancelledError:
            pass

    async def _flush_events(self):
        """Flush buffered events to file"""
        if not self._event_buffer:
            return

        from pathlib import Path
        import json

        events_dir = Path(".ucts/events")
        events_dir.mkdir(parents=True, exist_ok=True)

        events_file = events_dir / f"events_{datetime.now().strftime('%Y%m%d%H%M%S')}.json"

        with open(events_file, 'w') as f:
            json.dump(
                [e.to_dict() for e in self._event_buffer],
                f,
                indent=2
            )

        logger.info(f"Flushed {len(self._event_buffer)} events to {events_file}")
        self._event_buffer.clear()

    def get_status(self) -> Dict[str, Any]:
        """Get SDK status"""
        return {
            "mode": self.config.mode.value,
            "initialized": self._initialized,
            "connected": self._ws_client.connected if self._ws_client else False,
            "agent_id": self.config.agent_id,
            "coherence_url": self.config.coherence_url or "Not configured",
            "a2a_enabled": self.config.enable_a2a,
            "virtuous_cycle_enabled": self.config.enable_virtuous_cycle,
            "buffered_events": len(self._event_buffer),
        }


# Singleton instance
_sdk: Optional[EcosystemSDK] = None


def get_ecosystem_sdk(config: Optional[SDKConfig] = None) -> EcosystemSDK:
    """Get the global ecosystem SDK instance"""
    global _sdk
    if _sdk is None or config is not None:
        _sdk = EcosystemSDK(config)
    return _sdk


async def initialize_sdk(config: Optional[SDKConfig] = None) -> EcosystemSDK:
    """Initialize and return the ecosystem SDK"""
    sdk = get_ecosystem_sdk(config)
    await sdk.initialize()
    return sdk
