"""
Sentinel Protocol Integration

Integration with Sentinel Protocol for:
- Constitutional AI policy synchronization
- Pre-forge security validation
- Code pattern security scanning
- Policy violation alerting
"""
import json
import logging
import re
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime
from enum import Enum
from dataclasses import dataclass, field

# Configure logging
logger = logging.getLogger(__name__)


class PolicyDecision(Enum):
    """Policy validation decision"""
    APPROVED = "approved"
    DENIED = "denied"
    REQUIRES_REVIEW = "requires_review"


class ViolationSeverity(Enum):
    """Policy violation severity"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


@dataclass
class SecurityPattern:
    """Pattern for security scanning"""
    pattern_id: str
    name: str
    regex: str
    severity: ViolationSeverity
    description: str
    recommendation: str


@dataclass
class SecurityScanResult:
    """Result of security scan"""
    passed: bool
    violations: List[Dict[str, Any]] = field(default_factory=list)
    warnings: List[Dict[str, Any]] = field(default_factory=list)
    scanned_files: int = 0
    scan_duration_ms: int = 0


class SentinelError(Exception):
    """Base exception for Sentinel integration errors"""
    pass


class SentinelNotFoundError(SentinelError):
    """Sentinel project or files not found"""
    pass


class SentinelStateError(SentinelError):
    """Error reading or writing Sentinel state"""
    pass


class SentinelModuleError(SentinelError):
    """Error creating Sentinel module"""
    pass


class SentinelIntegration:
    """Integration with Sentinel Protocol for insurance vertical"""

    def __init__(self, sentinel_path: Optional[str] = None):
        """
        Initialize Sentinel integration.

        Args:
            sentinel_path: Path to Sentinel project root
        """
        if sentinel_path:
            self.sentinel_path = Path(sentinel_path)
            logger.debug(f"Using provided Sentinel path: {self.sentinel_path}")
        else:
            # Try to find Sentinel project
            possible_paths = [
                Path.cwd(),
                Path.cwd().parent / "sentinel",
                Path.home() / "sentinel",
            ]
            self.sentinel_path = next(
                (p for p in possible_paths if (p / "execution").exists()),
                Path.cwd()
            )
            logger.debug(f"Auto-detected Sentinel path: {self.sentinel_path}")

    def _validate_sentinel_path(self) -> bool:
        """Check if Sentinel path appears valid."""
        execution_dir = self.sentinel_path / "execution"
        return execution_dir.exists()

    def capture_execution_state(self) -> Dict[str, Any]:
        """
        Capture current Sentinel execution state.

        Returns:
            Current execution state dictionary

        Raises:
            SentinelStateError: If state cannot be read
        """
        state_file = self.sentinel_path / "execution" / "execution_state.json"

        if not state_file.exists():
            logger.info(f"Sentinel execution state not found at {state_file}")
            return {
                "status": "not_found",
                "message": "Sentinel execution state not found",
                "path": str(state_file)
            }

        try:
            with open(state_file, 'r', encoding='utf-8') as f:
                execution_state = json.load(f)
        except json.JSONDecodeError as e:
            raise SentinelStateError(f"Invalid JSON in execution state: {e}")
        except PermissionError as e:
            raise SentinelStateError(f"Permission denied reading execution state: {e}")
        except OSError as e:
            raise SentinelStateError(f"Failed to read execution state: {e}")

        # Extract key information safely
        tasks = execution_state.get("tasks", {})
        if not isinstance(tasks, dict):
            tasks = {}

        agent_registry = execution_state.get("agent_registry", {})
        if not isinstance(agent_registry, dict):
            agent_registry = {}

        ready_list = execution_state.get("ready", [])
        if not isinstance(ready_list, list):
            ready_list = []

        return {
            "status": "captured",
            "captured_at": datetime.now().isoformat(),
            "phase": execution_state.get("current_phase", "unknown"),
            "tasks": {
                "total": len(tasks),
                "completed": sum(
                    1 for t in tasks.values()
                    if isinstance(t, dict) and t.get("status") == "completed"
                ),
            },
            "agents": list(agent_registry.keys()),
            "ready_for_work": len(ready_list) > 0,
        }

    def get_agent_registry(self) -> List[Dict[str, Any]]:
        """
        Get current Sentinel agent registry.

        Returns:
            List of registered agents

        Raises:
            SentinelStateError: If registry cannot be read
        """
        agents_file = self.sentinel_path / "agents" / "registry" / "agents.json"

        if not agents_file.exists():
            logger.debug(f"Agent registry not found at {agents_file}")
            return []

        try:
            with open(agents_file, 'r', encoding='utf-8') as f:
                agents_data = json.load(f)
        except json.JSONDecodeError as e:
            raise SentinelStateError(f"Invalid JSON in agent registry: {e}")
        except PermissionError as e:
            raise SentinelStateError(f"Permission denied reading agent registry: {e}")
        except OSError as e:
            raise SentinelStateError(f"Failed to read agent registry: {e}")

        if not isinstance(agents_data, dict):
            logger.warning("Agent registry is not a dictionary")
            return []

        agents_list = agents_data.get("agents", [])
        if not isinstance(agents_list, list):
            logger.warning("Agents list is not an array")
            return []

        return [
            {
                "id": agent.get("id"),
                "type": agent.get("type"),
                "status": agent.get("status"),
                "capabilities": agent.get("capabilities", []),
            }
            for agent in agents_list
            if isinstance(agent, dict)
        ]

    def forge_sentinel_module(self, session, module_type: str = "generic") -> Dict[str, Any]:
        """
        Create new Sentinel module from conversation session.

        Args:
            session: Ingested session object
            module_type: Type of module (agent, engine, integration)

        Returns:
            Module generation result

        Raises:
            ValueError: If session is None or module_type is invalid
            SentinelModuleError: If module cannot be created
        """
        # Validate inputs
        if session is None:
            raise ValueError("session cannot be None")

        valid_module_types = ("agent", "engine", "integration", "generic")
        if module_type not in valid_module_types:
            logger.warning(f"Non-standard module type: {module_type}")

        try:
            from ucts.analysis import AnalysisEngine
            from ucts.generators import VSCodeGenerator
        except ImportError as e:
            raise SentinelModuleError(f"Failed to import UCTS modules: {e}")

        # Analyze session for Sentinel-specific patterns
        try:
            engine = AnalysisEngine()
            structure = engine.analyze(session)
        except Exception as e:
            raise SentinelModuleError(f"Failed to analyze session: {e}")

        if not structure or not structure.name:
            raise SentinelModuleError("Analysis produced invalid structure")

        # Customize for Sentinel
        original_name = structure.name
        structure.name = f"sentinel-{structure.name}"

        # Add Sentinel-specific files
        sentinel_config = {
            "module_type": module_type,
            "version": "1.0.0",
            "created_from": "ucts-forge",
            "original_name": original_name,
            "created_at": datetime.now().isoformat(),
            "integration": {
                "coherence": True,
                "meaning_engine": module_type == "engine",
            }
        }

        try:
            structure.files["sentinel_config.json"] = json.dumps(sentinel_config, indent=2)
        except (TypeError, ValueError) as e:
            raise SentinelModuleError(f"Failed to create sentinel config: {e}")

        # Generate the module
        output_path = self.sentinel_path / "modules" / structure.name

        try:
            output_path.parent.mkdir(parents=True, exist_ok=True)
            generator = VSCodeGenerator()
            result_path = generator.generate(structure, str(output_path))
        except PermissionError as e:
            raise SentinelModuleError(f"Permission denied creating module at {output_path}: {e}")
        except OSError as e:
            raise SentinelModuleError(f"Failed to create module directory: {e}")
        except Exception as e:
            raise SentinelModuleError(f"Failed to generate module: {e}")

        logger.info(f"Created Sentinel module '{structure.name}' at {result_path}")

        return {
            "status": "created",
            "module_name": structure.name,
            "module_type": module_type,
            "path": str(result_path),
            "files": list(structure.files.keys()),
        }

    def sync_with_coherence(self, ucts_state: Dict[str, Any]) -> Dict[str, Any]:
        """
        Synchronize UCTS state with Sentinel's Coherence integration.

        Args:
            ucts_state: Current UCTS state

        Returns:
            Sync result

        Raises:
            ValueError: If ucts_state is not a dictionary
            SentinelStateError: If sync fails
        """
        if not isinstance(ucts_state, dict):
            raise ValueError("ucts_state must be a dictionary")

        coherence_file = self.sentinel_path / "execution" / "coherence_state.json"

        # Read existing Coherence state
        existing: Dict[str, Any] = {}
        if coherence_file.exists():
            try:
                with open(coherence_file, 'r', encoding='utf-8') as f:
                    existing = json.load(f)
                if not isinstance(existing, dict):
                    logger.warning("Coherence state is not a dictionary, resetting")
                    existing = {}
            except json.JSONDecodeError as e:
                logger.warning(f"Invalid JSON in coherence state, resetting: {e}")
                existing = {}
            except OSError as e:
                logger.warning(f"Could not read coherence state: {e}")

        # Update with UCTS state
        sync_time = datetime.now().isoformat()
        existing["ucts"] = {
            "synced_at": sync_time,
            "status": ucts_state.get("status", "unknown"),
            "capabilities": [
                "session_ingestion",
                "project_generation",
                "context_transfer",
            ],
            "available_actions": ucts_state.get("available_actions", []),
        }

        # Write back
        try:
            coherence_file.parent.mkdir(parents=True, exist_ok=True)
            with open(coherence_file, 'w', encoding='utf-8') as f:
                json.dump(existing, f, indent=2)
            logger.info(f"Synced UCTS state with Coherence at {coherence_file}")
        except PermissionError as e:
            raise SentinelStateError(f"Permission denied writing coherence state: {e}")
        except OSError as e:
            raise SentinelStateError(f"Failed to write coherence state: {e}")

        return {
            "status": "synced",
            "synced_at": sync_time,
            "path": str(coherence_file)
        }

    def get_available_modules(self) -> List[Dict[str, Any]]:
        """
        Get list of available Sentinel modules.

        Returns:
            List of module information dictionaries
        """
        modules_dir = self.sentinel_path / "modules"

        if not modules_dir.exists():
            return []

        modules = []
        try:
            for entry in modules_dir.iterdir():
                if entry.is_dir():
                    config_file = entry / "sentinel_config.json"
                    module_info: Dict[str, Any] = {
                        "name": entry.name,
                        "path": str(entry),
                        "type": "unknown",
                        "version": "unknown",
                    }

                    if config_file.exists():
                        try:
                            with open(config_file, 'r', encoding='utf-8') as f:
                                config = json.load(f)
                            module_info["type"] = config.get("module_type", "unknown")
                            module_info["version"] = config.get("version", "unknown")
                        except (json.JSONDecodeError, OSError):
                            pass  # Keep defaults

                    modules.append(module_info)
        except OSError as e:
            logger.warning(f"Could not list modules: {e}")

        return modules

    # Security Scanning Methods

    # Default security patterns
    SECURITY_PATTERNS = [
        SecurityPattern(
            "hardcoded-secret",
            "Hardcoded Secret",
            r'(?i)(password|secret|api_key|apikey|token|auth)\s*[=:]\s*["\'][^"\']{8,}["\']',
            ViolationSeverity.CRITICAL,
            "Hardcoded secrets detected in code",
            "Use environment variables or a secrets manager"
        ),
        SecurityPattern(
            "aws-key",
            "AWS Access Key",
            r'AKIA[0-9A-Z]{16}',
            ViolationSeverity.CRITICAL,
            "AWS access key detected",
            "Remove and rotate the AWS key immediately"
        ),
        SecurityPattern(
            "private-key",
            "Private Key",
            r'-----BEGIN (?:RSA |EC |DSA |OPENSSH )?PRIVATE KEY-----',
            ViolationSeverity.CRITICAL,
            "Private key detected in code",
            "Never commit private keys to code"
        ),
        SecurityPattern(
            "sql-injection",
            "Potential SQL Injection",
            r'(?i)execute\s*\(\s*["\']?\s*SELECT.*\+|f["\']SELECT.*\{|\.format\s*\(.*SELECT',
            ViolationSeverity.HIGH,
            "Potential SQL injection vulnerability",
            "Use parameterized queries instead of string formatting"
        ),
        SecurityPattern(
            "eval-usage",
            "Dangerous eval() Usage",
            r'\beval\s*\([^)]*\)',
            ViolationSeverity.HIGH,
            "Use of eval() can execute arbitrary code",
            "Avoid eval() or use safer alternatives like ast.literal_eval"
        ),
        SecurityPattern(
            "shell-injection",
            "Potential Shell Injection",
            r'(?i)(?:subprocess\.call|os\.system|os\.popen)\s*\([^)]*\+|shell\s*=\s*True',
            ViolationSeverity.HIGH,
            "Potential shell injection vulnerability",
            "Use subprocess with shell=False and list arguments"
        ),
        SecurityPattern(
            "debug-enabled",
            "Debug Mode Enabled",
            r'(?i)DEBUG\s*=\s*True|\.run\s*\([^)]*debug\s*=\s*True',
            ViolationSeverity.MEDIUM,
            "Debug mode should be disabled in production",
            "Set DEBUG=False for production deployments"
        ),
        SecurityPattern(
            "weak-crypto",
            "Weak Cryptography",
            r'(?i)\b(?:md5|sha1)\s*\(|DES\b|RC4\b',
            ViolationSeverity.MEDIUM,
            "Weak cryptographic algorithm detected",
            "Use SHA-256 or stronger algorithms"
        ),
        SecurityPattern(
            "cors-wildcard",
            "CORS Wildcard",
            r'(?i)(?:Access-Control-Allow-Origin|CORS_ORIGINS?)\s*[=:]\s*["\']?\*',
            ViolationSeverity.MEDIUM,
            "CORS allows all origins",
            "Restrict CORS to specific trusted domains"
        ),
        SecurityPattern(
            "insecure-random",
            "Insecure Random",
            r'\brandom\.random\s*\(|\brandom\.randint\s*\(',
            ViolationSeverity.LOW,
            "Using non-cryptographic random for potentially sensitive data",
            "Use secrets module for security-sensitive randomness"
        ),
    ]

    def scan_code_security(
        self,
        code: str,
        filename: Optional[str] = None,
        language: Optional[str] = None
    ) -> SecurityScanResult:
        """
        Scan code for security vulnerabilities.

        Args:
            code: Code content to scan
            filename: Optional filename for context
            language: Optional language for targeted scanning

        Returns:
            SecurityScanResult with violations and warnings
        """
        import time
        start_time = time.time()

        violations = []
        warnings = []

        for pattern in self.SECURITY_PATTERNS:
            matches = list(re.finditer(pattern.regex, code))

            for match in matches:
                # Find line number
                line_num = code[:match.start()].count('\n') + 1
                line_content = code.split('\n')[line_num - 1] if line_num <= len(code.split('\n')) else ""

                finding = {
                    "pattern_id": pattern.pattern_id,
                    "name": pattern.name,
                    "severity": pattern.severity.value,
                    "description": pattern.description,
                    "recommendation": pattern.recommendation,
                    "line": line_num,
                    "line_content": line_content[:100],  # Truncate
                    "filename": filename
                }

                if pattern.severity in [ViolationSeverity.CRITICAL, ViolationSeverity.HIGH]:
                    violations.append(finding)
                else:
                    warnings.append(finding)

        duration_ms = int((time.time() - start_time) * 1000)

        passed = len(violations) == 0

        return SecurityScanResult(
            passed=passed,
            violations=violations,
            warnings=warnings,
            scanned_files=1,
            scan_duration_ms=duration_ms
        )

    def scan_project_security(
        self,
        project_files: Dict[str, str]
    ) -> SecurityScanResult:
        """
        Scan entire project for security vulnerabilities.

        Args:
            project_files: Dictionary of filename -> content

        Returns:
            Aggregated SecurityScanResult
        """
        import time
        start_time = time.time()

        all_violations = []
        all_warnings = []

        for filename, content in project_files.items():
            result = self.scan_code_security(content, filename)
            all_violations.extend(result.violations)
            all_warnings.extend(result.warnings)

        duration_ms = int((time.time() - start_time) * 1000)

        passed = len(all_violations) == 0

        return SecurityScanResult(
            passed=passed,
            violations=all_violations,
            warnings=all_warnings,
            scanned_files=len(project_files),
            scan_duration_ms=duration_ms
        )

    def validate_forge_policy(
        self,
        languages: List[str],
        has_auth_patterns: bool = False,
        has_db_operations: bool = False,
        has_api_keys: bool = False,
        estimated_complexity: str = "medium"
    ) -> Dict[str, Any]:
        """
        Validate forge operation against policies.

        Args:
            languages: Languages being generated
            has_auth_patterns: Whether auth code is present
            has_db_operations: Whether database code is present
            has_api_keys: Whether API key handling is present
            estimated_complexity: Complexity level

        Returns:
            Policy validation result
        """
        decision = PolicyDecision.APPROVED
        reasons = []
        requirements = []

        # Check for sensitive patterns
        if has_api_keys:
            reasons.append("API key handling detected - ensure secure storage")
            requirements.append("Use environment variables for API keys")

        if has_auth_patterns:
            reasons.append("Authentication code detected")
            requirements.append("Review auth implementation for security best practices")
            if estimated_complexity == "high":
                decision = PolicyDecision.REQUIRES_REVIEW

        if has_db_operations:
            reasons.append("Database operations detected")
            requirements.append("Ensure parameterized queries are used")

        # Check for high-risk languages
        high_risk_langs = {"php", "perl", "shell", "bash"}
        if any(lang.lower() in high_risk_langs for lang in languages):
            reasons.append("High-risk language detected")
            requirements.append("Extra security review recommended")

        result = {
            "decision": decision.value,
            "approved": decision == PolicyDecision.APPROVED,
            "reasons": reasons,
            "requirements": requirements,
            "languages": languages,
            "timestamp": datetime.now().isoformat()
        }

        # Log the validation
        self._log_policy_validation(result)

        return result

    def _log_policy_validation(self, result: Dict[str, Any]):
        """Log policy validation to audit file"""
        audit_path = self.sentinel_path / "execution" / "policy_audit.json"

        try:
            audit_path.parent.mkdir(parents=True, exist_ok=True)

            audits = []
            if audit_path.exists():
                try:
                    with open(audit_path) as f:
                        audits = json.load(f)
                except (json.JSONDecodeError, OSError):
                    audits = []

            audits.append(result)
            audits = audits[-500:]  # Keep last 500

            with open(audit_path, 'w') as f:
                json.dump(audits, f, indent=2)
        except OSError as e:
            logger.warning(f"Failed to log policy validation: {e}")

    def report_violation(
        self,
        violation_type: str,
        severity: ViolationSeverity,
        details: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Report a policy violation.

        Args:
            violation_type: Type of violation
            severity: Violation severity
            details: Violation details

        Returns:
            Report result
        """
        violation = {
            "violation_id": f"ucts-{datetime.now().strftime('%Y%m%d%H%M%S%f')}",
            "type": violation_type,
            "severity": severity.value,
            "source": "ucts",
            "details": details,
            "timestamp": datetime.now().isoformat()
        }

        violations_path = self.sentinel_path / "execution" / "violations.json"

        try:
            violations_path.parent.mkdir(parents=True, exist_ok=True)

            violations = []
            if violations_path.exists():
                try:
                    with open(violations_path) as f:
                        violations = json.load(f)
                except (json.JSONDecodeError, OSError):
                    violations = []

            violations.append(violation)

            with open(violations_path, 'w') as f:
                json.dump(violations, f, indent=2)

            logger.warning(f"Reported violation: {violation_type} ({severity.value})")
        except OSError as e:
            logger.error(f"Failed to report violation: {e}")

        return violation
