"""
WB2-Platform Dashboard Integration

Provides:
- UCTS metrics for dashboard display
- Activity feed events
- Status widgets data
- Real-time updates
"""

import json
import logging
from pathlib import Path
from typing import Dict, Any, Optional, List
from datetime import datetime, timedelta
from dataclasses import dataclass, field

logger = logging.getLogger(__name__)


@dataclass
class ForgeMetrics:
    """Metrics for forge operations"""
    total_forges: int = 0
    successful_forges: int = 0
    failed_forges: int = 0
    total_files_created: int = 0
    languages_used: Dict[str, int] = field(default_factory=dict)
    targets_used: Dict[str, int] = field(default_factory=dict)
    avg_duration_seconds: float = 0.0


@dataclass
class ActivityEvent:
    """Activity feed event"""
    event_id: str
    event_type: str
    title: str
    description: str
    timestamp: datetime
    metadata: Dict[str, Any] = field(default_factory=dict)


class WB2Integration:
    """
    Integration with WB2-Platform dashboard.

    Provides data for:
    - UCTS activity widget
    - Forge statistics charts
    - Recent activity feed
    - Status indicators
    """

    def __init__(self, config=None):
        from ucts.integrations.ecosystem import EcosystemConfig
        self.config = config or EcosystemConfig()
        self._metrics_cache: Optional[ForgeMetrics] = None

    @property
    def base_path(self) -> Path:
        return Path(self.config.base_path)

    @property
    def wb2_path(self) -> Path:
        return self.base_path / "wb2"

    def health_check(self) -> Dict[str, Any]:
        """Perform health check"""
        return {
            "status": "healthy",
            "project": "wb2",
            "timestamp": datetime.now().isoformat()
        }

    def record_forge_activity(
        self,
        session_id: str,
        success: bool,
        files_created: int,
        languages: List[str],
        target_type: str,
        duration_seconds: float,
        project_name: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Record forge activity for dashboard.

        Args:
            session_id: Session identifier
            success: Whether forge succeeded
            files_created: Number of files created
            languages: Languages detected
            target_type: Target platform
            duration_seconds: Operation duration
            project_name: Optional project name

        Returns:
            Activity record
        """
        activity = {
            "activity_id": f"forge-{datetime.now().strftime('%Y%m%d%H%M%S%f')}",
            "activity_type": "forge",
            "session_id": session_id,
            "success": success,
            "files_created": files_created,
            "languages": languages,
            "target_type": target_type,
            "duration_seconds": duration_seconds,
            "project_name": project_name,
            "timestamp": datetime.now().isoformat()
        }

        # Append to activity log
        activity_path = self.wb2_path / "activity.json"
        activity_path.parent.mkdir(parents=True, exist_ok=True)

        activities = []
        if activity_path.exists():
            try:
                with open(activity_path) as f:
                    activities = json.load(f)
            except (json.JSONDecodeError, OSError):
                activities = []

        activities.append(activity)
        activities = activities[-500:]  # Keep last 500

        with open(activity_path, 'w') as f:
            json.dump(activities, f, indent=2)

        # Update metrics
        self._update_metrics(activity)

        # Create activity feed event
        self._create_feed_event(
            event_type="forge_complete" if success else "forge_failed",
            title=f"Project forged: {project_name or session_id}",
            description=f"Created {files_created} files in {languages}",
            metadata=activity
        )

        return activity

    def _update_metrics(self, activity: Dict[str, Any]):
        """Update aggregate metrics"""
        metrics_path = self.wb2_path / "metrics.json"

        metrics = {
            "total_forges": 0,
            "successful_forges": 0,
            "failed_forges": 0,
            "total_files_created": 0,
            "languages_used": {},
            "targets_used": {},
            "total_duration": 0.0,
            "updated_at": datetime.now().isoformat()
        }

        if metrics_path.exists():
            try:
                with open(metrics_path) as f:
                    metrics = json.load(f)
            except (json.JSONDecodeError, OSError):
                pass

        # Update counts
        metrics["total_forges"] = metrics.get("total_forges", 0) + 1

        if activity.get("success"):
            metrics["successful_forges"] = metrics.get("successful_forges", 0) + 1
        else:
            metrics["failed_forges"] = metrics.get("failed_forges", 0) + 1

        metrics["total_files_created"] = metrics.get("total_files_created", 0) + activity.get("files_created", 0)
        metrics["total_duration"] = metrics.get("total_duration", 0) + activity.get("duration_seconds", 0)

        # Update language counts
        langs = metrics.get("languages_used", {})
        for lang in activity.get("languages", []):
            langs[lang] = langs.get(lang, 0) + 1
        metrics["languages_used"] = langs

        # Update target counts
        targets = metrics.get("targets_used", {})
        target = activity.get("target_type", "unknown")
        targets[target] = targets.get(target, 0) + 1
        metrics["targets_used"] = targets

        metrics["updated_at"] = datetime.now().isoformat()

        with open(metrics_path, 'w') as f:
            json.dump(metrics, f, indent=2)

    def _create_feed_event(
        self,
        event_type: str,
        title: str,
        description: str,
        metadata: Optional[Dict[str, Any]] = None
    ):
        """Create activity feed event"""
        event = {
            "event_id": f"ucts-{datetime.now().strftime('%Y%m%d%H%M%S%f')}",
            "event_type": event_type,
            "source": "ucts",
            "title": title,
            "description": description,
            "metadata": metadata or {},
            "timestamp": datetime.now().isoformat()
        }

        feed_path = self.wb2_path / "feed.json"
        feed_path.parent.mkdir(parents=True, exist_ok=True)

        feed = []
        if feed_path.exists():
            try:
                with open(feed_path) as f:
                    feed = json.load(f)
            except (json.JSONDecodeError, OSError):
                feed = []

        feed.append(event)
        feed = feed[-100:]  # Keep last 100

        with open(feed_path, 'w') as f:
            json.dump(feed, f, indent=2)

    def get_dashboard_data(self) -> Dict[str, Any]:
        """
        Get all dashboard data for WB2.

        Returns:
            Complete dashboard data
        """
        return {
            "metrics": self.get_metrics(),
            "recent_activity": self.get_recent_activity(limit=10),
            "status": self.get_status(),
            "charts": self.get_chart_data(),
            "timestamp": datetime.now().isoformat()
        }

    def get_metrics(self) -> Dict[str, Any]:
        """Get aggregate metrics"""
        metrics_path = self.wb2_path / "metrics.json"

        if not metrics_path.exists():
            return {
                "total_forges": 0,
                "successful_forges": 0,
                "failed_forges": 0,
                "success_rate": 0,
                "total_files_created": 0,
                "avg_duration_seconds": 0
            }

        try:
            with open(metrics_path) as f:
                metrics = json.load(f)

            total = metrics.get("total_forges", 0)
            successful = metrics.get("successful_forges", 0)

            return {
                "total_forges": total,
                "successful_forges": successful,
                "failed_forges": metrics.get("failed_forges", 0),
                "success_rate": (successful / total * 100) if total > 0 else 0,
                "total_files_created": metrics.get("total_files_created", 0),
                "avg_duration_seconds": metrics.get("total_duration", 0) / total if total > 0 else 0,
                "languages_used": metrics.get("languages_used", {}),
                "targets_used": metrics.get("targets_used", {})
            }
        except (json.JSONDecodeError, OSError):
            return {}

    def get_recent_activity(self, limit: int = 20) -> List[Dict[str, Any]]:
        """Get recent activity feed"""
        feed_path = self.wb2_path / "feed.json"

        if not feed_path.exists():
            return []

        try:
            with open(feed_path) as f:
                feed = json.load(f)
            return feed[-limit:][::-1]  # Newest first
        except (json.JSONDecodeError, OSError):
            return []

    def get_status(self) -> Dict[str, Any]:
        """Get UCTS status for dashboard"""
        metrics = self.get_metrics()
        recent = self.get_recent_activity(limit=10)

        # Determine health based on recent success rate
        recent_successes = sum(1 for a in recent if a.get("metadata", {}).get("success", True))
        recent_rate = recent_successes / len(recent) if recent else 1.0

        if recent_rate >= 0.9:
            status = "healthy"
        elif recent_rate >= 0.7:
            status = "degraded"
        else:
            status = "unhealthy"

        return {
            "status": status,
            "version": "2.0.0",
            "total_forges": metrics.get("total_forges", 0),
            "success_rate": metrics.get("success_rate", 0),
            "last_activity": recent[0]["timestamp"] if recent else None
        }

    def get_chart_data(self) -> Dict[str, Any]:
        """Get data for dashboard charts"""
        activity_path = self.wb2_path / "activity.json"

        if not activity_path.exists():
            return {"daily": [], "by_language": [], "by_target": []}

        try:
            with open(activity_path) as f:
                activities = json.load(f)
        except (json.JSONDecodeError, OSError):
            return {"daily": [], "by_language": [], "by_target": []}

        # Daily activity (last 7 days)
        daily = {}
        cutoff = datetime.now() - timedelta(days=7)

        for activity in activities:
            try:
                ts = datetime.fromisoformat(activity["timestamp"].replace("Z", "+00:00"))
                if ts.replace(tzinfo=None) < cutoff:
                    continue

                date_key = ts.strftime("%Y-%m-%d")
                if date_key not in daily:
                    daily[date_key] = {"date": date_key, "total": 0, "success": 0}

                daily[date_key]["total"] += 1
                if activity.get("success"):
                    daily[date_key]["success"] += 1
            except (ValueError, KeyError):
                continue

        # Sort by date
        daily_list = sorted(daily.values(), key=lambda x: x["date"])

        # Language distribution
        metrics = self.get_metrics()
        by_language = [
            {"language": lang, "count": count}
            for lang, count in metrics.get("languages_used", {}).items()
        ]
        by_language.sort(key=lambda x: x["count"], reverse=True)

        # Target distribution
        by_target = [
            {"target": target, "count": count}
            for target, count in metrics.get("targets_used", {}).items()
        ]
        by_target.sort(key=lambda x: x["count"], reverse=True)

        return {
            "daily": daily_list,
            "by_language": by_language[:10],
            "by_target": by_target
        }

    def get_widget_data(self, widget_type: str) -> Dict[str, Any]:
        """
        Get data for specific widget type.

        Args:
            widget_type: Widget type (status, metrics, activity, chart)

        Returns:
            Widget-specific data
        """
        if widget_type == "status":
            return self.get_status()
        elif widget_type == "metrics":
            return self.get_metrics()
        elif widget_type == "activity":
            return {"events": self.get_recent_activity(limit=5)}
        elif widget_type == "chart":
            return self.get_chart_data()
        else:
            return {"error": f"Unknown widget type: {widget_type}"}

    def send_activity(
        self,
        activity_type: str,
        project: str,
        data: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Send activity event to WB2 dashboard.

        Args:
            activity_type: Type of activity
            project: Source project
            data: Activity data

        Returns:
            Activity send result
        """
        self._create_feed_event(
            event_type=activity_type,
            title=f"{project}: {activity_type}",
            description=data.get("description", ""),
            metadata=data
        )

        return {
            "status": "sent",
            "activity_type": activity_type,
            "project": project
        }
