"""
Plugin/Extension System

Allow community extensions:
- Custom ingesters for new platforms
- Custom generators (new frameworks)
- Custom analyzers (domain-specific)
"""

import importlib
import importlib.util
import json
import logging
import os
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Set, Type

logger = logging.getLogger(__name__)


class PluginType(Enum):
    """Types of plugins"""
    INGESTER = "ingester"       # New platform ingesters
    GENERATOR = "generator"     # New framework generators
    ANALYZER = "analyzer"       # Domain-specific analyzers
    HOOK = "hook"               # Pipeline hooks
    FORMATTER = "formatter"     # Output formatters
    TRANSFORMER = "transformer" # Code transformers


class HookType(Enum):
    """Hook points in the UCTS pipeline"""
    PRE_INGEST = "pre_ingest"
    POST_INGEST = "post_ingest"
    PRE_ANALYZE = "pre_analyze"
    POST_ANALYZE = "post_analyze"
    PRE_GENERATE = "pre_generate"
    POST_GENERATE = "post_generate"
    PRE_FORGE = "pre_forge"
    POST_FORGE = "post_forge"


@dataclass
class PluginConfig:
    """Configuration for a plugin"""
    name: str
    version: str
    description: str = ""
    author: str = ""
    plugin_type: PluginType = PluginType.HOOK
    dependencies: List[str] = field(default_factory=list)
    config: Dict[str, Any] = field(default_factory=dict)
    enabled: bool = True


class Plugin(ABC):
    """
    Base class for UCTS plugins.

    Extend this to create custom plugins for:
    - Ingesters: Support new conversation formats
    - Generators: Generate for new frameworks
    - Analyzers: Domain-specific analysis
    - Hooks: Custom pipeline processing
    """

    def __init__(self, config: Optional[PluginConfig] = None):
        self.config = config or self.default_config()
        self._initialized = False

    @classmethod
    @abstractmethod
    def default_config(cls) -> PluginConfig:
        """Return default plugin configuration"""
        pass

    @abstractmethod
    def initialize(self) -> bool:
        """Initialize the plugin. Return True if successful."""
        pass

    def shutdown(self):
        """Clean up plugin resources"""
        pass

    @property
    def name(self) -> str:
        return self.config.name

    @property
    def version(self) -> str:
        return self.config.version

    @property
    def plugin_type(self) -> PluginType:
        return self.config.plugin_type


class IngesterPlugin(Plugin):
    """Base class for ingester plugins"""

    @abstractmethod
    def can_ingest(self, source: str) -> bool:
        """Check if this ingester can handle the source"""
        pass

    @abstractmethod
    def ingest(self, source: str) -> "Session":
        """Ingest the source and return a Session"""
        pass


class GeneratorPlugin(Plugin):
    """Base class for generator plugins"""

    @abstractmethod
    def can_generate(self, structure: "ProjectStructure", target: str) -> bool:
        """Check if this generator can handle the target"""
        pass

    @abstractmethod
    def generate(self, structure: "ProjectStructure", output: str) -> Dict[str, str]:
        """Generate project files"""
        pass


class AnalyzerPlugin(Plugin):
    """Base class for analyzer plugins"""

    @abstractmethod
    def analyze(self, session: "Session") -> Dict[str, Any]:
        """Perform domain-specific analysis"""
        pass


class HookPlugin(Plugin):
    """Base class for hook plugins"""

    @property
    @abstractmethod
    def hook_type(self) -> HookType:
        """Return the hook type"""
        pass

    @abstractmethod
    def execute(self, context: Dict[str, Any]) -> Dict[str, Any]:
        """Execute the hook with given context"""
        pass


@dataclass
class PluginMetadata:
    """Metadata for a loaded plugin"""
    name: str
    version: str
    plugin_type: PluginType
    path: str
    class_name: str
    enabled: bool = True
    loaded_at: datetime = field(default_factory=datetime.now)
    instance: Optional[Plugin] = None


class PluginManager:
    """
    Manages UCTS plugins.

    - Discovers plugins from directories
    - Loads and initializes plugins
    - Provides plugin registry
    - Executes hooks in order
    """

    def __init__(self, plugin_dirs: Optional[List[str]] = None):
        self.plugin_dirs = plugin_dirs or self._default_plugin_dirs()
        self._plugins: Dict[str, PluginMetadata] = {}
        self._hooks: Dict[HookType, List[HookPlugin]] = {h: [] for h in HookType}
        self._ingesters: List[IngesterPlugin] = []
        self._generators: List[GeneratorPlugin] = []
        self._analyzers: List[AnalyzerPlugin] = []

    def _default_plugin_dirs(self) -> List[str]:
        """Get default plugin directories"""
        dirs = []

        # User plugin dir
        home = os.path.expanduser("~")
        user_plugin_dir = os.path.join(home, ".ucts", "plugins")
        dirs.append(user_plugin_dir)

        # Local plugin dir (for development)
        local_plugin_dir = os.path.join(os.getcwd(), ".ucts", "plugins")
        if os.path.exists(local_plugin_dir):
            dirs.append(local_plugin_dir)

        return dirs

    def discover_plugins(self) -> List[PluginMetadata]:
        """Discover plugins from plugin directories"""
        discovered = []

        for plugin_dir in self.plugin_dirs:
            if not os.path.isdir(plugin_dir):
                continue

            # Look for plugin.json files
            for item in os.listdir(plugin_dir):
                plugin_path = os.path.join(plugin_dir, item)

                if os.path.isdir(plugin_path):
                    manifest = os.path.join(plugin_path, "plugin.json")
                    if os.path.exists(manifest):
                        try:
                            metadata = self._load_manifest(manifest)
                            metadata.path = plugin_path
                            discovered.append(metadata)
                        except Exception as e:
                            logger.warning(f"Failed to load plugin manifest {manifest}: {e}")

                elif item.endswith(".py") and not item.startswith("_"):
                    # Single-file plugins
                    try:
                        metadata = self._discover_single_file(plugin_path)
                        if metadata:
                            discovered.append(metadata)
                    except Exception as e:
                        logger.warning(f"Failed to discover plugin {plugin_path}: {e}")

        return discovered

    def _load_manifest(self, manifest_path: str) -> PluginMetadata:
        """Load plugin metadata from manifest"""
        with open(manifest_path) as f:
            data = json.load(f)

        return PluginMetadata(
            name=data["name"],
            version=data.get("version", "0.0.1"),
            plugin_type=PluginType(data.get("type", "hook")),
            path=os.path.dirname(manifest_path),
            class_name=data.get("class", "Plugin"),
            enabled=data.get("enabled", True),
        )

    def _discover_single_file(self, plugin_path: str) -> Optional[PluginMetadata]:
        """Discover plugin from a single Python file"""
        spec = importlib.util.spec_from_file_location("plugin", plugin_path)
        if not spec or not spec.loader:
            return None

        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)

        # Look for Plugin subclass
        for name in dir(module):
            obj = getattr(module, name)
            if (isinstance(obj, type) and
                issubclass(obj, Plugin) and
                obj != Plugin and
                not name.startswith("_")):

                # Try to get config from class
                try:
                    config = obj.default_config()
                    return PluginMetadata(
                        name=config.name,
                        version=config.version,
                        plugin_type=config.plugin_type,
                        path=plugin_path,
                        class_name=name,
                    )
                except Exception:
                    pass

        return None

    def load_plugin(self, metadata: PluginMetadata) -> bool:
        """Load and initialize a plugin"""
        try:
            if os.path.isfile(metadata.path):
                # Single file plugin
                spec = importlib.util.spec_from_file_location(
                    f"ucts_plugin_{metadata.name}",
                    metadata.path
                )
                if not spec or not spec.loader:
                    return False
                module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(module)
            else:
                # Directory plugin
                init_path = os.path.join(metadata.path, "__init__.py")
                if os.path.exists(init_path):
                    spec = importlib.util.spec_from_file_location(
                        f"ucts_plugin_{metadata.name}",
                        init_path
                    )
                    if not spec or not spec.loader:
                        return False
                    module = importlib.util.module_from_spec(spec)
                    spec.loader.exec_module(module)
                else:
                    return False

            # Get plugin class
            plugin_class = getattr(module, metadata.class_name, None)
            if not plugin_class:
                logger.error(f"Plugin class {metadata.class_name} not found in {metadata.path}")
                return False

            # Instantiate and initialize
            instance = plugin_class()
            if not instance.initialize():
                logger.error(f"Plugin {metadata.name} failed to initialize")
                return False

            metadata.instance = instance
            self._plugins[metadata.name] = metadata

            # Register by type
            self._register_plugin(instance)

            logger.info(f"Loaded plugin: {metadata.name} v{metadata.version}")
            return True

        except Exception as e:
            logger.error(f"Failed to load plugin {metadata.name}: {e}")
            return False

    def _register_plugin(self, plugin: Plugin):
        """Register plugin by its type"""
        if isinstance(plugin, HookPlugin):
            self._hooks[plugin.hook_type].append(plugin)
        elif isinstance(plugin, IngesterPlugin):
            self._ingesters.append(plugin)
        elif isinstance(plugin, GeneratorPlugin):
            self._generators.append(plugin)
        elif isinstance(plugin, AnalyzerPlugin):
            self._analyzers.append(plugin)

    def load_all_plugins(self) -> int:
        """Discover and load all plugins"""
        discovered = self.discover_plugins()
        loaded = 0

        for metadata in discovered:
            if metadata.enabled and self.load_plugin(metadata):
                loaded += 1

        return loaded

    def unload_plugin(self, name: str) -> bool:
        """Unload a plugin"""
        if name not in self._plugins:
            return False

        metadata = self._plugins[name]
        if metadata.instance:
            metadata.instance.shutdown()

        del self._plugins[name]
        return True

    def get_plugin(self, name: str) -> Optional[Plugin]:
        """Get a plugin by name"""
        metadata = self._plugins.get(name)
        return metadata.instance if metadata else None

    def list_plugins(self) -> List[PluginMetadata]:
        """List all loaded plugins"""
        return list(self._plugins.values())

    def execute_hooks(self, hook_type: HookType, context: Dict[str, Any]) -> Dict[str, Any]:
        """Execute all hooks of a given type"""
        for hook in self._hooks[hook_type]:
            try:
                context = hook.execute(context)
            except Exception as e:
                logger.error(f"Hook {hook.name} failed: {e}")
        return context

    def get_ingester(self, source: str) -> Optional[IngesterPlugin]:
        """Find an ingester that can handle the source"""
        for ingester in self._ingesters:
            if ingester.can_ingest(source):
                return ingester
        return None

    def get_generator(self, structure: Any, target: str) -> Optional[GeneratorPlugin]:
        """Find a generator that can handle the target"""
        for generator in self._generators:
            if generator.can_generate(structure, target):
                return generator
        return None

    def get_analyzers(self) -> List[AnalyzerPlugin]:
        """Get all loaded analyzers"""
        return self._analyzers[:]

    def create_plugin_template(self, name: str, plugin_type: PluginType, output_dir: str) -> Dict[str, str]:
        """Create a new plugin template"""
        output_path = Path(output_dir) / name
        output_path.mkdir(parents=True, exist_ok=True)

        files = {}

        # plugin.json
        manifest = {
            "name": name,
            "version": "0.1.0",
            "type": plugin_type.value,
            "description": f"A custom {plugin_type.value} plugin for UCTS",
            "author": "Your Name",
            "class": f"{name.title().replace('-', '')}Plugin",
            "enabled": True,
        }
        manifest_path = output_path / "plugin.json"
        manifest_path.write_text(json.dumps(manifest, indent=2))
        files["plugin.json"] = str(manifest_path)

        # __init__.py
        class_name = manifest["class"]
        init_content = self._generate_plugin_code(name, class_name, plugin_type)
        init_path = output_path / "__init__.py"
        init_path.write_text(init_content)
        files["__init__.py"] = str(init_path)

        # README.md
        readme = f"""# {name}

A custom {plugin_type.value} plugin for UCTS.

## Installation

Copy this folder to `~/.ucts/plugins/`

## Usage

The plugin will be automatically discovered and loaded by UCTS.

## Development

Edit `__init__.py` to customize the plugin behavior.
"""
        readme_path = output_path / "README.md"
        readme_path.write_text(readme)
        files["README.md"] = str(readme_path)

        return files

    def _generate_plugin_code(self, name: str, class_name: str, plugin_type: PluginType) -> str:
        """Generate plugin code based on type"""
        base_imports = '''"""
{name} plugin for UCTS
"""
from typing import Any, Dict, Optional
from ucts.plugins import Plugin, PluginConfig, PluginType, HookType

'''

        if plugin_type == PluginType.INGESTER:
            return base_imports.format(name=name) + f'''
from ucts.plugins import IngesterPlugin
from ucts.core.models import Session


class {class_name}(IngesterPlugin):
    """Custom ingester for {name}"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="{name}",
            version="0.1.0",
            description="Custom ingester plugin",
            plugin_type=PluginType.INGESTER,
        )

    def initialize(self) -> bool:
        # Add initialization logic here
        return True

    def can_ingest(self, source: str) -> bool:
        # Return True if this ingester can handle the source
        return source.endswith(".custom")

    def ingest(self, source: str) -> Session:
        # Parse the source and return a Session
        return Session(source=source, messages=[], code_blocks=[])
'''

        elif plugin_type == PluginType.GENERATOR:
            return base_imports.format(name=name) + f'''
from ucts.plugins import GeneratorPlugin
from ucts.core.models import ProjectStructure


class {class_name}(GeneratorPlugin):
    """Custom generator for {name}"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="{name}",
            version="0.1.0",
            description="Custom generator plugin",
            plugin_type=PluginType.GENERATOR,
        )

    def initialize(self) -> bool:
        return True

    def can_generate(self, structure: ProjectStructure, target: str) -> bool:
        return target == "{name}"

    def generate(self, structure: ProjectStructure, output: str) -> Dict[str, str]:
        # Generate project files
        files = {{}}
        # Add generated files here
        return files
'''

        elif plugin_type == PluginType.ANALYZER:
            return base_imports.format(name=name) + f'''
from ucts.plugins import AnalyzerPlugin
from ucts.core.models import Session


class {class_name}(AnalyzerPlugin):
    """Custom analyzer for {name}"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="{name}",
            version="0.1.0",
            description="Custom analyzer plugin",
            plugin_type=PluginType.ANALYZER,
        )

    def initialize(self) -> bool:
        return True

    def analyze(self, session: Session) -> Dict[str, Any]:
        # Perform analysis and return results
        return {{
            "analyzed_by": "{name}",
            "message_count": len(session.messages),
        }}
'''

        else:  # Hook plugin
            return base_imports.format(name=name) + f'''
from ucts.plugins import HookPlugin


class {class_name}(HookPlugin):
    """Custom hook for {name}"""

    @classmethod
    def default_config(cls) -> PluginConfig:
        return PluginConfig(
            name="{name}",
            version="0.1.0",
            description="Custom hook plugin",
            plugin_type=PluginType.HOOK,
        )

    def initialize(self) -> bool:
        return True

    @property
    def hook_type(self) -> HookType:
        return HookType.POST_FORGE

    def execute(self, context: Dict[str, Any]) -> Dict[str, Any]:
        # Process context and return modified context
        print(f"Hook {{self.name}} executed")
        return context
'''


# Singleton instance
_plugin_manager: Optional[PluginManager] = None


def get_plugin_manager() -> PluginManager:
    """Get the global plugin manager instance"""
    global _plugin_manager
    if _plugin_manager is None:
        _plugin_manager = PluginManager()
    return _plugin_manager
