"""
Bidirectional Sync

Push updates back to AI contexts:
- Generate context summaries for new sessions
- Create "handoff" documents for session continuation
- Track what the AI knows vs. what changed
- Project state diff tracking
- Smart context compression
"""

import hashlib
import json
import logging
from dataclasses import dataclass, field
from datetime import datetime
from difflib import unified_diff
from pathlib import Path
from typing import Any, Dict, List, Optional, Set, Tuple

logger = logging.getLogger(__name__)


@dataclass
class SyncConfig:
    """Configuration for bidirectional sync"""
    max_context_tokens: int = 8000
    include_file_contents: bool = True
    include_git_history: bool = True
    compression_level: str = "medium"  # low, medium, high
    diff_context_lines: int = 3
    storage_path: str = ".ucts/sync"


@dataclass
class FileState:
    """State of a single file"""
    path: str
    content_hash: str
    size: int
    modified: datetime
    language: Optional[str] = None

    @staticmethod
    def from_file(path: Path) -> "FileState":
        """Create FileState from a file path"""
        content = path.read_bytes()
        return FileState(
            path=str(path),
            content_hash=hashlib.sha256(content).hexdigest()[:16],
            size=len(content),
            modified=datetime.fromtimestamp(path.stat().st_mtime),
            language=_detect_language(path.suffix)
        )


@dataclass
class ProjectState:
    """Snapshot of project state"""
    timestamp: datetime
    files: Dict[str, FileState] = field(default_factory=dict)
    dependencies: Dict[str, List[str]] = field(default_factory=dict)
    git_commit: Optional[str] = None
    git_branch: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        return {
            "timestamp": self.timestamp.isoformat(),
            "files": {
                p: {"hash": f.content_hash, "size": f.size, "language": f.language}
                for p, f in self.files.items()
            },
            "dependencies": self.dependencies,
            "git_commit": self.git_commit,
            "git_branch": self.git_branch,
            "metadata": self.metadata,
        }


@dataclass
class FileDiff:
    """Diff for a single file"""
    path: str
    change_type: str  # added, modified, deleted, renamed
    old_hash: Optional[str] = None
    new_hash: Optional[str] = None
    diff_lines: List[str] = field(default_factory=list)
    additions: int = 0
    deletions: int = 0


@dataclass
class ProjectDiff:
    """Differences between two project states"""
    from_state: datetime
    to_state: datetime
    files_added: List[str] = field(default_factory=list)
    files_modified: List[str] = field(default_factory=list)
    files_deleted: List[str] = field(default_factory=list)
    file_diffs: List[FileDiff] = field(default_factory=list)
    dependencies_added: Dict[str, List[str]] = field(default_factory=dict)
    dependencies_removed: Dict[str, List[str]] = field(default_factory=dict)
    summary: str = ""

    @property
    def has_changes(self) -> bool:
        return bool(self.files_added or self.files_modified or self.files_deleted)

    def to_markdown(self) -> str:
        """Convert diff to markdown format"""
        lines = [
            "# Project Changes",
            f"\n*From {self.from_state.isoformat()} to {self.to_state.isoformat()}*\n",
        ]

        if self.summary:
            lines.append(f"## Summary\n{self.summary}\n")

        if self.files_added:
            lines.append("## Files Added")
            for f in self.files_added:
                lines.append(f"- `{f}`")
            lines.append("")

        if self.files_modified:
            lines.append("## Files Modified")
            for f in self.files_modified:
                lines.append(f"- `{f}`")
            lines.append("")

        if self.files_deleted:
            lines.append("## Files Deleted")
            for f in self.files_deleted:
                lines.append(f"- `{f}`")
            lines.append("")

        if self.file_diffs:
            lines.append("## Detailed Changes")
            for diff in self.file_diffs:
                lines.append(f"\n### {diff.path}")
                lines.append(f"*{diff.change_type}* (+{diff.additions}/-{diff.deletions})")
                if diff.diff_lines:
                    lines.append("```diff")
                    lines.extend(diff.diff_lines[:50])  # Limit diff size
                    if len(diff.diff_lines) > 50:
                        lines.append(f"... and {len(diff.diff_lines) - 50} more lines")
                    lines.append("```")

        if self.dependencies_added:
            lines.append("\n## Dependencies Added")
            for eco, deps in self.dependencies_added.items():
                lines.append(f"- **{eco}**: {', '.join(deps)}")

        if self.dependencies_removed:
            lines.append("\n## Dependencies Removed")
            for eco, deps in self.dependencies_removed.items():
                lines.append(f"- **{eco}**: {', '.join(deps)}")

        return "\n".join(lines)


@dataclass
class ContextSummary:
    """Compressed context summary for AI sessions"""
    project_name: str
    description: str
    languages: List[str]
    key_files: List[str]
    architecture_overview: str
    recent_changes: str
    current_state: str
    dependencies: Dict[str, List[str]]
    todos: List[str]
    token_count: int = 0

    def to_markdown(self) -> str:
        """Generate markdown context summary"""
        lines = [
            f"# {self.project_name}",
            f"\n{self.description}\n",
            "## Languages & Stack",
            ", ".join(self.languages) or "Not detected",
            "",
            "## Key Files",
        ]

        for f in self.key_files[:10]:
            lines.append(f"- `{f}`")

        lines.extend([
            "",
            "## Architecture Overview",
            self.architecture_overview,
            "",
            "## Current State",
            self.current_state,
        ])

        if self.recent_changes:
            lines.extend([
                "",
                "## Recent Changes",
                self.recent_changes,
            ])

        if self.dependencies:
            lines.extend(["", "## Dependencies"])
            for eco, deps in self.dependencies.items():
                lines.append(f"- **{eco}**: {', '.join(deps[:5])}")
                if len(deps) > 5:
                    lines.append(f"  ... and {len(deps) - 5} more")

        if self.todos:
            lines.extend(["", "## Outstanding TODOs"])
            for todo in self.todos[:10]:
                lines.append(f"- {todo}")

        return "\n".join(lines)

    def to_compressed(self, max_tokens: int = 4000) -> str:
        """Generate compressed version for token limits"""
        full = self.to_markdown()

        # Rough token estimation (4 chars per token)
        if len(full) / 4 <= max_tokens:
            return full

        # Compress by removing less critical sections
        lines = [
            f"# {self.project_name}",
            self.description[:200],
            f"\nStack: {', '.join(self.languages)}",
            f"\nKey files: {', '.join(self.key_files[:5])}",
            f"\n{self.current_state[:500]}",
        ]

        if self.recent_changes:
            lines.append(f"\nRecent: {self.recent_changes[:300]}")

        return "\n".join(lines)


@dataclass
class HandoffDocument:
    """Document for handing off session to another AI instance"""
    session_id: str
    created: datetime
    context_summary: ContextSummary
    conversation_summary: str
    key_decisions: List[str]
    pending_tasks: List[str]
    important_context: List[str]
    files_discussed: List[str]
    code_snippets: List[Dict[str, str]]

    def to_markdown(self) -> str:
        """Generate handoff document in markdown"""
        lines = [
            "# Session Handoff Document",
            f"\n*Session: {self.session_id}*",
            f"*Created: {self.created.isoformat()}*\n",
            "---",
            "",
            self.context_summary.to_markdown(),
            "",
            "---",
            "",
            "## Conversation Summary",
            self.conversation_summary,
            "",
        ]

        if self.key_decisions:
            lines.extend(["## Key Decisions Made"])
            for d in self.key_decisions:
                lines.append(f"- {d}")
            lines.append("")

        if self.pending_tasks:
            lines.extend(["## Pending Tasks"])
            for t in self.pending_tasks:
                lines.append(f"- [ ] {t}")
            lines.append("")

        if self.important_context:
            lines.extend(["## Important Context"])
            for c in self.important_context:
                lines.append(f"- {c}")
            lines.append("")

        if self.files_discussed:
            lines.extend(["## Files Discussed"])
            for f in self.files_discussed:
                lines.append(f"- `{f}`")
            lines.append("")

        if self.code_snippets:
            lines.extend(["## Key Code Snippets"])
            for snippet in self.code_snippets[:5]:
                lines.append(f"\n### {snippet.get('description', 'Code')}")
                lines.append(f"```{snippet.get('language', '')}")
                lines.append(snippet.get('code', '')[:500])
                lines.append("```")

        return "\n".join(lines)

    def to_resume_prompt(self) -> str:
        """Generate a prompt for resuming the session"""
        return f"""Continue from a previous session. Here's the context:

{self.context_summary.to_compressed(2000)}

## Previous Conversation Summary
{self.conversation_summary}

## Key Decisions
{chr(10).join('- ' + d for d in self.key_decisions[:5])}

## Pending Tasks
{chr(10).join('- ' + t for t in self.pending_tasks)}

Please continue from where we left off."""


class BidirectionalSync:
    """
    Manages bidirectional synchronization between UCTS and AI contexts.

    Features:
    - Track project state changes
    - Generate context summaries
    - Create handoff documents
    - Smart context compression
    """

    def __init__(self, config: Optional[SyncConfig] = None):
        self.config = config or SyncConfig()
        self._states: List[ProjectState] = []
        self._storage_path = Path(self.config.storage_path)

    def capture_state(self, project_path: str) -> ProjectState:
        """
        Capture current project state.

        Args:
            project_path: Path to project root

        Returns:
            ProjectState snapshot
        """
        project = Path(project_path)
        state = ProjectState(timestamp=datetime.now())

        # Scan files
        ignore_patterns = {'.git', '__pycache__', 'node_modules', '.ucts', 'venv', '.venv'}

        for path in project.rglob('*'):
            if path.is_file():
                # Skip ignored directories
                if any(p in path.parts for p in ignore_patterns):
                    continue

                try:
                    rel_path = str(path.relative_to(project))
                    state.files[rel_path] = FileState.from_file(path)
                except Exception as e:
                    logger.debug(f"Skipping {path}: {e}")

        # Get git info
        state.git_commit, state.git_branch = self._get_git_info(project)

        # Parse dependencies
        state.dependencies = self._parse_dependencies(project)

        # Store state
        self._states.append(state)
        self._save_state(state)

        return state

    def compute_diff(
        self,
        from_state: ProjectState,
        to_state: ProjectState,
        include_content: bool = True
    ) -> ProjectDiff:
        """
        Compute differences between two project states.

        Args:
            from_state: Earlier state
            to_state: Later state
            include_content: Include file content diffs

        Returns:
            ProjectDiff with all changes
        """
        diff = ProjectDiff(
            from_state=from_state.timestamp,
            to_state=to_state.timestamp
        )

        from_files = set(from_state.files.keys())
        to_files = set(to_state.files.keys())

        # Added files
        diff.files_added = list(to_files - from_files)

        # Deleted files
        diff.files_deleted = list(from_files - to_files)

        # Modified files
        for path in from_files & to_files:
            if from_state.files[path].content_hash != to_state.files[path].content_hash:
                diff.files_modified.append(path)

        # Dependency changes
        for eco in set(from_state.dependencies.keys()) | set(to_state.dependencies.keys()):
            from_deps = set(from_state.dependencies.get(eco, []))
            to_deps = set(to_state.dependencies.get(eco, []))

            added = to_deps - from_deps
            removed = from_deps - to_deps

            if added:
                diff.dependencies_added[eco] = list(added)
            if removed:
                diff.dependencies_removed[eco] = list(removed)

        # Generate summary
        diff.summary = self._generate_diff_summary(diff)

        return diff

    def generate_context_summary(
        self,
        project_path: str,
        session: Optional[Any] = None
    ) -> ContextSummary:
        """
        Generate a context summary for AI sessions.

        Args:
            project_path: Path to project
            session: Optional UCTS session for additional context

        Returns:
            ContextSummary ready for AI context
        """
        from ucts.analysis import AnalysisEngine

        project = Path(project_path)

        # Get current state
        state = self.capture_state(project_path)

        # Analyze project
        structure = None
        if session:
            engine = AnalysisEngine()
            structure = engine.analyze(session)

        # Determine key files
        key_files = self._identify_key_files(state)

        # Generate architecture overview
        architecture = self._generate_architecture_overview(state, structure)

        # Get recent changes
        recent_changes = ""
        if len(self._states) >= 2:
            diff = self.compute_diff(self._states[-2], self._states[-1])
            if diff.has_changes:
                recent_changes = f"Added: {len(diff.files_added)}, Modified: {len(diff.files_modified)}, Deleted: {len(diff.files_deleted)}"

        summary = ContextSummary(
            project_name=structure.name if structure else project.name,
            description=structure.description if structure else f"Project at {project_path}",
            languages=structure.languages if structure else list(set(
                f.language for f in state.files.values() if f.language
            )),
            key_files=key_files,
            architecture_overview=architecture,
            recent_changes=recent_changes,
            current_state=f"{len(state.files)} files tracked, last update {state.timestamp.isoformat()}",
            dependencies=state.dependencies,
            todos=structure.todos if structure else [],
        )

        # Estimate token count
        summary.token_count = len(summary.to_markdown()) // 4

        return summary

    def create_handoff(
        self,
        session: Any,
        project_path: str,
        conversation_summary: str = "",
        decisions: Optional[List[str]] = None,
        pending_tasks: Optional[List[str]] = None
    ) -> HandoffDocument:
        """
        Create a handoff document for session continuation.

        Args:
            session: UCTS session
            project_path: Project path
            conversation_summary: Summary of conversation
            decisions: Key decisions made
            pending_tasks: Tasks still pending

        Returns:
            HandoffDocument ready for use
        """
        context = self.generate_context_summary(project_path, session)

        # Extract important context from session
        important_context = []
        files_discussed = []
        code_snippets = []

        if session:
            # Extract files mentioned
            for msg in session.messages:
                # Look for file paths in content
                import re
                paths = re.findall(r'[\w/.-]+\.\w+', msg.content)
                files_discussed.extend(p for p in paths if len(p) < 100)

            # Extract code blocks
            for block in session.code_blocks[:10]:
                code_snippets.append({
                    "language": block.language,
                    "code": block.code[:500],
                    "description": f"{block.language} code from conversation"
                })

        return HandoffDocument(
            session_id=session.source if session else f"handoff-{datetime.now().strftime('%Y%m%d%H%M%S')}",
            created=datetime.now(),
            context_summary=context,
            conversation_summary=conversation_summary or "No summary provided",
            key_decisions=decisions or [],
            pending_tasks=pending_tasks or context.todos,
            important_context=important_context,
            files_discussed=list(set(files_discussed))[:20],
            code_snippets=code_snippets,
        )

    def export_for_resume(
        self,
        handoff: HandoffDocument,
        output_path: str,
        format: str = "markdown"
    ) -> str:
        """
        Export handoff document for session resume.

        Args:
            handoff: HandoffDocument to export
            output_path: Output file path
            format: Export format (markdown, json, prompt)

        Returns:
            Path to exported file
        """
        output = Path(output_path)
        output.parent.mkdir(parents=True, exist_ok=True)

        if format == "markdown":
            content = handoff.to_markdown()
            output = output.with_suffix('.md')
        elif format == "json":
            content = json.dumps({
                "session_id": handoff.session_id,
                "created": handoff.created.isoformat(),
                "context": handoff.context_summary.to_markdown(),
                "conversation_summary": handoff.conversation_summary,
                "decisions": handoff.key_decisions,
                "pending_tasks": handoff.pending_tasks,
                "files_discussed": handoff.files_discussed,
            }, indent=2)
            output = output.with_suffix('.json')
        elif format == "prompt":
            content = handoff.to_resume_prompt()
            output = output.with_suffix('.txt')
        else:
            raise ValueError(f"Unknown format: {format}")

        output.write_text(content)
        logger.info(f"Exported handoff to {output}")

        return str(output)

    def _get_git_info(self, project: Path) -> Tuple[Optional[str], Optional[str]]:
        """Get git commit and branch"""
        git_dir = project / '.git'
        if not git_dir.exists():
            return None, None

        try:
            import subprocess
            commit = subprocess.check_output(
                ['git', 'rev-parse', 'HEAD'],
                cwd=project,
                stderr=subprocess.DEVNULL
            ).decode().strip()[:8]

            branch = subprocess.check_output(
                ['git', 'rev-parse', '--abbrev-ref', 'HEAD'],
                cwd=project,
                stderr=subprocess.DEVNULL
            ).decode().strip()

            return commit, branch
        except Exception:
            return None, None

    def _parse_dependencies(self, project: Path) -> Dict[str, List[str]]:
        """Parse project dependencies"""
        deps = {}

        # Python
        req_file = project / 'requirements.txt'
        if req_file.exists():
            deps['python'] = [
                line.split('==')[0].split('>=')[0].strip()
                for line in req_file.read_text().splitlines()
                if line.strip() and not line.startswith('#')
            ]

        # Node.js
        pkg_file = project / 'package.json'
        if pkg_file.exists():
            try:
                pkg = json.loads(pkg_file.read_text())
                deps['npm'] = list(pkg.get('dependencies', {}).keys())
                deps['npm_dev'] = list(pkg.get('devDependencies', {}).keys())
            except Exception:
                pass

        return deps

    def _identify_key_files(self, state: ProjectState) -> List[str]:
        """Identify key files in the project"""
        key_files = []

        # Priority patterns
        priority_patterns = [
            'main.py', 'app.py', 'index.ts', 'index.js',
            'package.json', 'requirements.txt', 'pyproject.toml',
            'README.md', 'Dockerfile', 'docker-compose.yml',
        ]

        for pattern in priority_patterns:
            matches = [p for p in state.files.keys() if p.endswith(pattern)]
            key_files.extend(matches)

        # Add largest files by language
        by_language: Dict[str, List[Tuple[str, int]]] = {}
        for path, file_state in state.files.items():
            lang = file_state.language
            if lang:
                if lang not in by_language:
                    by_language[lang] = []
                by_language[lang].append((path, file_state.size))

        for lang, files in by_language.items():
            files.sort(key=lambda x: -x[1])
            key_files.extend(f[0] for f in files[:3])

        return list(dict.fromkeys(key_files))[:15]  # Dedupe and limit

    def _generate_architecture_overview(
        self,
        state: ProjectState,
        structure: Optional[Any]
    ) -> str:
        """Generate architecture overview"""
        # Count files by directory
        dir_counts: Dict[str, int] = {}
        for path in state.files.keys():
            parts = Path(path).parts
            if len(parts) > 1:
                top_dir = parts[0]
                dir_counts[top_dir] = dir_counts.get(top_dir, 0) + 1

        # Build overview
        lines = []

        if dir_counts:
            lines.append("Directory structure:")
            for d, count in sorted(dir_counts.items(), key=lambda x: -x[1])[:5]:
                lines.append(f"- {d}/ ({count} files)")

        if state.dependencies:
            lines.append("\nDependency ecosystems: " + ", ".join(state.dependencies.keys()))

        if state.git_branch:
            lines.append(f"\nGit: {state.git_branch} @ {state.git_commit}")

        return "\n".join(lines) or "No architecture detected"

    def _generate_diff_summary(self, diff: ProjectDiff) -> str:
        """Generate human-readable diff summary"""
        parts = []

        if diff.files_added:
            parts.append(f"{len(diff.files_added)} files added")
        if diff.files_modified:
            parts.append(f"{len(diff.files_modified)} files modified")
        if diff.files_deleted:
            parts.append(f"{len(diff.files_deleted)} files deleted")

        return ", ".join(parts) or "No changes"

    def _save_state(self, state: ProjectState):
        """Save state to disk"""
        self._storage_path.mkdir(parents=True, exist_ok=True)
        state_file = self._storage_path / f"state_{state.timestamp.strftime('%Y%m%d%H%M%S')}.json"

        with open(state_file, 'w') as f:
            json.dump(state.to_dict(), f, indent=2)

    def load_states(self) -> List[ProjectState]:
        """Load saved states from disk"""
        if not self._storage_path.exists():
            return []

        states = []
        for state_file in sorted(self._storage_path.glob('state_*.json')):
            try:
                with open(state_file) as f:
                    data = json.load(f)
                    state = ProjectState(
                        timestamp=datetime.fromisoformat(data['timestamp']),
                        git_commit=data.get('git_commit'),
                        git_branch=data.get('git_branch'),
                        dependencies=data.get('dependencies', {}),
                        metadata=data.get('metadata', {}),
                    )
                    for path, file_data in data.get('files', {}).items():
                        state.files[path] = FileState(
                            path=path,
                            content_hash=file_data['hash'],
                            size=file_data['size'],
                            modified=datetime.now(),  # Not stored
                            language=file_data.get('language'),
                        )
                    states.append(state)
            except Exception as e:
                logger.error(f"Failed to load state {state_file}: {e}")

        self._states = states
        return states


def _detect_language(suffix: str) -> Optional[str]:
    """Detect language from file suffix"""
    mapping = {
        '.py': 'python',
        '.js': 'javascript',
        '.ts': 'typescript',
        '.tsx': 'typescript',
        '.jsx': 'javascript',
        '.go': 'go',
        '.rs': 'rust',
        '.java': 'java',
        '.rb': 'ruby',
        '.php': 'php',
        '.c': 'c',
        '.cpp': 'cpp',
        '.h': 'c',
        '.cs': 'csharp',
        '.swift': 'swift',
        '.kt': 'kotlin',
        '.md': 'markdown',
        '.json': 'json',
        '.yaml': 'yaml',
        '.yml': 'yaml',
        '.toml': 'toml',
        '.sql': 'sql',
        '.sh': 'shell',
        '.bash': 'shell',
    }
    return mapping.get(suffix.lower())


# Singleton instance
_sync_manager: Optional[BidirectionalSync] = None


def get_sync_manager(config: Optional[SyncConfig] = None) -> BidirectionalSync:
    """Get the global sync manager"""
    global _sync_manager
    if _sync_manager is None or config is not None:
        _sync_manager = BidirectionalSync(config)
    return _sync_manager
