"""
SSO/Authentication Module for UCTS Team Features

Provides foundation for enterprise authentication:
- OAuth 2.0 / OpenID Connect (OIDC) support
- SAML 2.0 support
- API Key authentication
- Session management
- Token refresh handling
"""

import base64
import hashlib
import hmac
import json
import logging
import os
import secrets
import time
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from enum import Enum
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Tuple
from urllib.parse import urlencode, parse_qs, urlparse
import uuid

logger = logging.getLogger(__name__)


class AuthProvider(Enum):
    """Supported authentication providers"""
    LOCAL = "local"           # Local username/password
    API_KEY = "api_key"       # API key authentication
    OAUTH2 = "oauth2"         # Generic OAuth 2.0
    OIDC = "oidc"             # OpenID Connect
    SAML = "saml"             # SAML 2.0
    AZURE_AD = "azure_ad"     # Azure Active Directory
    OKTA = "okta"             # Okta
    GOOGLE = "google"         # Google Workspace
    GITHUB = "github"         # GitHub OAuth


class AuthStatus(Enum):
    """Authentication status"""
    AUTHENTICATED = "authenticated"
    PENDING = "pending"
    EXPIRED = "expired"
    INVALID = "invalid"
    MFA_REQUIRED = "mfa_required"


@dataclass
class AuthConfig:
    """Authentication configuration"""
    provider: AuthProvider = AuthProvider.LOCAL
    enabled: bool = False

    # OAuth/OIDC settings
    client_id: str = ""
    client_secret: str = ""
    authorization_url: str = ""
    token_url: str = ""
    userinfo_url: str = ""
    scopes: List[str] = field(default_factory=lambda: ["openid", "profile", "email"])
    redirect_uri: str = "http://localhost:8765/callback"

    # SAML settings
    idp_entity_id: str = ""
    idp_sso_url: str = ""
    idp_slo_url: str = ""
    idp_certificate: str = ""
    sp_entity_id: str = ""
    sp_acs_url: str = ""

    # Session settings
    session_timeout: int = 3600  # 1 hour
    refresh_token_lifetime: int = 86400 * 7  # 7 days
    require_mfa: bool = False

    def to_dict(self) -> Dict[str, Any]:
        return {
            "provider": self.provider.value,
            "enabled": self.enabled,
            "client_id": self.client_id,
            "authorization_url": self.authorization_url,
            "token_url": self.token_url,
            "userinfo_url": self.userinfo_url,
            "scopes": self.scopes,
            "redirect_uri": self.redirect_uri,
            "idp_entity_id": self.idp_entity_id,
            "idp_sso_url": self.idp_sso_url,
            "sp_entity_id": self.sp_entity_id,
            "session_timeout": self.session_timeout,
            "require_mfa": self.require_mfa,
        }


@dataclass
class AuthToken:
    """Authentication token"""
    token_id: str
    access_token: str
    token_type: str = "Bearer"
    expires_at: datetime = field(default_factory=lambda: datetime.now() + timedelta(hours=1))
    refresh_token: Optional[str] = None
    refresh_expires_at: Optional[datetime] = None
    scopes: List[str] = field(default_factory=list)
    user_id: Optional[str] = None
    provider: AuthProvider = AuthProvider.LOCAL

    @property
    def is_expired(self) -> bool:
        return datetime.now() >= self.expires_at

    @property
    def can_refresh(self) -> bool:
        if not self.refresh_token:
            return False
        if self.refresh_expires_at and datetime.now() >= self.refresh_expires_at:
            return False
        return True

    def to_dict(self) -> Dict[str, Any]:
        return {
            "token_id": self.token_id,
            "access_token": self.access_token,
            "token_type": self.token_type,
            "expires_at": self.expires_at.isoformat(),
            "refresh_token": self.refresh_token,
            "refresh_expires_at": self.refresh_expires_at.isoformat() if self.refresh_expires_at else None,
            "scopes": self.scopes,
            "user_id": self.user_id,
            "provider": self.provider.value,
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "AuthToken":
        return cls(
            token_id=data["token_id"],
            access_token=data["access_token"],
            token_type=data.get("token_type", "Bearer"),
            expires_at=datetime.fromisoformat(data["expires_at"]),
            refresh_token=data.get("refresh_token"),
            refresh_expires_at=datetime.fromisoformat(data["refresh_expires_at"]) if data.get("refresh_expires_at") else None,
            scopes=data.get("scopes", []),
            user_id=data.get("user_id"),
            provider=AuthProvider(data.get("provider", "local")),
        )


@dataclass
class UserIdentity:
    """User identity from authentication"""
    user_id: str
    email: str
    name: str = ""
    given_name: str = ""
    family_name: str = ""
    picture: str = ""
    provider: AuthProvider = AuthProvider.LOCAL
    provider_user_id: str = ""
    groups: List[str] = field(default_factory=list)
    roles: List[str] = field(default_factory=list)
    attributes: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        return {
            "user_id": self.user_id,
            "email": self.email,
            "name": self.name,
            "given_name": self.given_name,
            "family_name": self.family_name,
            "picture": self.picture,
            "provider": self.provider.value,
            "provider_user_id": self.provider_user_id,
            "groups": self.groups,
            "roles": self.roles,
            "attributes": self.attributes,
        }


@dataclass
class AuthSession:
    """User authentication session"""
    session_id: str
    user_id: str
    identity: UserIdentity
    token: AuthToken
    created_at: datetime = field(default_factory=datetime.now)
    last_activity: datetime = field(default_factory=datetime.now)
    ip_address: str = ""
    user_agent: str = ""
    mfa_verified: bool = False

    @property
    def is_valid(self) -> bool:
        return not self.token.is_expired

    def to_dict(self) -> Dict[str, Any]:
        return {
            "session_id": self.session_id,
            "user_id": self.user_id,
            "identity": self.identity.to_dict(),
            "token": self.token.to_dict(),
            "created_at": self.created_at.isoformat(),
            "last_activity": self.last_activity.isoformat(),
            "ip_address": self.ip_address,
            "user_agent": self.user_agent,
            "mfa_verified": self.mfa_verified,
        }


class AuthProviderBase(ABC):
    """Base class for authentication providers"""

    def __init__(self, config: AuthConfig):
        self.config = config

    @abstractmethod
    def get_authorization_url(self, state: str) -> str:
        """Get URL to redirect user for authorization"""
        pass

    @abstractmethod
    def exchange_code(self, code: str, state: str) -> AuthToken:
        """Exchange authorization code for tokens"""
        pass

    @abstractmethod
    def refresh_token(self, token: AuthToken) -> AuthToken:
        """Refresh an expired token"""
        pass

    @abstractmethod
    def get_user_info(self, token: AuthToken) -> UserIdentity:
        """Get user information from provider"""
        pass

    @abstractmethod
    def revoke_token(self, token: AuthToken) -> bool:
        """Revoke a token"""
        pass


class OAuth2Provider(AuthProviderBase):
    """OAuth 2.0 / OpenID Connect provider"""

    def get_authorization_url(self, state: str) -> str:
        """Generate OAuth2 authorization URL"""
        params = {
            "client_id": self.config.client_id,
            "redirect_uri": self.config.redirect_uri,
            "response_type": "code",
            "scope": " ".join(self.config.scopes),
            "state": state,
        }

        # PKCE support
        code_verifier = secrets.token_urlsafe(32)
        code_challenge = base64.urlsafe_b64encode(
            hashlib.sha256(code_verifier.encode()).digest()
        ).decode().rstrip("=")

        params["code_challenge"] = code_challenge
        params["code_challenge_method"] = "S256"

        # Store verifier for later (would be in session in real implementation)
        self._code_verifier = code_verifier

        return f"{self.config.authorization_url}?{urlencode(params)}"

    def exchange_code(self, code: str, state: str) -> AuthToken:
        """Exchange authorization code for tokens

        NOTE: This is a stub implementation. In production, this would make
        an actual HTTP request to the token endpoint.
        """
        logger.info(f"OAuth2: Exchanging code for tokens (provider: {self.config.provider.value})")

        # In production, this would be an HTTP POST to token_url
        # For now, return a stub token for development/testing
        return AuthToken(
            token_id=str(uuid.uuid4()),
            access_token=f"stub_access_token_{secrets.token_urlsafe(16)}",
            token_type="Bearer",
            expires_at=datetime.now() + timedelta(seconds=self.config.session_timeout),
            refresh_token=f"stub_refresh_token_{secrets.token_urlsafe(16)}",
            refresh_expires_at=datetime.now() + timedelta(seconds=self.config.refresh_token_lifetime),
            scopes=self.config.scopes,
            provider=self.config.provider,
        )

    def refresh_token(self, token: AuthToken) -> AuthToken:
        """Refresh an expired token

        NOTE: Stub implementation for development.
        """
        if not token.can_refresh:
            raise ValueError("Token cannot be refreshed")

        logger.info(f"OAuth2: Refreshing token (provider: {self.config.provider.value})")

        return AuthToken(
            token_id=str(uuid.uuid4()),
            access_token=f"stub_access_token_{secrets.token_urlsafe(16)}",
            token_type="Bearer",
            expires_at=datetime.now() + timedelta(seconds=self.config.session_timeout),
            refresh_token=token.refresh_token,  # Keep same refresh token
            refresh_expires_at=token.refresh_expires_at,
            scopes=token.scopes,
            user_id=token.user_id,
            provider=token.provider,
        )

    def get_user_info(self, token: AuthToken) -> UserIdentity:
        """Get user info from provider

        NOTE: Stub implementation for development.
        """
        logger.info(f"OAuth2: Fetching user info (provider: {self.config.provider.value})")

        # In production, this would call the userinfo endpoint
        return UserIdentity(
            user_id=str(uuid.uuid4()),
            email="user@example.com",
            name="Test User",
            provider=self.config.provider,
        )

    def revoke_token(self, token: AuthToken) -> bool:
        """Revoke a token

        NOTE: Stub implementation for development.
        """
        logger.info(f"OAuth2: Revoking token (provider: {self.config.provider.value})")
        return True


class SAMLProvider(AuthProviderBase):
    """SAML 2.0 provider"""

    def get_authorization_url(self, state: str) -> str:
        """Generate SAML AuthnRequest URL

        NOTE: Stub implementation. In production, this would generate
        a proper SAML AuthnRequest XML document.
        """
        logger.info("SAML: Generating AuthnRequest")

        # In production, this would:
        # 1. Generate SAML AuthnRequest XML
        # 2. Sign the request if required
        # 3. Encode it (base64 + deflate for redirect binding)
        # 4. Add RelayState parameter

        params = {
            "SAMLRequest": base64.b64encode(f"stub_saml_request_{state}".encode()).decode(),
            "RelayState": state,
        }

        return f"{self.config.idp_sso_url}?{urlencode(params)}"

    def exchange_code(self, code: str, state: str) -> AuthToken:
        """Process SAML Response

        NOTE: Stub implementation. In production, this would:
        1. Decode the SAML Response
        2. Validate the signature
        3. Check assertion conditions
        4. Extract user attributes
        """
        logger.info("SAML: Processing SAML Response")

        return AuthToken(
            token_id=str(uuid.uuid4()),
            access_token=f"saml_session_{secrets.token_urlsafe(16)}",
            token_type="SAML",
            expires_at=datetime.now() + timedelta(seconds=self.config.session_timeout),
            provider=AuthProvider.SAML,
        )

    def refresh_token(self, token: AuthToken) -> AuthToken:
        """SAML doesn't support token refresh - require re-authentication"""
        raise ValueError("SAML sessions cannot be refreshed. User must re-authenticate.")

    def get_user_info(self, token: AuthToken) -> UserIdentity:
        """Extract user info from SAML assertion

        NOTE: In production, user info is extracted during exchange_code
        from the SAML assertion attributes.
        """
        return UserIdentity(
            user_id=str(uuid.uuid4()),
            email="user@example.com",
            name="SAML User",
            provider=AuthProvider.SAML,
        )

    def revoke_token(self, token: AuthToken) -> bool:
        """Initiate SAML logout

        NOTE: Stub implementation. In production, this would
        generate a SAML LogoutRequest.
        """
        logger.info("SAML: Initiating logout")
        return True


class APIKeyProvider(AuthProviderBase):
    """API Key authentication provider"""

    def __init__(self, config: AuthConfig, storage_path: Path):
        super().__init__(config)
        self._storage_path = storage_path / "api_keys"
        self._storage_path.mkdir(parents=True, exist_ok=True)
        self._keys: Dict[str, Dict[str, Any]] = {}
        self._load_keys()

    def _load_keys(self):
        """Load API keys from storage"""
        keys_file = self._storage_path / "keys.json"
        if keys_file.exists():
            try:
                with open(keys_file) as f:
                    self._keys = json.load(f)
            except Exception as e:
                logger.warning(f"Failed to load API keys: {e}")

    def _save_keys(self):
        """Save API keys to storage"""
        keys_file = self._storage_path / "keys.json"
        with open(keys_file, 'w') as f:
            json.dump(self._keys, f, indent=2)

    def create_api_key(
        self,
        user_id: str,
        name: str,
        scopes: Optional[List[str]] = None,
        expires_in: Optional[int] = None
    ) -> Tuple[str, str]:
        """Create a new API key

        Returns:
            Tuple of (key_id, api_key)
            The api_key is only shown once and should be stored securely.
        """
        key_id = str(uuid.uuid4())[:8]
        api_key = f"ucts_{secrets.token_urlsafe(32)}"

        # Store hash of key, not the key itself
        key_hash = hashlib.sha256(api_key.encode()).hexdigest()

        self._keys[key_id] = {
            "key_hash": key_hash,
            "user_id": user_id,
            "name": name,
            "scopes": scopes or ["read", "write"],
            "created_at": datetime.now().isoformat(),
            "expires_at": (datetime.now() + timedelta(seconds=expires_in)).isoformat() if expires_in else None,
            "last_used": None,
        }

        self._save_keys()
        logger.info(f"Created API key {key_id} for user {user_id}")

        return key_id, api_key

    def validate_api_key(self, api_key: str) -> Optional[AuthToken]:
        """Validate an API key and return a token"""
        key_hash = hashlib.sha256(api_key.encode()).hexdigest()

        for key_id, key_data in self._keys.items():
            if key_data["key_hash"] == key_hash:
                # Check expiration
                if key_data.get("expires_at"):
                    expires_at = datetime.fromisoformat(key_data["expires_at"])
                    if datetime.now() >= expires_at:
                        return None

                # Update last used
                key_data["last_used"] = datetime.now().isoformat()
                self._save_keys()

                return AuthToken(
                    token_id=key_id,
                    access_token=api_key,
                    token_type="APIKey",
                    expires_at=datetime.fromisoformat(key_data["expires_at"]) if key_data.get("expires_at") else datetime.now() + timedelta(days=365),
                    scopes=key_data["scopes"],
                    user_id=key_data["user_id"],
                    provider=AuthProvider.API_KEY,
                )

        return None

    def revoke_api_key(self, key_id: str) -> bool:
        """Revoke an API key"""
        if key_id in self._keys:
            del self._keys[key_id]
            self._save_keys()
            logger.info(f"Revoked API key {key_id}")
            return True
        return False

    def list_api_keys(self, user_id: str) -> List[Dict[str, Any]]:
        """List API keys for a user"""
        return [
            {
                "key_id": key_id,
                "name": data["name"],
                "scopes": data["scopes"],
                "created_at": data["created_at"],
                "expires_at": data.get("expires_at"),
                "last_used": data.get("last_used"),
            }
            for key_id, data in self._keys.items()
            if data["user_id"] == user_id
        ]

    # These methods are not used for API key auth but required by base class
    def get_authorization_url(self, state: str) -> str:
        raise NotImplementedError("API key auth doesn't use authorization URLs")

    def exchange_code(self, code: str, state: str) -> AuthToken:
        raise NotImplementedError("API key auth doesn't use authorization codes")

    def refresh_token(self, token: AuthToken) -> AuthToken:
        raise NotImplementedError("API keys don't need refresh")

    def get_user_info(self, token: AuthToken) -> UserIdentity:
        key_data = self._keys.get(token.token_id)
        if key_data:
            return UserIdentity(
                user_id=key_data["user_id"],
                email="",
                name=key_data["name"],
                provider=AuthProvider.API_KEY,
            )
        raise ValueError("Invalid API key token")

    def revoke_token(self, token: AuthToken) -> bool:
        return self.revoke_api_key(token.token_id)


class AuthManager:
    """
    Manages authentication for UCTS.

    Features:
    - Multiple provider support (OAuth, SAML, API keys)
    - Session management
    - Token refresh
    - MFA support (stub)
    """

    def __init__(self, storage_dir: str, config: Optional[AuthConfig] = None):
        self._storage_path = Path(storage_dir) / "auth"
        self._storage_path.mkdir(parents=True, exist_ok=True)

        self.config = config or AuthConfig()
        self._providers: Dict[AuthProvider, AuthProviderBase] = {}
        self._sessions: Dict[str, AuthSession] = {}

        # Initialize providers
        self._init_providers()
        self._load_sessions()

    def _init_providers(self):
        """Initialize authentication providers"""
        # Always enable API key auth
        self._providers[AuthProvider.API_KEY] = APIKeyProvider(
            AuthConfig(provider=AuthProvider.API_KEY, enabled=True),
            self._storage_path
        )

        # OAuth providers
        if self.config.provider in [AuthProvider.OAUTH2, AuthProvider.OIDC,
                                    AuthProvider.AZURE_AD, AuthProvider.OKTA,
                                    AuthProvider.GOOGLE, AuthProvider.GITHUB]:
            self._providers[self.config.provider] = OAuth2Provider(self.config)

        # SAML provider
        if self.config.provider == AuthProvider.SAML:
            self._providers[AuthProvider.SAML] = SAMLProvider(self.config)

    def _load_sessions(self):
        """Load active sessions from storage"""
        sessions_file = self._storage_path / "sessions.json"
        if sessions_file.exists():
            try:
                with open(sessions_file) as f:
                    data = json.load(f)
                    for s in data:
                        # Only load non-expired sessions
                        token_data = s.get("token", {})
                        expires_at = datetime.fromisoformat(token_data.get("expires_at", "2000-01-01"))
                        if datetime.now() < expires_at:
                            session = AuthSession(
                                session_id=s["session_id"],
                                user_id=s["user_id"],
                                identity=UserIdentity(**s["identity"]) if isinstance(s["identity"], dict) else s["identity"],
                                token=AuthToken.from_dict(token_data),
                                created_at=datetime.fromisoformat(s["created_at"]),
                                last_activity=datetime.fromisoformat(s["last_activity"]),
                                ip_address=s.get("ip_address", ""),
                                user_agent=s.get("user_agent", ""),
                                mfa_verified=s.get("mfa_verified", False),
                            )
                            self._sessions[session.session_id] = session
            except Exception as e:
                logger.warning(f"Failed to load sessions: {e}")

    def _save_sessions(self):
        """Save sessions to storage"""
        sessions_file = self._storage_path / "sessions.json"
        with open(sessions_file, 'w') as f:
            json.dump([s.to_dict() for s in self._sessions.values()], f, indent=2)

    def get_provider(self, provider: AuthProvider) -> Optional[AuthProviderBase]:
        """Get an authentication provider"""
        return self._providers.get(provider)

    def start_oauth_flow(self, provider: Optional[AuthProvider] = None) -> Tuple[str, str]:
        """Start OAuth authentication flow

        Returns:
            Tuple of (authorization_url, state)
        """
        provider = provider or self.config.provider
        auth_provider = self._providers.get(provider)

        if not auth_provider:
            raise ValueError(f"Provider {provider} not configured")

        state = secrets.token_urlsafe(32)
        url = auth_provider.get_authorization_url(state)

        return url, state

    def complete_oauth_flow(
        self,
        code: str,
        state: str,
        provider: Optional[AuthProvider] = None,
        ip_address: str = "",
        user_agent: str = ""
    ) -> AuthSession:
        """Complete OAuth authentication flow"""
        provider = provider or self.config.provider
        auth_provider = self._providers.get(provider)

        if not auth_provider:
            raise ValueError(f"Provider {provider} not configured")

        # Exchange code for tokens
        token = auth_provider.exchange_code(code, state)

        # Get user info
        identity = auth_provider.get_user_info(token)
        token.user_id = identity.user_id

        # Create session
        session = AuthSession(
            session_id=str(uuid.uuid4()),
            user_id=identity.user_id,
            identity=identity,
            token=token,
            ip_address=ip_address,
            user_agent=user_agent,
        )

        self._sessions[session.session_id] = session
        self._save_sessions()

        logger.info(f"Created session for user {identity.user_id} via {provider.value}")
        return session

    def authenticate_api_key(self, api_key: str) -> Optional[AuthSession]:
        """Authenticate using an API key"""
        provider = self._providers.get(AuthProvider.API_KEY)
        if not isinstance(provider, APIKeyProvider):
            return None

        token = provider.validate_api_key(api_key)
        if not token:
            return None

        identity = provider.get_user_info(token)

        session = AuthSession(
            session_id=str(uuid.uuid4()),
            user_id=identity.user_id,
            identity=identity,
            token=token,
        )

        self._sessions[session.session_id] = session
        self._save_sessions()

        return session

    def get_session(self, session_id: str) -> Optional[AuthSession]:
        """Get a session by ID"""
        session = self._sessions.get(session_id)
        if session and session.is_valid:
            session.last_activity = datetime.now()
            return session
        return None

    def validate_token(self, access_token: str) -> Optional[AuthSession]:
        """Validate an access token and return the session"""
        for session in self._sessions.values():
            if session.token.access_token == access_token and session.is_valid:
                session.last_activity = datetime.now()
                return session
        return None

    def refresh_session(self, session_id: str) -> Optional[AuthSession]:
        """Refresh a session's token"""
        session = self._sessions.get(session_id)
        if not session:
            return None

        provider = self._providers.get(session.token.provider)
        if not provider:
            return None

        try:
            new_token = provider.refresh_token(session.token)
            session.token = new_token
            session.last_activity = datetime.now()
            self._save_sessions()
            return session
        except Exception as e:
            logger.warning(f"Failed to refresh session: {e}")
            return None

    def logout(self, session_id: str) -> bool:
        """End a session"""
        session = self._sessions.get(session_id)
        if not session:
            return False

        # Revoke token if provider supports it
        provider = self._providers.get(session.token.provider)
        if provider:
            try:
                provider.revoke_token(session.token)
            except Exception:
                pass

        del self._sessions[session_id]
        self._save_sessions()

        logger.info(f"Logged out session {session_id}")
        return True

    def create_api_key(
        self,
        user_id: str,
        name: str,
        scopes: Optional[List[str]] = None
    ) -> Tuple[str, str]:
        """Create an API key for a user"""
        provider = self._providers.get(AuthProvider.API_KEY)
        if not isinstance(provider, APIKeyProvider):
            raise ValueError("API key provider not configured")

        return provider.create_api_key(user_id, name, scopes)

    def list_api_keys(self, user_id: str) -> List[Dict[str, Any]]:
        """List API keys for a user"""
        provider = self._providers.get(AuthProvider.API_KEY)
        if not isinstance(provider, APIKeyProvider):
            return []

        return provider.list_api_keys(user_id)

    def revoke_api_key(self, key_id: str) -> bool:
        """Revoke an API key"""
        provider = self._providers.get(AuthProvider.API_KEY)
        if not isinstance(provider, APIKeyProvider):
            return False

        return provider.revoke_api_key(key_id)

    def get_active_sessions(self, user_id: str) -> List[AuthSession]:
        """Get all active sessions for a user"""
        return [
            s for s in self._sessions.values()
            if s.user_id == user_id and s.is_valid
        ]

    def cleanup_expired_sessions(self) -> int:
        """Remove expired sessions"""
        expired = [
            sid for sid, session in self._sessions.items()
            if not session.is_valid
        ]

        for sid in expired:
            del self._sessions[sid]

        if expired:
            self._save_sessions()
            logger.info(f"Cleaned up {len(expired)} expired sessions")

        return len(expired)


# Provider-specific configurations for common IdPs

def get_azure_ad_config(tenant_id: str, client_id: str, client_secret: str) -> AuthConfig:
    """Get Azure AD configuration"""
    return AuthConfig(
        provider=AuthProvider.AZURE_AD,
        enabled=True,
        client_id=client_id,
        client_secret=client_secret,
        authorization_url=f"https://login.microsoftonline.com/{tenant_id}/oauth2/v2.0/authorize",
        token_url=f"https://login.microsoftonline.com/{tenant_id}/oauth2/v2.0/token",
        userinfo_url="https://graph.microsoft.com/oidc/userinfo",
        scopes=["openid", "profile", "email", "User.Read"],
    )


def get_okta_config(domain: str, client_id: str, client_secret: str) -> AuthConfig:
    """Get Okta configuration"""
    return AuthConfig(
        provider=AuthProvider.OKTA,
        enabled=True,
        client_id=client_id,
        client_secret=client_secret,
        authorization_url=f"https://{domain}/oauth2/default/v1/authorize",
        token_url=f"https://{domain}/oauth2/default/v1/token",
        userinfo_url=f"https://{domain}/oauth2/default/v1/userinfo",
        scopes=["openid", "profile", "email", "groups"],
    )


def get_google_config(client_id: str, client_secret: str) -> AuthConfig:
    """Get Google Workspace configuration"""
    return AuthConfig(
        provider=AuthProvider.GOOGLE,
        enabled=True,
        client_id=client_id,
        client_secret=client_secret,
        authorization_url="https://accounts.google.com/o/oauth2/v2/auth",
        token_url="https://oauth2.googleapis.com/token",
        userinfo_url="https://openidconnect.googleapis.com/v1/userinfo",
        scopes=["openid", "profile", "email"],
    )


def get_github_config(client_id: str, client_secret: str) -> AuthConfig:
    """Get GitHub OAuth configuration"""
    return AuthConfig(
        provider=AuthProvider.GITHUB,
        enabled=True,
        client_id=client_id,
        client_secret=client_secret,
        authorization_url="https://github.com/login/oauth/authorize",
        token_url="https://github.com/login/oauth/access_token",
        userinfo_url="https://api.github.com/user",
        scopes=["read:user", "user:email"],
    )
