"""
Team/Enterprise Features

- Shared Conversation Library - Team-wide knowledge base
- Project Templates from Conversations - Learn patterns from team
- Audit Trail - Track what AI generated what
- RBAC - Role-based access control
"""

import hashlib
import json
import logging
import os
from dataclasses import dataclass, field
from datetime import datetime
from enum import Enum
from pathlib import Path
from typing import Any, Dict, List, Optional, Set
import uuid

logger = logging.getLogger(__name__)


class Permission(Enum):
    """Available permissions"""
    READ = "read"
    WRITE = "write"
    DELETE = "delete"
    ADMIN = "admin"
    FORGE = "forge"
    SHARE = "share"
    AUDIT = "audit"


class Role(Enum):
    """Predefined roles"""
    VIEWER = "viewer"       # Read-only access
    MEMBER = "member"       # Read + write + forge
    CONTRIBUTOR = "contributor"  # + share
    ADMIN = "admin"         # All permissions


ROLE_PERMISSIONS = {
    Role.VIEWER: {Permission.READ},
    Role.MEMBER: {Permission.READ, Permission.WRITE, Permission.FORGE},
    Role.CONTRIBUTOR: {Permission.READ, Permission.WRITE, Permission.FORGE, Permission.SHARE},
    Role.ADMIN: set(Permission),
}


@dataclass
class TeamConfig:
    """Team configuration"""
    team_id: str = ""
    team_name: str = ""
    storage_dir: str = ""
    enable_audit: bool = True
    enable_rbac: bool = False
    default_role: Role = Role.MEMBER

    def __post_init__(self):
        if not self.team_id:
            self.team_id = str(uuid.uuid4())[:8]
        if not self.storage_dir:
            home = os.path.expanduser("~")
            self.storage_dir = os.path.join(home, ".ucts", "team", self.team_id)


@dataclass
class AuditEntry:
    """Single audit log entry"""
    entry_id: str
    timestamp: datetime
    action: str
    user_id: str
    resource_type: str
    resource_id: str
    details: Dict[str, Any] = field(default_factory=dict)
    source_hash: str = ""  # Hash of source content

    def to_dict(self) -> Dict[str, Any]:
        return {
            "entry_id": self.entry_id,
            "timestamp": self.timestamp.isoformat(),
            "action": self.action,
            "user_id": self.user_id,
            "resource_type": self.resource_type,
            "resource_id": self.resource_id,
            "details": self.details,
            "source_hash": self.source_hash,
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "AuditEntry":
        return cls(
            entry_id=data["entry_id"],
            timestamp=datetime.fromisoformat(data["timestamp"]),
            action=data["action"],
            user_id=data["user_id"],
            resource_type=data["resource_type"],
            resource_id=data["resource_id"],
            details=data.get("details", {}),
            source_hash=data.get("source_hash", ""),
        )


class AuditLog:
    """
    Audit trail for tracking AI-generated content.

    Tracks:
    - What was generated
    - From which conversation
    - By which user
    - When it was generated
    """

    def __init__(self, storage_dir: str):
        self._storage_path = Path(storage_dir) / "audit"
        self._storage_path.mkdir(parents=True, exist_ok=True)
        self._entries: List[AuditEntry] = []
        self._load_entries()

    def _load_entries(self):
        """Load audit entries from storage"""
        for file in self._storage_path.glob("*.json"):
            try:
                with open(file) as f:
                    data = json.load(f)
                    self._entries.append(AuditEntry.from_dict(data))
            except Exception as e:
                logger.warning(f"Failed to load audit entry {file}: {e}")

    def _save_entry(self, entry: AuditEntry):
        """Save single audit entry"""
        file_path = self._storage_path / f"{entry.entry_id}.json"
        with open(file_path, 'w') as f:
            json.dump(entry.to_dict(), f, indent=2)

    def log(
        self,
        action: str,
        user_id: str,
        resource_type: str,
        resource_id: str,
        details: Optional[Dict[str, Any]] = None,
        content: Optional[str] = None
    ) -> AuditEntry:
        """Log an action"""
        entry = AuditEntry(
            entry_id=str(uuid.uuid4()),
            timestamp=datetime.now(),
            action=action,
            user_id=user_id,
            resource_type=resource_type,
            resource_id=resource_id,
            details=details or {},
            source_hash=hashlib.sha256(content.encode()).hexdigest()[:16] if content else "",
        )

        self._entries.append(entry)
        self._save_entry(entry)

        logger.info(f"Audit: {action} on {resource_type}/{resource_id} by {user_id}")
        return entry

    def log_forge(
        self,
        user_id: str,
        session_source: str,
        output_path: str,
        files_created: List[str],
        languages: List[str]
    ) -> AuditEntry:
        """Log a forge operation"""
        return self.log(
            action="forge",
            user_id=user_id,
            resource_type="project",
            resource_id=output_path,
            details={
                "session_source": session_source,
                "files_created": files_created,
                "languages": languages,
                "file_count": len(files_created),
            },
            content=session_source,
        )

    def log_share(
        self,
        user_id: str,
        resource_id: str,
        shared_with: List[str]
    ) -> AuditEntry:
        """Log a share operation"""
        return self.log(
            action="share",
            user_id=user_id,
            resource_type="session",
            resource_id=resource_id,
            details={
                "shared_with": shared_with,
            },
        )

    def get_entries(
        self,
        user_id: Optional[str] = None,
        action: Optional[str] = None,
        resource_type: Optional[str] = None,
        start_date: Optional[datetime] = None,
        end_date: Optional[datetime] = None,
        limit: int = 100
    ) -> List[AuditEntry]:
        """Query audit entries"""
        results = self._entries[:]

        if user_id:
            results = [e for e in results if e.user_id == user_id]
        if action:
            results = [e for e in results if e.action == action]
        if resource_type:
            results = [e for e in results if e.resource_type == resource_type]
        if start_date:
            results = [e for e in results if e.timestamp >= start_date]
        if end_date:
            results = [e for e in results if e.timestamp <= end_date]

        # Sort by timestamp descending
        results.sort(key=lambda e: e.timestamp, reverse=True)

        return results[:limit]

    def get_resource_history(self, resource_id: str) -> List[AuditEntry]:
        """Get all audit entries for a resource"""
        return [e for e in self._entries if e.resource_id == resource_id]

    def export_report(
        self,
        output_path: str,
        format: str = "json",
        **filters
    ) -> str:
        """Export audit report"""
        entries = self.get_entries(**filters)

        if format == "json":
            output_file = f"{output_path}.json"
            with open(output_file, 'w') as f:
                json.dump([e.to_dict() for e in entries], f, indent=2)

        elif format == "csv":
            output_file = f"{output_path}.csv"
            import csv
            with open(output_file, 'w', newline='') as f:
                writer = csv.writer(f)
                writer.writerow(["Timestamp", "Action", "User", "Resource Type", "Resource ID"])
                for e in entries:
                    writer.writerow([
                        e.timestamp.isoformat(),
                        e.action,
                        e.user_id,
                        e.resource_type,
                        e.resource_id,
                    ])

        elif format == "markdown":
            output_file = f"{output_path}.md"
            lines = [
                "# UCTS Audit Report",
                "",
                f"Generated: {datetime.now().isoformat()}",
                f"Entries: {len(entries)}",
                "",
                "| Timestamp | Action | User | Resource |",
                "|-----------|--------|------|----------|",
            ]
            for e in entries:
                lines.append(
                    f"| {e.timestamp.strftime('%Y-%m-%d %H:%M')} | {e.action} | "
                    f"{e.user_id} | {e.resource_type}/{e.resource_id[:20]} |"
                )

            with open(output_file, 'w') as f:
                f.write("\n".join(lines))

        else:
            raise ValueError(f"Unknown format: {format}")

        return output_file


@dataclass
class SharedSession:
    """A shared conversation session"""
    session_id: str
    title: str
    description: str
    owner_id: str
    created_at: datetime
    updated_at: datetime
    tags: List[str] = field(default_factory=list)
    languages: List[str] = field(default_factory=list)
    code_block_count: int = 0
    shared_with: Set[str] = field(default_factory=set)  # User IDs or "public"
    source_path: str = ""

    def to_dict(self) -> Dict[str, Any]:
        return {
            "session_id": self.session_id,
            "title": self.title,
            "description": self.description,
            "owner_id": self.owner_id,
            "created_at": self.created_at.isoformat(),
            "updated_at": self.updated_at.isoformat(),
            "tags": self.tags,
            "languages": self.languages,
            "code_block_count": self.code_block_count,
            "shared_with": list(self.shared_with),
            "source_path": self.source_path,
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "SharedSession":
        return cls(
            session_id=data["session_id"],
            title=data["title"],
            description=data.get("description", ""),
            owner_id=data["owner_id"],
            created_at=datetime.fromisoformat(data["created_at"]),
            updated_at=datetime.fromisoformat(data["updated_at"]),
            tags=data.get("tags", []),
            languages=data.get("languages", []),
            code_block_count=data.get("code_block_count", 0),
            shared_with=set(data.get("shared_with", [])),
            source_path=data.get("source_path", ""),
        )


@dataclass
class ProjectTemplate:
    """A template extracted from conversations"""
    template_id: str
    name: str
    description: str
    source_sessions: List[str]  # Session IDs this was learned from
    created_by: str
    created_at: datetime
    languages: List[str] = field(default_factory=list)
    patterns: List[str] = field(default_factory=list)
    files: Dict[str, str] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        return {
            "template_id": self.template_id,
            "name": self.name,
            "description": self.description,
            "source_sessions": self.source_sessions,
            "created_by": self.created_by,
            "created_at": self.created_at.isoformat(),
            "languages": self.languages,
            "patterns": self.patterns,
            "files": self.files,
        }


class SharedLibrary:
    """
    Team-wide shared conversation library.

    Features:
    - Store and index sessions
    - Search across team sessions
    - Extract templates from patterns
    """

    def __init__(self, storage_dir: str):
        self._storage_path = Path(storage_dir) / "library"
        self._storage_path.mkdir(parents=True, exist_ok=True)
        self._sessions: Dict[str, SharedSession] = {}
        self._templates: Dict[str, ProjectTemplate] = {}
        self._load_library()

    def _load_library(self):
        """Load library from storage"""
        # Load sessions
        sessions_dir = self._storage_path / "sessions"
        if sessions_dir.exists():
            for file in sessions_dir.glob("*.json"):
                try:
                    with open(file) as f:
                        data = json.load(f)
                        session = SharedSession.from_dict(data)
                        self._sessions[session.session_id] = session
                except Exception as e:
                    logger.warning(f"Failed to load session {file}: {e}")

        # Load templates
        templates_file = self._storage_path / "templates.json"
        if templates_file.exists():
            try:
                with open(templates_file) as f:
                    data = json.load(f)
                    for t in data:
                        template = ProjectTemplate(**t)
                        self._templates[template.template_id] = template
            except Exception as e:
                logger.warning(f"Failed to load templates: {e}")

    def _save_session(self, session: SharedSession):
        """Save session to storage"""
        sessions_dir = self._storage_path / "sessions"
        sessions_dir.mkdir(exist_ok=True)

        file_path = sessions_dir / f"{session.session_id}.json"
        with open(file_path, 'w') as f:
            json.dump(session.to_dict(), f, indent=2)

    def _save_templates(self):
        """Save templates to storage"""
        templates_file = self._storage_path / "templates.json"
        with open(templates_file, 'w') as f:
            json.dump([t.to_dict() for t in self._templates.values()], f, indent=2)

    def add_session(
        self,
        session: "Session",
        owner_id: str,
        title: Optional[str] = None,
        description: str = "",
        tags: Optional[List[str]] = None
    ) -> SharedSession:
        """Add a session to the library"""
        from ucts.analysis import AnalysisEngine

        # Analyze session
        engine = AnalysisEngine()
        structure = engine.analyze(session)

        shared = SharedSession(
            session_id=str(uuid.uuid4()),
            title=title or structure.name,
            description=description or structure.description,
            owner_id=owner_id,
            created_at=datetime.now(),
            updated_at=datetime.now(),
            tags=tags or [],
            languages=structure.languages,
            code_block_count=len(session.code_blocks),
            source_path=session.source,
        )

        self._sessions[shared.session_id] = shared
        self._save_session(shared)

        return shared

    def get_session(self, session_id: str) -> Optional[SharedSession]:
        """Get a session by ID"""
        return self._sessions.get(session_id)

    def list_sessions(
        self,
        user_id: Optional[str] = None,
        tags: Optional[List[str]] = None,
        languages: Optional[List[str]] = None,
        limit: int = 50
    ) -> List[SharedSession]:
        """List sessions with filters"""
        results = list(self._sessions.values())

        # Filter by user access
        if user_id:
            results = [
                s for s in results
                if s.owner_id == user_id or user_id in s.shared_with or "public" in s.shared_with
            ]

        # Filter by tags
        if tags:
            results = [s for s in results if any(t in s.tags for t in tags)]

        # Filter by languages
        if languages:
            results = [s for s in results if any(l in s.languages for l in languages)]

        # Sort by updated_at
        results.sort(key=lambda s: s.updated_at, reverse=True)

        return results[:limit]

    def search_sessions(
        self,
        query: str,
        user_id: Optional[str] = None
    ) -> List[SharedSession]:
        """Search sessions by query"""
        query_lower = query.lower()

        results = []
        for session in self._sessions.values():
            # Check access
            if user_id and session.owner_id != user_id:
                if user_id not in session.shared_with and "public" not in session.shared_with:
                    continue

            # Search in title, description, tags
            if (query_lower in session.title.lower() or
                query_lower in session.description.lower() or
                any(query_lower in tag.lower() for tag in session.tags)):
                results.append(session)

        return results

    def share_session(
        self,
        session_id: str,
        share_with: List[str],
        owner_id: str
    ) -> bool:
        """Share a session with users"""
        session = self._sessions.get(session_id)
        if not session:
            return False

        if session.owner_id != owner_id:
            return False

        session.shared_with.update(share_with)
        session.updated_at = datetime.now()
        self._save_session(session)

        return True

    def extract_template(
        self,
        session_ids: List[str],
        name: str,
        description: str,
        user_id: str
    ) -> Optional[ProjectTemplate]:
        """Extract a template from sessions"""
        # Collect patterns from sessions
        languages = set()
        patterns = []

        for session_id in session_ids:
            session = self._sessions.get(session_id)
            if session:
                languages.update(session.languages)

        template = ProjectTemplate(
            template_id=str(uuid.uuid4()),
            name=name,
            description=description,
            source_sessions=session_ids,
            created_by=user_id,
            created_at=datetime.now(),
            languages=list(languages),
            patterns=patterns,
        )

        self._templates[template.template_id] = template
        self._save_templates()

        return template

    def get_templates(self) -> List[ProjectTemplate]:
        """Get all templates"""
        return list(self._templates.values())


@dataclass
class TeamMember:
    """A team member"""
    user_id: str
    name: str
    email: str
    role: Role
    added_at: datetime = field(default_factory=datetime.now)

    def has_permission(self, permission: Permission) -> bool:
        """Check if member has a permission"""
        return permission in ROLE_PERMISSIONS.get(self.role, set())


class TeamManager:
    """
    Manages team features.

    - Shared conversation library
    - Audit logging
    - Role-based access control
    """

    def __init__(self, config: Optional[TeamConfig] = None):
        self.config = config or TeamConfig()
        self._storage_path = Path(self.config.storage_dir)
        self._storage_path.mkdir(parents=True, exist_ok=True)

        self.library = SharedLibrary(self.config.storage_dir)
        self.audit = AuditLog(self.config.storage_dir) if self.config.enable_audit else None

        self._members: Dict[str, TeamMember] = {}
        self._load_members()

    def _load_members(self):
        """Load team members"""
        members_file = self._storage_path / "members.json"
        if members_file.exists():
            try:
                with open(members_file) as f:
                    data = json.load(f)
                    for m in data:
                        member = TeamMember(
                            user_id=m["user_id"],
                            name=m["name"],
                            email=m["email"],
                            role=Role(m["role"]),
                            added_at=datetime.fromisoformat(m["added_at"]),
                        )
                        self._members[member.user_id] = member
            except Exception as e:
                logger.warning(f"Failed to load members: {e}")

    def _save_members(self):
        """Save team members"""
        members_file = self._storage_path / "members.json"
        data = [
            {
                "user_id": m.user_id,
                "name": m.name,
                "email": m.email,
                "role": m.role.value,
                "added_at": m.added_at.isoformat(),
            }
            for m in self._members.values()
        ]
        with open(members_file, 'w') as f:
            json.dump(data, f, indent=2)

    def add_member(
        self,
        user_id: str,
        name: str,
        email: str,
        role: Optional[Role] = None
    ) -> TeamMember:
        """Add a team member"""
        member = TeamMember(
            user_id=user_id,
            name=name,
            email=email,
            role=role or self.config.default_role,
        )

        self._members[user_id] = member
        self._save_members()

        if self.audit:
            self.audit.log(
                action="add_member",
                user_id="system",
                resource_type="member",
                resource_id=user_id,
                details={"name": name, "role": member.role.value},
            )

        return member

    def remove_member(self, user_id: str) -> bool:
        """Remove a team member"""
        if user_id not in self._members:
            return False

        del self._members[user_id]
        self._save_members()

        if self.audit:
            self.audit.log(
                action="remove_member",
                user_id="system",
                resource_type="member",
                resource_id=user_id,
            )

        return True

    def get_member(self, user_id: str) -> Optional[TeamMember]:
        """Get a team member"""
        return self._members.get(user_id)

    def list_members(self) -> List[TeamMember]:
        """List all team members"""
        return list(self._members.values())

    def update_role(self, user_id: str, new_role: Role) -> bool:
        """Update a member's role"""
        member = self._members.get(user_id)
        if not member:
            return False

        old_role = member.role
        member.role = new_role
        self._save_members()

        if self.audit:
            self.audit.log(
                action="update_role",
                user_id="system",
                resource_type="member",
                resource_id=user_id,
                details={"old_role": old_role.value, "new_role": new_role.value},
            )

        return True

    def check_permission(
        self,
        user_id: str,
        permission: Permission
    ) -> bool:
        """Check if a user has a permission"""
        if not self.config.enable_rbac:
            return True

        member = self._members.get(user_id)
        if not member:
            return False

        return member.has_permission(permission)

    def share_session(
        self,
        session_id: str,
        owner_id: str,
        share_with: List[str]
    ) -> bool:
        """Share a session with team members"""
        # Check permission
        if not self.check_permission(owner_id, Permission.SHARE):
            return False

        result = self.library.share_session(session_id, share_with, owner_id)

        if result and self.audit:
            self.audit.log_share(owner_id, session_id, share_with)

        return result

    def add_to_library(
        self,
        session: "Session",
        user_id: str,
        title: Optional[str] = None,
        tags: Optional[List[str]] = None
    ) -> Optional[SharedSession]:
        """Add a session to the library"""
        # Check permission
        if not self.check_permission(user_id, Permission.WRITE):
            return None

        shared = self.library.add_session(
            session=session,
            owner_id=user_id,
            title=title,
            tags=tags,
        )

        if self.audit:
            self.audit.log(
                action="add_to_library",
                user_id=user_id,
                resource_type="session",
                resource_id=shared.session_id,
                details={
                    "title": shared.title,
                    "languages": shared.languages,
                },
            )

        return shared

    def get_team_status(self) -> Dict[str, Any]:
        """Get team status summary"""
        return {
            "team_id": self.config.team_id,
            "team_name": self.config.team_name,
            "member_count": len(self._members),
            "session_count": len(self.library._sessions),
            "template_count": len(self.library._templates),
            "audit_enabled": self.config.enable_audit,
            "rbac_enabled": self.config.enable_rbac,
        }


# Singleton instance
_team_manager: Optional[TeamManager] = None


def get_team_manager(config: Optional[TeamConfig] = None) -> TeamManager:
    """Get the global team manager instance"""
    global _team_manager
    if _team_manager is None or config is not None:
        _team_manager = TeamManager(config)
    return _team_manager
