"""
Base classes for project templates
"""
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any
from pathlib import Path


@dataclass
class TemplateFile:
    """A file to be generated from a template"""
    path: str
    content: str
    executable: bool = False


@dataclass
class TemplateConfig:
    """Configuration for template generation"""
    project_name: str
    author: str = "Author"
    description: str = ""
    version: str = "0.1.0"
    extras: Dict[str, Any] = field(default_factory=dict)


class ProjectTemplate(ABC):
    """Base class for project templates"""

    @property
    @abstractmethod
    def name(self) -> str:
        """Template identifier (e.g., 'fastapi', 'nextjs')"""
        pass

    @property
    @abstractmethod
    def display_name(self) -> str:
        """Human-readable name"""
        pass

    @property
    @abstractmethod
    def description(self) -> str:
        """Template description"""
        pass

    @property
    @abstractmethod
    def language(self) -> str:
        """Primary language (python, javascript, typescript)"""
        pass

    @property
    def tags(self) -> List[str]:
        """Tags for categorization"""
        return []

    @abstractmethod
    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        """
        Generate template files.

        Args:
            config: Template configuration

        Returns:
            List of files to create
        """
        pass

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        """Dependencies to install (by ecosystem)"""
        return {}

    @property
    def dev_dependencies(self) -> Dict[str, List[str]]:
        """Dev dependencies to install"""
        return {}

    def write_to(self, output_path: str, config: TemplateConfig) -> List[str]:
        """
        Generate and write template files to disk.

        Args:
            output_path: Destination directory
            config: Template configuration

        Returns:
            List of created file paths
        """
        root = Path(output_path)
        root.mkdir(parents=True, exist_ok=True)

        files = self.generate(config)
        created = []

        for file in files:
            file_path = root / file.path
            file_path.parent.mkdir(parents=True, exist_ok=True)
            file_path.write_text(file.content, encoding='utf-8')

            if file.executable:
                file_path.chmod(file_path.stat().st_mode | 0o111)

            created.append(str(file_path))

        return created


class TemplateRegistry:
    """Registry for project templates"""

    def __init__(self):
        self._templates: Dict[str, ProjectTemplate] = {}

    def register(self, template: ProjectTemplate) -> None:
        """Register a template"""
        self._templates[template.name] = template

    def get(self, name: str) -> Optional[ProjectTemplate]:
        """Get a template by name"""
        return self._templates.get(name)

    def list_all(self) -> List[Dict[str, str]]:
        """List all registered templates"""
        return [
            {
                "name": t.name,
                "display_name": t.display_name,
                "description": t.description,
                "language": t.language,
                "tags": t.tags,
            }
            for t in self._templates.values()
        ]

    def search(self, query: str) -> List[ProjectTemplate]:
        """Search templates by name, description, or tags"""
        query = query.lower()
        results = []

        for template in self._templates.values():
            if (query in template.name.lower() or
                query in template.display_name.lower() or
                query in template.description.lower() or
                any(query in tag.lower() for tag in template.tags)):
                results.append(template)

        return results

    def by_language(self, language: str) -> List[ProjectTemplate]:
        """Get templates for a specific language"""
        language = language.lower()
        return [t for t in self._templates.values() if t.language.lower() == language]
